#ifdef n2p2



!***********************************************************************
      module n2p2_predict_interface
!***********************************************************************

      use iso_c_binding

      implicit none

      interface

         subroutine n2p2_atomic_energy_and_forces( &
     &      interface_ptr, index_i, coo_i, type_i, &
     &      n_j, coo_j, elem_j, idx_j, &
     &      e_i, fx_i, fy_i, fz_i, vir_i) &
     &      bind(c, name='n2p2_atomic_energy_and_forces')

            import c_double
            import c_char
            import c_int
            import c_ptr
            type(c_ptr), value, intent(in) :: interface_ptr
            integer(c_int), intent(in) :: index_i
            real(c_double), intent(in) :: coo_i(3)
            character(1, kind=c_char), intent(in) :: type_i(*)
            integer(c_int), intent(in) :: n_j
            type(c_ptr), value, intent(in) :: coo_j  ! (3,n_j)
            type(c_ptr), value, intent(in) :: elem_j  ! (n_j)
            type(c_ptr), value, intent(in) :: idx_j  ! (n_j)
            real(c_double), intent(out) :: e_i
            type(c_ptr), value :: fx_i
            type(c_ptr), value :: fy_i
            type(c_ptr), value :: fz_i
            type(c_ptr), value :: vir_i  ! (3,3)

         end subroutine n2p2_atomic_energy_and_forces

      end interface

!***********************************************************************
      end module n2p2_predict_interface
!***********************************************************************





!***********************************************************************
      subroutine reset_flag_predict_n2p2_MPI
!***********************************************************************

      use common_variables, only : myrank
      use n2p2_variables, only : iset_predict
      use n2p2_general_interface, only : &
     &  n2p2_pimd_closeinterface, n2p2_interface_setup, &
     &  n2p2interface_ptr

      iset_predict = 0

      call n2p2_pimd_closeinterface(n2p2interface_ptr)
      call n2p2_interface_setup(myrank, n2p2interface_ptr)

!***********************************************************************
      end subroutine reset_flag_predict_n2p2_MPI
!***********************************************************************





!***********************************************************************
      subroutine predict_n2p2_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vx, vy, vz, vux, vuy, vuz, vir, pot, &
     &   au_length, au_charge, au_energy, box, &
     &   physmass, natom, nbead, iounit, istep, method, &
     &   myrank, myrank_main, nprocs_main, myrank_sub, nprocs_sub, &
     &   char_spec, boxinv
!      use common_variables, only : mbox

      use n2p2_variables, only : &
     &   coo_i, coo_j, hfx, hfy, hfz, e_i, fx_i, fy_i, fz_i, vir_i, &
     &   r2_j, type_i, rcut2_n2p2, &
     &   iprint_hfx_n2p2, iset_predict, n2p2_lenscale, n2p2_enscale,&
     &   idx_j, neighbor_e, ielement, en_scale_n2p2, len_scale_n2p2, &
     &   n2p2_x, n2p2_y, n2p2_z, n2p2_box, n2p2_box_inv

      use n2p2_general_interface, only : n2p2interface_ptr

      use n2p2_predict_interface, only : &
     &   n2p2_atomic_energy_and_forces

      use mm_variables, only : n_list, j_list, &
     &   jx_list, jy_list, jz_list, nbox_list, bigbox, bigboxinv

      use iso_c_binding

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: m, i, j, n, n_at, jx, jy, jz
!      integer :: m1, m2, m3

!     //   real variables
      real(8) :: px, py, pz, t_i, r2, xij, yij, zij, aij, bij, cij

!     //   initial setting
      integer, save :: iset = 0

!     //   status
      integer :: stat = 0

!     //   unit conversions
      real(8) :: const_r, const_e, const_f, const_s

!     //   status
      integer, save :: ioption = 0

!-----------------------------------------------------------------------
!     /*   initialize n2p2                                           */
!-----------------------------------------------------------------------

      if (iset_predict == 0) iset = 0

!     //   on initial entry
      if ( iset .eq. 0 ) then

!        //   memory allocation
         if ( .not. allocated(vir_i) ) allocate( vir_i(3,3) )
         if ( .not. allocated(fx_i) ) allocate( fx_i(natom) )
         if ( .not. allocated(fy_i) ) allocate( fy_i(natom) )
         if ( .not. allocated(fz_i) ) allocate( fz_i(natom) )

!        //   initialize energy, force, virial, box
         e_i = 0.d0
         vir_i(:,:) = 0.d0

!        //   molecular dynamics
         if ( ( method(1:5) .eq. 'PIMD ' ) .or. &
     &        ( method(1:5) .eq. 'MD   ' ) .or. &
     &        ( method(1:5) .eq. 'CMD  ' ) .or. &
     &        ( method(1:5) .eq. 'RPMD ' ) .or. &
     &        ( method(1:5) .eq. 'BCMD ' ) ) then

!           //   memory allocation of heat flux
            if ( .not. allocated(hfx) ) allocate( hfx(nbead) )
            if ( .not. allocated(hfy) ) allocate( hfy(nbead) )
            if ( .not. allocated(hfz) ) allocate( hfz(nbead) )

!           //   read print interval
            call read_int1_MPI &
     &         ( iprint_hfx_n2p2, '<iprint_hfx_n2p2>', 17, iounit )

!           //   flag
            ioption = 1

!        //   molecular dynamics
         end if

!        //   set finished
         iset = 1
         iset_predict = 1

!     //   on initial entry
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   initialize heat flux   */
      if ( ioption .eq. 1 ) then
         hfx(:) = 0.d0
         hfy(:) = 0.d0
         hfz(:) = 0.d0
      end if

!     /*   unit conversion factor   */
      const_r = n2p2_lenscale
      const_e = n2p2_enscale
      const_f = const_e * const_r
      const_s = const_e

!-----------------------------------------------------------------------
!     /*   call neighbour list                                        */
!-----------------------------------------------------------------------

      call predict_n2p2_makelist_MPI

!-----------------------------------------------------------------------
!     /*   if we have length scaling apply it to the coordinates now  */
!-----------------------------------------------------------------------

      if ( len_scale_n2p2 ) then
         n2p2_x(:,:) = x(:,:) * const_r
         n2p2_y(:,:) = y(:,:) * const_r
         n2p2_z(:,:) = z(:,:) * const_r

         n2p2_box(:,:) = box(:,:) * const_r
         n2p2_box_inv(:,:) = boxinv(:,:) * const_r
      end if

!-----------------------------------------------------------------------
!     /*   execute n2p2 for minimum image convention                  */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   bead parallel   */
         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   loop of atom pairs   */
         do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   atomic coordinates   */
            coo_i(1) = n2p2_x(i,m)
            coo_i(2) = n2p2_y(i,m)
            coo_i(3) = n2p2_z(i,m)

!           /*   atomic kind   */
            type_i = trim(char_spec(i)) // c_null_char

!           /*   clear number of atom pairs   */
            n_at = 0

!           /*   loop of atoms   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = n2p2_x(i,m) - n2p2_x(j,m)
               yij = n2p2_y(i,m) - n2p2_y(j,m)
               zij = n2p2_z(i,m) - n2p2_z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

               r2 = xij*xij+yij*yij+zij*zij

!              /*   if distance if further than cutoff skip   */
               if( r2 > rcut2_n2p2) cycle

!              /*   update number of atom pairs   */
               n_at = n_at + 1

!              /*   atom index   */
               idx_j(n_at) = j

!              //    initialize forces
               fx_i(j) = 0.0
               fy_i(j) = 0.0
               fz_i(j) = 0.0

!              /*   add the distance to distance array   */
               r2_j(n_at) = r2

!              /*   relative interatomic distance   */
               coo_j(1,n_at) = xij
               coo_j(2,n_at) = yij
               coo_j(3,n_at) = zij

!              /*   atomic kind   */
               neighbor_e(n_at) = ielement(j)

!           /*   loop of atoms   */
            end do

!           //   initialize energy
            e_i = 0.d0

!           //   initialize forces
            fx_i(:) = 0.d0
            fy_i(:) = 0.d0
            fz_i(:) = 0.d0

!           //   initialize virial
            vir_i(:,:) = 0.d0

!           //   predict energy and force
            call n2p2_atomic_energy_and_forces( &
     &         n2p2interface_ptr, i, coo_i, type_i, &
     &         n_at, c_loc(coo_j(1,1)), c_loc(neighbor_e(1)), &
     &         c_loc(idx_j(1)), e_i, c_loc(fx_i(1)), c_loc(fy_i(1)), &
     &         c_loc(fz_i(1)), c_loc(vir_i(1,1)))

!           /*   substitute potential energy   */
            pot(m) = pot(m) + e_i

!           /*   substitute force   */
            fx(i,m) = fx(i,m) + fx_i(i)
            fy(i,m) = fy(i,m) + fy_i(i)
            fz(i,m) = fz(i,m) + fz_i(i)
            do n = 1, n_at
                j = idx_j(n)
                fx(j,m) = fx(j,m) + fx_i(j)
                fy(j,m) = fy(j,m) + fy_i(j)
                fz(j,m) = fz(j,m) + fz_i(j)
            end do

!           /*   substitute virial   */
            vir(:,:) = vir(:,:) + vir_i(:,:)

!           /*   molecular dynamics only   */
            if ( ioption .ne. 1 ) cycle

!           /*   define velocity   */
            px = 0.d0
            py = 0.d0
            pz = 0.d0

            if ( method(1:3) .eq. 'MD ' ) then
               px = vx(i,m)
               py = vy(i,m)
               pz = vz(i,m)
            else
               px = vux(i,1)
               py = vuy(i,1)
               pz = vuz(i,1)
            end if

!           /*   kinetic energy   */
            t_i = 0.5d0 * physmass(i) * ( px*px + py*py + pz*pz )

!           /*   substitute heat flux   */
            hfx(m) = hfx(m) &
     &             + ( ( t_i + e_i ) * px * const_e &
     &               + vir_i(1,1) * px * const_s &
     &               + vir_i(2,1) * py * const_s &
     &               + vir_i(3,1) * pz * const_s )
            hfy(m) = hfy(m) &
     &             + ( ( t_i + e_i ) * py * const_e &
     &               + vir_i(2,1) * px * const_s &
     &               + vir_i(2,2) * py * const_s &
     &               + vir_i(3,2) * pz * const_s )
            hfz(m) = hfz(m) &
     &             + ( ( e_i + t_i ) * pz * const_e &
     &               + vir_i(3,1) * px * const_s &
     &               + vir_i(3,2) * py * const_s &
     &               + vir_i(3,3) * pz * const_s )

!        /*   loop of atom pairs   */
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   execute n2p2 for multiple images                           */
!-----------------------------------------------------------------------

!     /*   periodic boundary with replicated boxes   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*n2p2_box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*n2p2_box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*n2p2_box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   bead parallel   */
         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   loop of atom pairs   */
         do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   atomic coordinates   */
            coo_i(1) = n2p2_x(i,m)
            coo_i(2) = n2p2_y(i,m)
            coo_i(3) = n2p2_z(i,m)

!           /*   atomic kind   */
            type_i = trim(char_spec(i)) // c_null_char

!           /*   clear number of atom pairs   */
            n_at = 0

!           /*   loop of atoms   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   replicated boxes   */
               jx = jx_list(n,i,m)
               jy = jy_list(n,i,m)
               jz = jz_list(n,i,m)

!              /*   interatomic distance of i and j in same box   */
               xij = n2p2_x(i,m) - n2p2_x(j,m)
               yij = n2p2_y(i,m) - n2p2_y(j,m)
               zij = n2p2_z(i,m) - n2p2_z(j,m)

!              /*   distance of i and j in different box   */
               xij = xij - n2p2_box(1,1)*jx - n2p2_box(1,2)*jy &
     &                   - n2p2_box(1,3)*jz
               yij = yij - n2p2_box(2,1)*jx - n2p2_box(2,2)*jy &
     &                   - n2p2_box(2,3)*jz
               zij = zij - n2p2_box(3,1)*jx - n2p2_box(3,2)*jy &
     &                   - n2p2_box(3,3)*jz

!               /*   vector in big box   */
                aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &              + bigboxinv(1,3)*zij
                bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &              + bigboxinv(2,3)*zij
                cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &              + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &             + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &             + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &             + bigbox(3,3)*cij

!              //   interatomic distance
               r2 = xij*xij + yij*yij + zij*zij

!              /*   If distance if further than cutoff skip   */
               if( r2 > rcut2_n2p2) cycle

!              /*   update number of atom pairs   */
               n_at = n_at + 1

!              //   atomc index
               idx_j(n_at) = j

!              /*   Add the distance to distance array   */
               r2_j(n_at) = r2

!              //   interatomic distance
               coo_j(1,n_at) = xij
               coo_j(2,n_at) = yij
               coo_j(3,n_at) = zij

!              /*   atomic kind   */
               neighbor_e(n_at) = ielement(j)


!           /*   loop of atoms   */
            end do

!           //   initialize energy
            e_i = 0.d0

!           //   initialize forces
            fx_i(:) = 0.d0
            fy_i(:) = 0.d0
            fz_i(:) = 0.d0

!           //   initialize virial
            vir_i(:,:) = 0.d0

!           //   predict energy and force
            call n2p2_atomic_energy_and_forces( &
     &         n2p2interface_ptr, i, coo_i, type_i, &
     &         n_at, c_loc(coo_j(1,1)), c_loc(neighbor_e(1)), &
     &         c_loc(idx_j(1)), e_i, c_loc(fx_i(1)), c_loc(fy_i(1)), &
     &         c_loc(fz_i(1)), c_loc(vir_i(1,1)))

!           /*   substitute potential energy   */
            pot(m) = pot(m) + e_i

!           /*   substitute force   */
            fx(:,m) = fx(:,m) + fx_i(:)
            fy(:,m) = fy(:,m) + fy_i(:)
            fz(:,m) = fz(:,m) + fz_i(:)

!           /*   substitute virial   */
            vir(:,:) = vir(:,:) + vir_i(:,:)

!           /*   molecular dynamics only   */
            if ( ioption .ne. 1 ) cycle

!           /*   define velocity   */
            px = 0.d0
            py = 0.d0
            pz = 0.d0

            if ( method(1:3) .eq. 'MD ' ) then
               px = vx(i,m)
               py = vy(i,m)
               pz = vz(i,m)
            else
               px = vux(i,1)
               py = vuy(i,1)
               pz = vuz(i,1)
            end if

!           /*   kinetic energy   */
            t_i = 0.5d0 * physmass(i) * ( px*px + py*py + pz*pz )

!           /*   substitute heat flux   */
            hfx(m) = hfx(m) &
     &             + ( ( t_i + e_i ) * px * const_e &
     &               + vir_i(1,1) * px * const_s &
     &               + vir_i(2,1) * py * const_s &
     &               + vir_i(3,1) * pz * const_s )
            hfy(m) = hfy(m) &
     &             + ( ( t_i + e_i ) * py * const_e &
     &               + vir_i(2,1) * px * const_s &
     &               + vir_i(2,2) * py * const_s &
     &               + vir_i(3,2) * pz * const_s )
            hfz(m) = hfz(m) &
     &             + ( ( e_i + t_i ) * pz * const_e &
     &               + vir_i(3,1) * px * const_s &
     &               + vir_i(3,2) * py * const_s &
     &               + vir_i(3,3) * pz * const_s )

!        /*   loop of atom pairs   */
         end do

!     /*   loop of beads   */
      end do

      end if

!-----------------------------------------------------------------------
!     /*   rescale forces and virial                                  */
!-----------------------------------------------------------------------

!     /*   Scale Forces and Virials from N2P2 units to AU   */
!     /*   If N2P2 units are already AU then this is skipped   */
      if( en_scale_n2p2 .or. len_scale_n2p2 ) then

            pot(:) = pot(:) * const_e

            vir(:,:) = vir(:,:) * const_s

            fx(:,:) = fx(:,:) * const_f
            fy(:,:) = fy(:,:) * const_f
            fz(:,:) = fz(:,:) * const_f
      end if

      call error_handling_MPI( stat, 'subroutine predict_n2p2_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

!     /*   heat flux   */
      if ( ioption .eq. 1 ) then
         call my_mpi_allreduce_real_1 ( hfx, nbead )
         call my_mpi_allreduce_real_1 ( hfy, nbead )
         call my_mpi_allreduce_real_1 ( hfz, nbead )
      end if

!-----------------------------------------------------------------------
!     /*   heat flux                                                  */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( iprint_hfx_n2p2 .le. 0 ) return

         if ( mod(istep,iprint_hfx_n2p2) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               open ( iounit, file = 'hfx.out', access='append' )

               do i = 1, nbead
                  write( iounit, '(i8,3f16.8)' ) &
     &               istep, hfx(i), hfy(i), hfz(i)
               end do

               close( iounit )

            end if

         end if

      end if

      return

!***********************************************************************
      end subroutine predict_n2p2_MPI
!***********************************************************************





!***********************************************************************
      subroutine predict_n2p2_makelist_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, box, volume, natom, nbead, iounit, &
     &   nprocs_main, myrank_main, nprocs_sub, myrank_sub, iboundary
!      use common_variables, only : mbox

      use mm_variables, only : &
     &   x_list, y_list, z_list, rcut_list, rcut2_list, bigbox, &
     &   bigboxinv, dmax_list, n_list, j_list, jx_list, jy_list, &
     &   jz_list, nmax_list, nbox_list

      use n2p2_variables, only : &
     &   skin_n2p2, coo_j, cut_skin_n2p2, r2_j, &
     &   idx_j, neighbor_e

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, jx, jy, jz, j2
!      integer :: m1, m2, m3

!     //   real variables
      real(8) :: dx, dy, dz, d2, d2max, rx, ry, rz, r2, aij, bij, cij, &
     &           ax, ay, az, bx, by, bz, cx, cy, cz, absa, absb, absc

!      //   real variables
!      real(8) :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   cutoff for the neighbor list
         rcut2_list = cut_skin_n2p2*cut_skin_n2p2
         rcut_list = cut_skin_n2p2

!        //   maximum deviation allowed without updating neighbor list
         dmax_list = 0.5d0 * skin_n2p2

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

!        //   last position
         if ( .not. allocated(x_list) ) allocate( x_list(natom,nbead) )
         if ( .not. allocated(y_list) ) allocate( y_list(natom,nbead) )
         if ( .not. allocated(z_list) ) allocate( z_list(natom,nbead) )

!-----------------------------------------------------------------------
!        //   setup end
!-----------------------------------------------------------------------

         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and atoms
         do k = 1, nbead

            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

            do i = 1, natom

               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   deviation
               dx = x(i,k) - x_list(i,k)
               dy = y(i,k) - y_list(i,k)
               dz = z(i,k) - z_list(i,k)

!              //   deviation squared
               d2 = dx*dx + dy*dy + dz*dz

!              //   maximum deviation squared
               d2max = max( d2, d2max )

!           //   loop of beads and atoms
            end do

!        //   loop of atoms
         end do

         call my_mpi_allreduce_max_0(d2max)

!        //   if maximum deviation is small, skip neighbor list update
         if ( d2max .lt. (dmax_list*dmax_list) ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_list(1) = 1
         nbox_list(2) = 1
         nbox_list(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_list(1) = int(2.d0*rcut_list/absa) + 1
         nbox_list(2) = int(2.d0*rcut_list/absb) + 1
         nbox_list(3) = int(2.d0*rcut_list/absc) + 1

!     /*   boundary condition   */
      end if

!      write(*,*) "nbox_list : ", nbox_list

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( n_list ) ) allocate( n_list(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        //   clear
         n_list(:,:) = 0

!        //   loop of beads
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atom i
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              //  loop of atom j
               do j = 1, natom

!                 /*   skip same atom   */
                  if ( j .eq. i ) cycle

!                 //   separation of atoms i and j
                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

!                 //   apply boundary condition
                  call pbc_atom_MPI( rx, ry, rz )

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                 //   end of if statement
                  end if

!              //   loop of atom j
               end do

!              //   number of j atoms in list for atom i, bead k
               n_list(i,k) = l

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_2( n_list, natom, nbead )

!     /*   periodic boundary with replicated boxes   */
      else

!        //   clear
         n_list(:,:) = 0

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              /*   loop of atom pairs   */
               do j = 1, natom

!                 /*   loop of replicated boxes   */
                  do jx = 0, nbox_list(1)-1
                  do jy = 0, nbox_list(2)-1
                  do jz = 0, nbox_list(3)-1

!                 /*   square of box index   */
                  j2 = jx*jx + jy*jy + jz*jz

!                 /*   skip same atom   */
                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   interatomic distance of i and j in same box   */
                  rx = x(i,k) - x(j,k)
                  ry = y(i,k) - y(j,k)
                  rz = z(i,k) - z(j,k)

!                 /*   distance of i and j in different box  */
                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                 /*   vector in big box   */
                  aij = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &                + bigboxinv(1,3)*rz
                  bij = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &                + bigboxinv(2,3)*rz
                  cij = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &                + bigboxinv(3,3)*rz

!                 /*   apply periodic boundary in big box   */
                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

!                 /*   distance of nearest i and j   */
                  rx = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &               + bigbox(1,3)*cij
                  ry = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &               + bigbox(2,3)*cij
                  rz = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &               + bigbox(3,3)*cij

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                 //   end of if statement
                  end if

!              /*   loop of replicated boxes   */
               end do
               end do
               end do

!              //   loop of atom j
               end do

!              //   number of j atoms in list for atom i, bead k
               n_list(i,k) = l

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_2( n_list, natom, nbead )

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   maximum number of j atoms in neighbor list
            nmax_list = max( n_list(i,k), nmax_list )

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!      write(*,*) "Maximum number of atoms : ", nmax_list

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated(coo_j) ) deallocate( coo_j )
      allocate( coo_j(3,nmax_list) )

      if ( allocated(r2_j) ) deallocate( r2_j )
      allocate( r2_j(nmax_list) )

      if ( allocated( idx_j ) ) deallocate ( idx_j )
      allocate( idx_j(nmax_list))

      if ( allocated( neighbor_e ) ) deallocate ( neighbor_e )
      allocate( neighbor_e(nmax_list))

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        //    memory allocation
         if ( allocated( j_list ) ) deallocate( j_list )
         allocate( j_list(nmax_list,natom,nbead) )

!        //   clear
         j_list(:,:,:) = 0

!        //   loop of beads
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atom i
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              //  loop of atom j
               do j = 1, natom

!                 /*   skip same atom   */
                  if ( j .eq. i ) cycle

!                 //   separation of atoms i and j
                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

!                 //   apply boundary condition
                  call pbc_atom_MPI( rx, ry, rz )

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                    //   list of j atoms for atom i, bead k
                     j_list(l,i,k) = j

!                 //   end of if statement
                  end if

!              //  loop of atom j
               end do

!           //   loop of atom i
            end do

!        //   loop of beads
         end do
!        //   communicate
         call my_mpi_allreduce_int_3( j_list, nmax_list, natom, nbead )

!     /*   periodic boundary   */
      else

!        //    memory allocation
         if ( allocated( j_list ) ) deallocate( j_list )
         allocate( j_list(nmax_list,natom,nbead) )

         if ( allocated( jx_list ) ) deallocate( jx_list )
         allocate( jx_list(nmax_list,natom,nbead) )

         if ( allocated( jy_list ) ) deallocate( jy_list )
         allocate( jy_list(nmax_list,natom,nbead) )

         if ( allocated( jz_list ) ) deallocate( jz_list )
         allocate( jz_list(nmax_list,natom,nbead) )

!        //   clear
         j_list(:,:,:) = 0
         jx_list(:,:,:) = 0
         jy_list(:,:,:) = 0
         jz_list(:,:,:) = 0

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              /*   loop of atom pairs   */
               do j = 1, natom

!                 /*   loop of replicated boxes   */
                  do jx = 0, nbox_list(1)-1
                  do jy = 0, nbox_list(2)-1
                  do jz = 0, nbox_list(3)-1

!                 /*   square of box index   */
                  j2 = jx*jx + jy*jy + jz*jz

!                 /*   skip same atom   */
                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   interatomic distance of i and j in same box   */
                  rx = x(i,k) - x(j,k)
                  ry = y(i,k) - y(j,k)
                  rz = z(i,k) - z(j,k)

!                 /*   distance of i and j in different box  */
                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                 /*   vector in big box   */
                  aij = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &                + bigboxinv(1,3)*rz
                  bij = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &                + bigboxinv(2,3)*rz
                  cij = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &                + bigboxinv(3,3)*rz

!                 /*   apply periodic boundary in big box   */
                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

!                 /*   distance of nearest i and j   */
                  rx = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &               + bigbox(1,3)*cij
                  ry = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &               + bigbox(2,3)*cij
                  rz = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &               + bigbox(3,3)*cij

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                    //   list of j atoms for atom i, bead k
                     j_list(l,i,k) = j

!                    //   list of box for atom i, bead k
                     jx_list(l,i,k) = jx
                     jy_list(l,i,k) = jy
                     jz_list(l,i,k) = jz

!                 //   end of if statement
                  end if

!                 /*   loop of replicated boxes   */
                  end do
                  end do
                  end do

!              //   loop of atom j
               end do

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_3( j_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jx_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jy_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jz_list, nmax_list, natom, nbead )

!     //   end of if statement
      end if


!-----------------------------------------------------------------------
!     /*   save last position                                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         rx = x(i,j)
         ry = y(i,j)
         rz = z(i,j)

         x_list(i,j) = rx
         y_list(i,j) = ry
         z_list(i,j) = rz

      end do
      end do

      return
      end





!***********************************************************************
      subroutine pbc_unfold_n2p2_MPI( xb, yb, zb, m1, m2, m3 )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      use n2p2_variables, only : n2p2_box

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m1, m2, m3

      real(8) :: xb, yb, zb

!-----------------------------------------------------------------------
!     /*   main part                                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 1 ) then

         xb = xb + dble(m1)*n2p2_box(1,1) &
     &           + dble(m2)*n2p2_box(1,2) &
     &           + dble(m3)*n2p2_box(1,3)

         yb = yb + dble(m1)*n2p2_box(2,1) &
     &           + dble(m2)*n2p2_box(2,2) &
     &           + dble(m3)*n2p2_box(2,3)

         zb = zb + dble(m1)*n2p2_box(3,1) &
     &           + dble(m2)*n2p2_box(3,2) &
     &           + dble(m3)*n2p2_box(3,3)

      end if

      return
      end





!***********************************************************************
      subroutine pbc_atom_n2p2_MPI( xi, yi, zi )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      use n2p2_variables, only : n2p2_box, n2p2_box_inv

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      real(8) :: ai, bi, ci, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 1 ) then

         ai = n2p2_box_inv(1,1)*xi + n2p2_box_inv(1,2)*yi &
     &      + n2p2_box_inv(1,3)*zi
         bi = n2p2_box_inv(2,1)*xi + n2p2_box_inv(2,2)*yi &
     &      + n2p2_box_inv(2,3)*zi
         ci = n2p2_box_inv(3,1)*xi + n2p2_box_inv(3,2)*yi &
     &      + n2p2_box_inv(3,3)*zi

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = n2p2_box(1,1)*ai + n2p2_box(1,2)*bi + n2p2_box(1,3)*ci
         yi = n2p2_box(2,1)*ai + n2p2_box(2,2)*bi + n2p2_box(2,3)*ci
         zi = n2p2_box(3,1)*ai + n2p2_box(3,2)*bi + n2p2_box(3,3)*ci

      else if ( iboundary .eq. 2 ) then

         ai = n2p2_box_inv(1,1)*xi + n2p2_box_inv(1,2)*yi &
     &      + n2p2_box_inv(1,3)*zi
         bi = n2p2_box_inv(2,1)*xi + n2p2_box_inv(2,2)*yi &
     &      + n2p2_box_inv(2,3)*zi
         ci = n2p2_box_inv(3,1)*xi + n2p2_box_inv(3,2)*yi &
     &      + n2p2_box_inv(3,3)*zi

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = n2p2_box(1,1)*ai + n2p2_box(1,2)*bi + n2p2_box(1,3)*ci
         yi = n2p2_box(2,1)*ai + n2p2_box(2,2)*bi + n2p2_box(2,3)*ci
         zi = n2p2_box(3,1)*ai + n2p2_box(3,2)*bi + n2p2_box(3,3)*ci

      end if

      return
      end



#endif
