!///////////////////////////////////////////////////////////////////////
!
!      Author:          K. Kobayashi, Y. Nagai, M. Shiga
!      Last updated:    Apr 10, 2025 by M. Shiga
!      Description:     predict energy and force by aenet
!
!///////////////////////////////////////////////////////////////////////



#if defined(aenet2) || defined(aenet_pytorch)
!aenet2
#ifdef nlist
      subroutine reset_flag_predict_aenet_MPI
            use aenet_variables, only :iset_predict
            iset_predict = 0
      end subroutine 


!***********************************************************************
      subroutine predict_aenet_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vx, vy, vz, vux, vuy, vuz, vir, pot, &
     &   au_length, au_charge, au_length, au_energy, box, &
     &   physmass, natom, nbead, iounit, istep, method, &
     &   myrank, myrank_main, nprocs_main, myrank_sub, nprocs_sub

!     use common_variables, only : mbox

      use mm_variables, only : &
     &   bigbox, bigboxinv, n_list, j_list, jx_list, jy_list, jz_list, &
     &   nbox_list

      use aenet_variables, only : &
     &   coo_i, coo_j, hfx, hfy, hfz, e_i, f_i, vir_i, n_j, itype_i, &
     &   itype_j, index_i, index_j, type_aenet, itype_aenet, &
     &   jtype_aenet, network_aenet, ntype_aenet, iprint_hfx_aenet, &
     &    iset_predict

      use aenet, only : &
     &   aenet_load_potential, aenet_atomic_energy_and_forces, &
     &   aenet_init, aenet_convert_atom_types

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: m, i, j, n, jx, jy, jz
!      integer :: m1, m2, m3

!     //   real variables
      real(8) :: xij, yij, zij, aij, bij, cij, px, py, pz, t_i

!     //   initial setting
      integer, save :: iset = 0

!     //   status
      integer :: stat = 0

!     //   unit conversions
      real(8) :: const_r, const_e, const_f, const_s, const_f_inv

!     //   status
      integer, save :: ioption = 0

!     //   status
      integer, save :: iload_aenet = 0

!-----------------------------------------------------------------------
!     /*   initialize aenet                                           */
!-----------------------------------------------------------------------

!     //   on initial entry
      if ( iset_predict .eq. 0 ) iset = 0

!     //   on initial entry
      if ( iset .eq. 0 ) then

!        //   initiate aenet
         call aenet_init( type_aenet, stat )

!        //   conversion of atomic type - aenet internal
         call aenet_convert_atom_types( &
     &        type_aenet, itype_aenet, jtype_aenet, stat )

!        //   error message
         if ( stat .ne. 0 ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - incorrect at aenet_init.'
               write( 6, '(a)' )
            end if
         end if

!        //   error handling
         call error_handling_MPI &
     &      ( stat, 'subroutine predict_aenet_MPI', 28 )

!        //   load aenet potential
         iload_aenet = 0
         do i = 1, ntype_aenet
            call aenet_load_potential( i, network_aenet(i), stat )
            if ( stat .ne. 0 ) then
               iload_aenet = iload_aenet + 1
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Ann file ' // trim(network_aenet(i)) // &
     &               ' is incorrect or not found.'
               end if
            end if
         end do
         if ( iload_aenet .ne. 0 ) then
            if ( myrank .eq. 0 ) write( 6, '(a)' )
         end if

!c        //   error handling
!         call error_handling_MPI
!     &      ( iload_aenet, 'subroutine predict_aenet_MPI', 28 )

!        //   memory allocation of heat flux
         if ( .not. allocated(f_i) ) allocate( f_i(3,natom) )

!        //   initialize energy, force, virial
         e_i = 0.d0
         f_i(:,:) = 0.d0
         vir_i(:,:) = 0.d0

!        //   molecular dynamics
         if ( ( method(1:5) .eq. 'PIMD ' ) .or. &
     &        ( method(1:5) .eq. 'MD   ' ) .or. &
     &        ( method(1:5) .eq. 'CMD  ' ) .or. &
     &        ( method(1:5) .eq. 'RPMD ' ) .or. &
     &        ( method(1:5) .eq. 'TRPMD' ) .or. &
     &        ( method(1:5) .eq. 'BCMD ' ) ) then

!           //   memory allocation of heat flux
            if ( .not. allocated(hfx) ) allocate( hfx(nbead) )
            if ( .not. allocated(hfy) ) allocate( hfy(nbead) )
            if ( .not. allocated(hfz) ) allocate( hfz(nbead) )

!           //   read print interval
            call read_int1_MPI &
     &         ( iprint_hfx_aenet, '<iprint_hfx_aenet>', 18, iounit )

!           //   flag
            ioption = 1

!        //   molecular dynamics
         end if

!        //   set finished
         iset = 1

!        //   set finished
         iset_predict = 1

!     //   on initial entry
      end if

!-----------------------------------------------------------------------
!     /*   if not aenet loaded, potential free                        */
!-----------------------------------------------------------------------

      if ( iload_aenet .ne. 0 ) then

!        /*   potential energy   */
         pot(:) = 0.d0

!        /*   force   */
         fx(:,:) = 0.d0
         fy(:,:) = 0.d0
         fz(:,:) = 0.d0

         return

      end if

!-----------------------------------------------------------------------
!     /*   make neighbor list when necessary                          */
!-----------------------------------------------------------------------

      call predict_aenet_makelist_MPI

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   initialize heat flux   */
      if ( ioption .eq. 1 ) then
         hfx(:) = 0.d0
         hfy(:) = 0.d0
         hfz(:) = 0.d0
      end if

!     /*   unit conversion factor   */
      const_r =  au_length / 1.d-10
      const_e =  au_charge / au_energy
      const_f =  au_charge / au_energy * au_length / 1.d-10
      const_s =  const_e
      const_f_inv = 1.d0 / const_f

!-----------------------------------------------------------------------
!     /*   for minimum image convention                               */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   atomic coordinates   */
               coo_i(1) = x(i,m) * const_r
               coo_i(2) = y(i,m) * const_r
               coo_i(3) = z(i,m) * const_r

!              /*   atomic kind   */
               itype_i = jtype_aenet(i)

!              /*   atomic number   */
               index_i = i

!              /*   loop of atom pairs   */
               do n = 1, n_list(i,m)

!                 /*   atom in neighbor list   */
                  j = j_list(n,i,m)

!                 /*   interatomic distance   */
                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

!                 /*   apply free or periodic boundary   */
                  call pbc_atom_MPI ( xij, yij, zij )

!                 /*   atomic coordinates   */
                  coo_j(1,n) = ( x(i,m) - xij ) * const_r
                  coo_j(2,n) = ( y(i,m) - yij ) * const_r
                  coo_j(3,n) = ( z(i,m) - zij ) * const_r

!                 /*   atomic kind   */
                  itype_j(n) = jtype_aenet(j)

!                 /*   atomic number   */
                  index_j(n) = j

!              /*   loop of atom pairs   */
               end do

!              /*   neighbor list   */
               n_j = n_list(i,m)

!              //   initialize force
               f_i(1,:) = fx(:,m) * const_f_inv
               f_i(2,:) = fy(:,m) * const_f_inv
               f_i(3,:) = fz(:,m) * const_f_inv

!              //   initialize virial
               vir_i(:,:) = 0.d0

!              /*  aenet atomic energy   */
               call aenet_atomic_energy_and_forces( &
     &              coo_i, itype_i, index_i, n_j, coo_j, itype_j, &
     &              index_j, natom, e_i, f_i, vir_i, stat )

!              /*   substitute potential energy   */
               pot(m) = pot(m) + e_i * const_e

!              /*   substitute force   */
               fx(:,m) = f_i(1,:) * const_f
               fy(:,m) = f_i(2,:) * const_f
               fz(:,m) = f_i(3,:) * const_f

!              /*   substitute virial   */
               vir(1,1) = vir(1,1) + vir_i(1,1) * const_s
               vir(1,2) = vir(1,2) + vir_i(2,1) * const_s
               vir(1,3) = vir(1,3) + vir_i(3,1) * const_s
               vir(2,1) = vir(2,1) + vir_i(2,1) * const_s
               vir(2,2) = vir(2,2) + vir_i(2,2) * const_s
               vir(2,3) = vir(2,3) + vir_i(3,2) * const_s
               vir(3,1) = vir(3,1) + vir_i(3,1) * const_s
               vir(3,2) = vir(3,2) + vir_i(3,2) * const_s
               vir(3,3) = vir(3,3) + vir_i(3,3) * const_s

!              /*   molecular dynamics only   */
               if ( ioption .ne. 1 ) cycle

!              /*   define velocity   */
               px = 0.d0
               py = 0.d0
               pz = 0.d0

               if ( method(1:3) .eq. 'MD ' ) then
                  px = vx(i,m)
                  py = vy(i,m)
                  pz = vz(i,m)
               else
                  px = vux(i,1)
                  py = vuy(i,1)
                  pz = vuz(i,1)
               end if

!              /*   kinetic energy   */
               t_i = 0.5d0 * physmass(i) * ( px*px + py*py + pz*pz )

!              /*   substitute heat flux   */
               hfx(m) = hfx(m) &
     &                + ( ( t_i + e_i ) * px * const_e &
     &                  + vir_i(1,1) * px * const_s &
     &                  + vir_i(2,1) * py * const_s &
     &                  + vir_i(3,1) * pz * const_s )
               hfy(m) = hfy(m) &
     &                + ( ( t_i + e_i ) * py * const_e &
     &                  + vir_i(2,1) * px * const_s &
     &                  + vir_i(2,2) * py * const_s &
     &                  + vir_i(3,2) * pz * const_s )
               hfz(m) = hfz(m) &
     &                + ( ( e_i + t_i ) * pz * const_e &
     &                  + vir_i(3,1) * px * const_s &
     &                  + vir_i(3,2) * py * const_s &
     &                  + vir_i(3,3) * pz * const_s )

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

         call error_handling_MPI &
     &      ( stat, 'subroutine predict_aenet_MPI', 28 )

!-----------------------------------------------------------------------
!     /*   multiple images                                            */
!-----------------------------------------------------------------------

!     /*   periodic boundary with replicated boxes   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   atomic coordinates   */
               coo_i(1) = x(i,m) * const_r
               coo_i(2) = y(i,m) * const_r
               coo_i(3) = z(i,m) * const_r

!              /*   atomic kind   */
               itype_i = jtype_aenet(i)

!              /*   atomic number   */
               index_i = i

!              /*   loop of atom pairs   */
               do n = 1, n_list(i,m)

!                 /*   atom in neighbor list   */
                  j = j_list(n,i,m)

!                 /*   replicated boxes   */
                  jx = jx_list(n,i,m)
                  jy = jy_list(n,i,m)
                  jz = jz_list(n,i,m)

!                 /*   interatomic distance of i and j in same box   */
                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

!!                 /*   original boxes   */
!                  m1 = mbox(1,i,m) - mbox(1,j,m)
!                  m2 = mbox(2,i,m) - mbox(2,j,m)
!                  m3 = mbox(3,i,m) - mbox(3,j,m)
!
!!                 /*   unfold   */
!                  call pbc_unfold_MPI( xij, yij, zij, m1, m2, m3 )

!                 /*   distance of i and j in different box  */
                  xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                 /*   vector in big box   */
                  aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &                + bigboxinv(1,3)*zij
                  bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &                + bigboxinv(2,3)*zij
                  cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &                + bigboxinv(3,3)*zij

!                 /*   apply periodic boundary in big box   */
                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

!                 /*   distance of nearest i and j   */
                  xij = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &                + bigbox(1,3)*cij
                  yij = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &                + bigbox(2,3)*cij
                  zij = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &                + bigbox(3,3)*cij

!                 /*   atomic coordinates   */
                  coo_j(1,n) = ( x(i,m) - xij ) * const_r
                  coo_j(2,n) = ( y(i,m) - yij ) * const_r
                  coo_j(3,n) = ( z(i,m) - zij ) * const_r

!                 /*   atomic kind   */
                  itype_j(n) = jtype_aenet(j)

!                 /*   atomic number   */
                  index_j(n) = j

!              /*   loop of atom pairs   */
               end do

!              /*   neighbor list   */
               n_j = n_list(i,m)

!              //   initialize force
               f_i(1,:) = fx(:,m) * const_f_inv
               f_i(2,:) = fy(:,m) * const_f_inv
               f_i(3,:) = fz(:,m) * const_f_inv

!              //   initialize virial
               vir_i(:,:) = 0.d0

!              /*  aenet atomic energy   */
               call aenet_atomic_energy_and_forces( &
     &              coo_i, itype_i, index_i, n_j, coo_j, itype_j, &
     &              index_j, natom, e_i, f_i, vir_i, stat )

!              /*   substitute potential energy   */
               pot(m) = pot(m) + e_i * const_e

!              /*   substitute force   */
               fx(:,m) = f_i(1,:) * const_f
               fy(:,m) = f_i(2,:) * const_f
               fz(:,m) = f_i(3,:) * const_f

!              /*   substitute virial   */
               vir(1,1) = vir(1,1) + vir_i(1,1) * const_s
               vir(1,2) = vir(1,2) + vir_i(2,1) * const_s
               vir(1,3) = vir(1,3) + vir_i(3,1) * const_s
               vir(2,1) = vir(2,1) + vir_i(2,1) * const_s
               vir(2,2) = vir(2,2) + vir_i(2,2) * const_s
               vir(2,3) = vir(2,3) + vir_i(3,2) * const_s
               vir(3,1) = vir(3,1) + vir_i(3,1) * const_s
               vir(3,2) = vir(3,2) + vir_i(3,2) * const_s
               vir(3,3) = vir(3,3) + vir_i(3,3) * const_s

!              /*   molecular dynamics only   */
               if ( ioption .ne. 1 ) cycle

!              /*   define velocity   */
               px = 0.d0
               py = 0.d0
               pz = 0.d0

               if ( method(1:3) .eq. 'MD ' ) then
                  px = vx(i,m)
                  py = vy(i,m)
                  pz = vz(i,m)
               else
                  px = vux(i,1)
                  py = vuy(i,1)
                  pz = vuz(i,1)
               end if

!              /*   kinetic energy   */
               t_i = 0.5d0 * physmass(i) * ( px*px + py*py + pz*pz )

!              /*   substitute heat flux   */
               hfx(m) = hfx(m) &
     &                + ( ( t_i + e_i ) * px * const_e &
     &                  + vir_i(1,1) * px * const_s &
     &                  + vir_i(2,1) * py * const_s &
     &                  + vir_i(3,1) * pz * const_s )
               hfy(m) = hfy(m) &
     &                + ( ( t_i + e_i ) * py * const_e &
     &                  + vir_i(2,1) * px * const_s &
     &                  + vir_i(2,2) * py * const_s &
     &                  + vir_i(3,2) * pz * const_s )
               hfz(m) = hfz(m) &
     &                + ( ( e_i + t_i ) * pz * const_e &
     &                  + vir_i(3,1) * px * const_s &
     &                  + vir_i(3,2) * py * const_s &
     &                  + vir_i(3,3) * pz * const_s )

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

         call error_handling_MPI &
     &      ( stat, 'subroutine predict_aenet_MPI', 28 )

!     /*   periodic boundary   */
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

!     /*   heat flux   */
      if ( ioption .eq. 1 ) then
         call my_mpi_allreduce_real_1 ( hfx, nbead )
         call my_mpi_allreduce_real_1 ( hfy, nbead )
         call my_mpi_allreduce_real_1 ( hfz, nbead )
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( iprint_hfx_aenet .le. 0 ) return

         if ( mod(istep,iprint_hfx_aenet) .eq. 0 ) then

            if ( myrank .eq. 0 ) then

               open ( iounit, file = 'hfx.out', access='append' )

               do i = 1, nbead
                  write( iounit, '(i8,3f16.8)' ) &
     &               istep, hfx(i), hfy(i), hfz(i)
               end do

               close( iounit )

            end if

         end if

      end if

      return
      end





!***********************************************************************
      subroutine predict_aenet_makelist_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, box, volume, natom, nbead, iounit, mbox, &
     &   iboundary, nprocs_main, myrank_main, nprocs_sub, myrank_sub, &
     &   myrank

      use mm_variables, only : &
     &   x_list, y_list, z_list, rcut_list, rcut2_list, bigbox, &
     &   bigboxinv, dmax_list, n_list, j_list, jx_list, jy_list, &
     &   jz_list, nmax_list, nbox_list

      use aenet_variables, only : &
     &   rcut_aenet, skin_aenet, coo_j, itype_j, index_j

      use aenet, only : &
     &   aenet_Rc_max

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, jx, jy, jz, j2, m1, m2, m3, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, d2max, rx, ry, rz, r2, aij, bij, cij, &
     &           ax, ay, az, bx, by, bz, cx, cy, cz, absa, absb, absc

!     //   real variables
      real(8) :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   read cutoff                                             */
!-----------------------------------------------------------------------

!c        //   master rank only
!         if ( myrank .eq. 0 ) then
!
!c           //   open file
!            open ( iounit, file = 'input.dat' )
!
!c           /*   tag   */
!            call search_tag ( '<rcut_aenet>', 12, iounit, ierr )
!
!c           /*   cut off distance   */
!            read( iounit, *, iostat=ierr ) rcut_aenet
!
!c           //   close file
!            close( iounit )
!
!c        //   master rank only
!         end if
!
!c        //   communicate
!         call my_mpi_bcast_int_0( ierr )
!
!c        //   default value
!         if ( ierr .ne. 0 ) then
!
!c           //   master rank only
!            if ( myrank .eq. 0 ) then
!
!c              //   open file
!               open ( iounit, file = 'input_default.dat' )
!
!c              /*   tag   */
!               call search_tag ( '<rcut_aenet>', 12, iounit, ierr )
!
!c              /*   cut off distance   */
!               read( iounit, *, iostat=ierr ) rcut_aenet
!
!c              //   close file
!               close( iounit )
!
!c           //   master rank only
!            end if
!
!c        //   default value
!         end if
!
!c        //   communicate
!         call my_mpi_bcast_int_0( ierr )
!
!c        //   error message
!         if ( ierr .ne. 0 ) then
!            if ( myrank .eq. 0 ) then
!               write( 6, '(a)' )
!     &            'Error - keyword <rcut_aenet> is incorrect.'
!               write( 6, '(a)' )
!            end if
!         end if
!
!c        /*   stop on error   */
!         call error_handling_MPI
!     &      ( ierr, 'subroutine predict_aenet_makelist_MPI', 37 )
!
!c        //   communicate
!         call my_mpi_bcast_real_0( rcut_aenet )

!        //   aenet cutoff
         rcut_aenet = aenet_Rc_max

!        //   angstrom to bohr
         rcut_aenet = rcut_aenet / bohr2ang

!-----------------------------------------------------------------------
!        /*   read skin                                               */
!-----------------------------------------------------------------------

!        //   master rank only
         if ( myrank .eq. 0 ) then

!           //   open file
            open ( iounit, file = 'input.dat' )

!           /*   tag   */
            call search_tag ( '<skin_aenet>', 12, iounit, ierr )

!           /*   cut off distance   */
            read( iounit, *, iostat=ierr ) skin_aenet

!           //   close file
            close( iounit )

!        //   master rank only
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   default value
         if ( ierr .ne. 0 ) then

!           //   master rank only
            if ( myrank .eq. 0 ) then

!              //   open file
               open ( iounit, file = 'input_default.dat' )

!              /*   tag   */
               call search_tag ( '<skin_aenet>', 12, iounit, ierr )

!              /*   cut off distance   */
               read( iounit, *, iostat=ierr ) skin_aenet

!              //   close file
               close( iounit )

!           //   master rank only
            end if

!        //   default value
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   error message
         if ( ierr .ne. 0 ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - keyword <skin_aenet> is incorrect.'
               write( 6, '(a)' )
            end if
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine predict_aenet_makelist_MPI', 37 )

!        //   communicate
         call my_mpi_bcast_real_0( skin_aenet )

!        //   angstrom to bohr
         skin_aenet = skin_aenet / bohr2ang

!-----------------------------------------------------------------------
!        /*   neighbor list cutoff                                    */
!-----------------------------------------------------------------------

!        //   list cutoff radius
         rcut_list = rcut_aenet + skin_aenet

!        //   list cutoff radius squared
         rcut2_list = rcut_list * rcut_list

!        //   maximum deviation allowed without updating neighbor list
         dmax_list = 0.5d0 * skin_aenet

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

!        //   last position
         if ( .not. allocated(x_list) ) allocate( x_list(natom,nbead) )
         if ( .not. allocated(y_list) ) allocate( y_list(natom,nbead) )
         if ( .not. allocated(z_list) ) allocate( z_list(natom,nbead) )

!-----------------------------------------------------------------------
!        //   setup end
!-----------------------------------------------------------------------

         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and atoms
         do k = 1, nbead
         do i = 1, natom

!           //   deviation
            dx = x(i,k) - x_list(i,k)
            dy = y(i,k) - y_list(i,k)
            dz = z(i,k) - z_list(i,k)

!           //   box
            m1 = mbox(1,i,k)
            m2 = mbox(2,i,k)
            m3 = mbox(3,i,k)

!           //   unfold deviation
            call pbc_unfold_MPI ( dx, dy, dz, m1, m2, m3 )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and atoms
         end do
         end do

!        //   if maximum deviation is small, skip neighbor list update
         if ( d2max .lt. (dmax_list*dmax_list) ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_list(1) = 1
         nbox_list(2) = 1
         nbox_list(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_list(1) = int(2.d0*rcut_list/absa) + 1
         nbox_list(2) = int(2.d0*rcut_list/absb) + 1
         nbox_list(3) = int(2.d0*rcut_list/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( n_list ) ) allocate( n_list(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        //   clear
         n_list(:,:) = 0

!        //   loop of beads
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atom i
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              //  loop of atom j
               do j = 1, natom

!                 /*   skip same atom   */
                  if ( j .eq. i ) cycle

!                 //   separation of atoms i and j
                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

!                 //   apply boundary condition
                  call pbc_atom_MPI( rx, ry, rz )

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                 //   end of if statement
                  end if

!              //   loop of atom j
               end do

!              //   number of j atoms in list for atom i, bead k
               n_list(i,k) = l

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_2( n_list, natom, nbead )

!     /*   periodic boundary with replicated boxes   */
      else

!        //   clear
         n_list(:,:) = 0

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              /*   loop of atom pairs   */
               do j = 1, natom

!                 /*   loop of replicated boxes   */
                  do jx = 0, nbox_list(1)-1
                  do jy = 0, nbox_list(2)-1
                  do jz = 0, nbox_list(3)-1

!                 /*   square of box index   */
                  j2 = jx*jx + jy*jy + jz*jz

!                 /*   skip same atom   */
                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   interatomic distance of i and j in same box   */
                  rx = x(i,k) - x(j,k)
                  ry = y(i,k) - y(j,k)
                  rz = z(i,k) - z(j,k)

!                 /*   original boxes   */
                  m1 = mbox(1,i,k) - mbox(1,j,k)
                  m2 = mbox(2,i,k) - mbox(2,j,k)
                  m3 = mbox(3,i,k) - mbox(3,j,k)

!!                 /*   unfold   */
!                  call pbc_unfold_MPI( rx, ry, rz, m1, m2, m3 )
!
!!                 /*   distance of i and j in different box  */
!                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
!                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
!                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                 /*   vector in big box   */
                  aij = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &                + bigboxinv(1,3)*rz
                  bij = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &                + bigboxinv(2,3)*rz
                  cij = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &                + bigboxinv(3,3)*rz

!                 /*   apply periodic boundary in big box   */
                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

!                 /*   distance of nearest i and j   */
                  rx = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &               + bigbox(1,3)*cij
                  ry = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &               + bigbox(2,3)*cij
                  rz = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &               + bigbox(3,3)*cij

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                 //   end of if statement
                  end if

!              /*   loop of replicated boxes   */
               end do
               end do
               end do

!              //   loop of atom j
               end do

!              //   number of j atoms in list for atom i, bead k
               n_list(i,k) = l

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_2( n_list, natom, nbead )

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   maximum number of j atoms in neighbor list
            nmax_list = max( n_list(i,k), nmax_list )

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated(coo_j) ) deallocate( coo_j )
      allocate( coo_j(3,nmax_list) )

      if ( allocated(itype_j) ) deallocate( itype_j )
      allocate( itype_j(nmax_list) )

      if ( allocated(index_j) ) deallocate( index_j )
      allocate( index_j(nmax_list) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        //    memory allocation
         if ( allocated( j_list ) ) deallocate( j_list )
         allocate( j_list(nmax_list,natom,nbead) )

!        //   clear
         j_list(:,:,:) = 0

!        //   loop of beads
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atom i
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              //  loop of atom j
               do j = 1, natom

!                 /*   skip same atom   */
                  if ( j .eq. i ) cycle

!                 //   separation of atoms i and j
                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

!                 //   apply boundary condition
                  call pbc_atom_MPI( rx, ry, rz )

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                    //   list of j atoms for atom i, bead k
                     j_list(l,i,k) = j

!                 //   end of if statement
                  end if

!              //  loop of atom j
               end do

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_3( j_list, nmax_list, natom, nbead )

!     /*   periodic boundary   */
      else

!        //    memory allocation
         if ( allocated( j_list ) ) deallocate( j_list )
         allocate( j_list(nmax_list,natom,nbead) )

         if ( allocated( jx_list ) ) deallocate( jx_list )
         allocate( jx_list(nmax_list,natom,nbead) )

         if ( allocated( jy_list ) ) deallocate( jy_list )
         allocate( jy_list(nmax_list,natom,nbead) )

         if ( allocated( jz_list ) ) deallocate( jz_list )
         allocate( jz_list(nmax_list,natom,nbead) )

!        //   clear
         j_list(:,:,:) = 0
         jx_list(:,:,:) = 0
         jy_list(:,:,:) = 0
         jz_list(:,:,:) = 0

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do k = 1, nbead

!           /*   bead parallel   */
            if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   counter
               l = 0

!              /*   loop of atom pairs   */
               do j = 1, natom

!                 /*   loop of replicated boxes   */
                  do jx = 0, nbox_list(1)-1
                  do jy = 0, nbox_list(2)-1
                  do jz = 0, nbox_list(3)-1

!                 /*   square of box index   */
                  j2 = jx*jx + jy*jy + jz*jz

!                 /*   skip same atom   */
                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                 /*   interatomic distance of i and j in same box   */
                  rx = x(i,k) - x(j,k)
                  ry = y(i,k) - y(j,k)
                  rz = z(i,k) - z(j,k)

!                 /*   original boxes   */
                  m1 = mbox(1,i,k) - mbox(1,j,k)
                  m2 = mbox(2,i,k) - mbox(2,j,k)
                  m3 = mbox(3,i,k) - mbox(3,j,k)

!!                 /*   unfold   */
!                  call pbc_unfold_MPI( rx, ry, rz, m1, m2, m3 )
!
!!                 /*   distance of i and j in different box  */
!                  rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
!                  ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
!                  rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                 /*   vector in big box   */
                  aij = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &                + bigboxinv(1,3)*rz
                  bij = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &                + bigboxinv(2,3)*rz
                  cij = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &                + bigboxinv(3,3)*rz

!                 /*   apply periodic boundary in big box   */
                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

!                 /*   distance of nearest i and j   */
                  rx = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &               + bigbox(1,3)*cij
                  ry = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &               + bigbox(2,3)*cij
                  rz = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &               + bigbox(3,3)*cij

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut2_list ) then

!                    //   update counter
                     l = l + 1

!                    //   list of j atoms for atom i, bead k
                     j_list(l,i,k) = j

!                    //   list of box for atom i, bead k
                     jx_list(l,i,k) = jx
                     jy_list(l,i,k) = jy
                     jz_list(l,i,k) = jz

!                 //   end of if statement
                  end if

!                 /*   loop of replicated boxes   */
                  end do
                  end do
                  end do

!              //   loop of atom j
               end do

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_3( j_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jx_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jy_list, nmax_list, natom, nbead )
         call my_mpi_allreduce_int_3( jz_list, nmax_list, natom, nbead )

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   save last position                                         */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         rx = x(i,j)
         ry = y(i,j)
         rz = z(i,j)

         m1 = mbox(1,i,j)
         m2 = mbox(2,i,j)
         m3 = mbox(3,i,j)

         call pbc_unfold_MPI ( rx, ry, rz, m1, m2, m3 )

         x_list(i,j) = rx
         y_list(i,j) = ry
         z_list(i,j) = rz

      end do
      end do

      return
      end



#else



!***********************************************************************
      subroutine predict_aenet_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, au_charge, au_energy, au_length, &
     &   box, box_bead, species, volume, volume_bead, vir, vir_bead, &
     &   natom, nbead, iounit, iboundary, myrank, nprocs, &
     &   myrank_main, nprocs_main, myrank_sub, mpi_comm_sub, method

      use aenet_variables, only : &
     &   type_aenet, network_aenet, ntype_aenet, iflag_aenet, &
     &   iann_start_aenet

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, ierr, jerr, itest

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz, const

!     /*   bead number   */
      character(len=3) :: char_num

!     /*   words   */
      character(len=80) :: char_word

!     /*   lines   */
      character(len=80) :: char_line

!     /*   real numbers   */
      real(8) :: const_e, const_f

!     /*   energy from aenet   */
      double precision :: Etot_vec(1)

!     /*   force from aenet   */
      double precision :: forCart_vec(3,natom,1)

!     /*   stress from aenet   */
      double precision :: stress_vec(3,3,1)

!     /*   real numbers   */
      real(8), dimension(3,3) :: stress_vec_sum, stress_tensor

!     /*   integers   */
      integer :: iounit_predict = 1040

!-----------------------------------------------------------------------
!     /*   set the communicator                                       */
!-----------------------------------------------------------------------

!     /*   if communicator not set   */
      if ( iflag_aenet .ne. 1 ) then

!        /*   set mpi comm   */
         call set_mpi_aenet( mpi_comm_sub )

!        /*   set flag   */
         iflag_aenet = 1

!     /*   if communicator not set   */
      end if

!-----------------------------------------------------------------------
!     /*   create input with a following format                       */
!-----------------------------------------------------------------------
!
!     TYPES
!     number_of_types
!     type_1  (atomic species)
!     type_2  (atomic species)
!     ......
!
!     NETWORKS
!     type_1 network_1
!     type_2 network_2
!     ...... .........
!
!     FORCES  (fixed)
!
!     VERBOSITY low  (fixed)
!
!     FILES
!     1 (fixed)
!     ./predict.xsf.bead_no
!
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   skip if this is not my bead   */
         if ( mod( j-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   only for rank zero   */
         if ( myrank_sub .ne. 0 ) cycle

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   open sample file   */
         open( iounit, file = './predict.' // char_num // '.in' )

!        /*   types   */
         write( iounit, '(a)' ) 'TYPES'
         write( iounit, '(i8)' ) ntype_aenet
         do i = 1, ntype_aenet
            write( iounit, '(a4)' ) type_aenet(i)
         end do

!        /*   networks   */
         write( iounit, '(a)' ) 'NETWORKS'
         do i = 1, ntype_aenet
            write( iounit, '(a4,a40)' ) type_aenet(i), network_aenet(i)
         end do

         write( iounit, '(a)' ) 'FORCES'
         write( iounit, '(a)' ) 'VERBOSITY low'
         write( iounit, '(a)' ) 'FILES'
         write( iounit, '(a)' ) '1'
         write( iounit, '(a)' ) './predict.' // char_num // '.xsf'

!        /*   close input file   */
         close( iounit )

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   write atomic position with the following format            */
!-----------------------------------------------------------------------
!
!     for periodic boundary condition:
!
!     CRYSTAL
!     PRIMVEC
!     ax  ay  az  (box unit cell)
!     bx  by  bz  (box unit cell)
!     cx  cy  cz  (box unit cell)
!     PRIMCOORD
!     number_of_atoms   1 (fixed)
!     species_1  x_1  y_1  z_1  (coordinates)
!     species_2  x_2  y_2  z_2  (coordinates)
!     .........  ...  ...  ...  (coordinates)
!
!     for free boundary condition:
!
!     PRIMCOORD
!     number_of_atoms   1 (fixed)
!     species_1  x_1  y_1  z_1  (coordinates)
!     species_2  x_2  y_2  z_2  (coordinates)
!     .........  ...  ...  ...  (coordinates)
!
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const = au_length * 1.d+10

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   skip if this is not my bead   */
         if ( mod( j-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   only for rank zero   */
         if ( myrank_sub .ne. 0 ) cycle

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   open sample file   */
         open( iounit, file = './predict.' // char_num // '.xsf' )

!        /*   for periodic boundary condition   */
         if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

            write( iounit, '(a)' ) '# total energy = 0.0000000000000 eV'
            write( iounit, '(a)' ) 'CRYSTAL'
            write( iounit, '(a)' ) 'PRIMVEC'

!           /*   lattice vectors   */
            if ( method(1:6) .eq. 'REHMC ' ) then
               ax = box_bead(1,1,j) * const
               ay = box_bead(2,1,j) * const
               az = box_bead(3,1,j) * const
               bx = box_bead(1,2,j) * const
               by = box_bead(2,2,j) * const
               bz = box_bead(3,2,j) * const
               cx = box_bead(1,3,j) * const
               cy = box_bead(2,3,j) * const
               cz = box_bead(3,3,j) * const
            else
               ax = box(1,1) * const
               ay = box(2,1) * const
               az = box(3,1) * const
               bx = box(1,2) * const
               by = box(2,2) * const
               bz = box(3,2) * const
               cx = box(1,3) * const
               cy = box(2,3) * const
               cz = box(3,3) * const
            end if

!           /*   write three lines   */
            write( iounit, '(3f16.8)' ) ax, ay, az
            write( iounit, '(3f16.8)' ) bx, by, bz
            write( iounit, '(3f16.8)' ) cx, cy, cz

!           /*   write one line   */
            write( iounit, '(a)' ) 'PRIMCOORD'

!           /*   write one line   */
            write( iounit, '(i8,i2)' ) natom, 1

!        /*   for free boundary condition   */
         else if ( iboundary .eq. 0 ) then

!           /*   write one line   */
            write( iounit, '(a)' ) 'ATOMS'

!        /*   for periodic boundary condition   */
         end if

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   geometry in angstroms   */
            xa = x(i,j) * const
            ya = y(i,j) * const
            za = z(i,j) * const

!           /*   write one line   */
            write( iounit, '(a4,3f16.8)' ) species(i)(1:4), xa, ya, za

!        /*   loop of atoms   */
         end do

!        /*   close input file   */
         close( iounit )

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   check existence of network file                            */
!-----------------------------------------------------------------------

!     /*   reset flag   */
      itest = 0

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   error flag   */
         iann_start_aenet = 1

!        /*  loop of atomic types   */
         do i = 1, ntype_aenet

!           /*   subdirectory file  */
            char_line = trim(adjustl(network_aenet(i)))

!           /*   check existence of file   */
            call testfile ( char_line, len(char_line), &
     &                      itest, iounit )

!           /*   error flag   */
            if ( itest .ne. 0 ) iann_start_aenet = 0

!c           /*   detect error   */
!            if ( itest .ne. 0 ) then
!               write( 6, '(a)' ) 'Error - trained network not found.'
!               exit
!            end if

!        /*  loop of atomic types   */
         end do

!     /*   master rank only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( iann_start_aenet )

!-----------------------------------------------------------------------
!     /*   execute predict                                            */
!-----------------------------------------------------------------------

!     /*   if network file exists   */
      if ( iann_start_aenet .eq. 1 ) then

!        /*   stress   */
         stress_vec_sum(:,:) = 0.d0

!        /*   loop of beads   */
         do j = 1, nbead

!           /*   skip if this is not my bead   */
            if ( mod( j-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   bead number   */
            call int3_to_char( j, char_num )

!           /*   open file   */
            open ( iounit_predict, &
     &             file = 'predict.' // char_num // '.scr' )

!           /*   execute predict   */
            call predict_sub( 'predict.' // char_num // '.in', &
     &                        iounit_predict, Etot_vec, &
     &                        forCart_vec, stress_vec, natom, 1 )

!           /*   close file   */
            close( iounit_predict )

!           /*   unit conversion factor   */
            const_e =  au_charge / au_energy

!           /*   unit conversion factor   */
            const_f =  au_charge / au_energy * au_length / 1.d-10

!           /*   eV to hartree   */
            if ( myrank_sub .eq. 0 ) then
               pot(j) = Etot_vec(1) * const_e
            end if

!           /*   eV per angstrom to hartree per bohr   */
            if ( myrank_sub .eq. 0 ) then
               fx(:,j) = forCart_vec(1,:,1) * const_f
               fy(:,j) = forCart_vec(2,:,1) * const_f
               fz(:,j) = forCart_vec(3,:,1) * const_f
            end if

!           /*   sum of stress   */
            if ( myrank_sub .eq. 0 ) then
               stress_vec_sum(:,:) = stress_vec_sum(:,:) &
     &                             + stress_vec(:,:,1)
               if ( method(1:6) .eq. 'REHMC ' ) then
                  vir_bead(:,:,j)  = stress_vec(:,:,1)
               end if
            end if

!        /*   loop of beads   */
         end do

!     /*   if network file does not exist   */
      else

!        /*   potential   */
         pot(:) = 0.d0

!        /*   force   */
         fx(:,:) = 0.d0
         fy(:,:) = 0.d0
         fz(:,:) = 0.d0

!        /*   stress   */
         stress_vec_sum(:,:) = 0.d0

!     /*   end of if statement   */
      end if

!-----------------------------------------------------------------------
!     /*   change units from kilobar to hartree/bohr**3               */
!-----------------------------------------------------------------------

      if ( myrank_sub .eq. 0 ) then
        stress_tensor(:,:) = stress_vec_sum(:,:) &
     &                     * 1.e+8 / au_energy * au_length**3
         if ( method(1:6) .eq. 'REHMC ' ) then
            do j = 1, nbead
               vir_bead(:,:,j) = vir_bead(:,:,j) &
     &            * 1.e+8 / au_energy * au_length**3 * volume_bead(j)
            end do
         end if
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_bcast_real_1_sub( pot, nbead )
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   force   */
      call my_mpi_bcast_real_2_sub( fx, natom, nbead )
      call my_mpi_bcast_real_2_sub( fy, natom, nbead )
      call my_mpi_bcast_real_2_sub( fz, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!     /*   stress   */
      call my_mpi_bcast_real_2_sub( stress_tensor, 3, 3 )
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!     /*   virial   */
      if ( method(1:6) .eq. 'REHMC ' ) then
         call my_mpi_bcast_real_3_sub( vir_bead, 3, 3, nbead )
         call my_mpi_allreduce_real_3_main( vir_bead, 3, 3, nbead )
      end if

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      if ( method(1:6) .eq. 'REHMC ' ) then

         do j = 1, nbead

            vir(1,1) = vir(1,1) + vir_bead(1,1,j) / nbead
            vir(1,2) = vir(1,2) + vir_bead(1,2,j) / nbead
            vir(1,3) = vir(1,3) + vir_bead(1,3,j) / nbead
            vir(2,1) = vir(2,1) + vir_bead(2,1,j) / nbead
            vir(2,2) = vir(2,2) + vir_bead(2,2,j) / nbead
            vir(2,3) = vir(2,3) + vir_bead(2,3,j) / nbead
            vir(3,1) = vir(3,1) + vir_bead(3,1,j) / nbead
            vir(3,2) = vir(3,2) + vir_bead(3,2,j) / nbead
            vir(3,3) = vir(3,3) + vir_bead(3,3,j) / nbead

         end do

      else

         do j = 1, nbead

            vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
            vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
            vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
            vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
            vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
            vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
            vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
            vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
            vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

         end do

      end if

!-----------------------------------------------------------------------
!     /*   return to main routine                                     */
!-----------------------------------------------------------------------

      return

!-----------------------------------------------------------------------
!     /*   THIS PART IS OBSOLETE: predict execution command           */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   skip if this is not my bead   */
         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   execute predict   */
         call system( 'predict.x ./predict.' // char_num // '.in' // &
     &                ' > ./predict.' // char_num // '.scr' )

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   THIS PART IS OBSOLETE: read potential in atomic units      */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const =  au_charge / au_energy

!     /*   reset error flag   */
      ierr = 1

!     /*   reset error flag   */
      if ( myrank .ge. nprocs ) ierr = 0

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   skip if this is not my bead   */
         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   open output file   */
         open ( iounit, file = './predict.' // char_num // '.scr' )

!        /*   line by line   */
         do

!           /*   read one line   */
            read ( iounit, '(a)', iostat=jerr ) char_line

!           /*   end of the file   */
            if ( jerr .ne. 0 ) exit

!           /*   remove blanks   */
            char_word = adjustl(char_line)

!           /*   if line matches   */
            if ( char_word(1:12) .eq. 'Total energy' ) then

!              /*   read potential in electron volts   */
               backspace( iounit )

!              /*   read potential in electron volts   */
               read ( char_line, *, iostat=ierr ) &
     &            char_word, char_word, char_word, pot(j)

!              /*   change in atomic units   */
               pot(j) = pot(j) * const

!              /*   end of read   */
               exit

!           /*   if line matches   */
            end if

!        /*   line by line   */
         end do

!        /*   close output file   */
         close( iounit )

!        /*   reset error flag   */
         if ( ierr .ne. 0 ) exit

!     /*   loop of beads   */
      end do

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine force_aenet_MPI', 26 )

!-----------------------------------------------------------------------
!     /*   THIS PART IS OBSOLETE: read forces in atomic units         */
!-----------------------------------------------------------------------

!     /*   conversion factor   */
      const =  au_charge / au_energy * au_length / 1.d-10

!     /*   reset error flag   */
      ierr = 1

!     /*   reset error flag   */
      if ( myrank .ge. nprocs ) ierr = 0

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   skip if this is not my bead   */
         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!        /*   bead number   */
         call int3_to_char( j, char_num )

!        /*   open output file   */
         open ( iounit, file = './predict.' // char_num // '.scr' )

!        /*   line by line   */
         do

!           /*   read one line   */
            read ( iounit, '(a)', iostat=jerr ) char_line

!           /*   end of the file   */
            if ( jerr .ne. 0 ) exit

!           /*   remove blanks   */
            char_word = adjustl(char_line)

!           /*   if line matches   */
            if ( char_word(1:5) .eq. '(Ang)' ) then

!              /*   skip one line   */
               read ( iounit, *, iostat=ierr )

!              /*   read potential in electron volts per angstroms  */
               do i = 1, natom
                  read ( iounit, *, iostat=ierr ) &
     &              char_word, char_word, char_word, &
     &                 char_word, fx(i,j), fy(i,j), fz(i,j)
               end do

!              /*   electron volts per angstrom to au   */
               fx(:,j) = fx(:,j) * const
               fy(:,j) = fy(:,j) * const
               fz(:,j) = fz(:,j) * const

!              /*   end of read   */
               exit

!           /*   end of if statement   */
            end if

!        /*   line by line   */
         end do

!        /*   close output file   */
         close( iounit )

!        /*   reset error flag   */
         if ( ierr .ne. 0 ) exit

!     /*   loop of beads   */
      end do

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine force_aenet_MPI', 26 )

!-----------------------------------------------------------------------
!     /*   THIS PART IS OBSOLETE: read virial in atomic units         */
!-----------------------------------------------------------------------

!     /*   WARNING: currently virial not available for aenet   */
      vir(:,:) = 0.d0

      return
      end



#endif
#endif
