!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Dec 27, 2018 by M. Shiga
!      Description:     polymers free energy
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine pmfc_polymers_atom_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : physmass, temperature, natom

      use polymers_variables, only : pmfc_poly, &
     &    ekinvir_poly, epot_poly, entropy_poly, rc_arc, pmfc_arc, &
     &    narc, npoly, ngrid_poly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s

!     /*   real numbers   */
      real(8) :: dx, dy, dz, fx, fy, fz, factor, sum, drc2

!     /*   integers   */
      integer :: i, j, k, m

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r1, r2, f1, f2

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_poly_atom

!-----------------------------------------------------------------------
!     /*   initialize spline - s:f                                    */
!-----------------------------------------------------------------------

      call spline_init_frc_poly_atom

!-----------------------------------------------------------------------
!     /*   calculate free energy                                      */
!-----------------------------------------------------------------------

!     /*   free energy at the reactant is zero   */
      pmfc_poly(1) = epot_poly(1)

!     /*   reaction coordinate   */
      rc_arc(1) = 0.d0

!     /*   reaction coordinate   */
      pmfc_arc(1) = 0.d0

!     /*   parameter s at the reactant is zero  */
      s = 0.d0

!     /*   free energy difference   */
      sum = 0.d0

!     /*   get old mass weighted coordinates and old force  */
      call spline_crd_frc_poly_atom( r1, f1, s )

!     /*   scan from the reactant to the product   */
      do j = 2, narc

!        /*   parameter s   */
         s = dble(j-1) / dble(narc-1)

!        /*   get new mass weighted coordinates and new force  */
         call spline_crd_frc_poly_atom( r2, f2, s )

!        /*   calculate arc length by linear approximation   */
         k = 0

!        /*   measure of reaction coordinate   */
         drc2 = 0.d0

         do i = 1, natom

!           /*   mass factor   */
            factor = sqrt( physmass(i) )

!           /*   index   */
            k = k + 1

!           /*   displacement    */
            dx = ( r2(k) - r1(k) ) / factor

!           /*   mean force    */
            fx = 0.5d0 * ( f2(k) + f1(k) )

!           /*   index   */
            k = k + 1

!           /*   displacement    */
            dy = ( r2(k) - r1(k) ) / factor

!           /*   mean force    */
            fy = 0.5d0 * ( f2(k) + f1(k) )

!           /*   index   */
            k = k + 1

!           /*   displacement    */
            dz = ( r2(k) - r1(k) ) / factor

!           /*   mean force    */
            fz = 0.5d0 * ( f2(k) + f1(k) )

!           /*   measure of reaction coordinate   */
            drc2 = drc2 + dx*dx + dy*dy + dz*dz

!           /*   reversible work   */
            sum = sum - fx*dx - fy*dy - fz*dz

         end do

!        /*   reaction coordinate   */
         rc_arc(j) = rc_arc(j-1) + sqrt(drc2)

!        /*   reversible work   */
         pmfc_arc(j) = pmfc_arc(1) + sum

         if ( mod(j-1,ngrid_poly-1) .eq. 0 ) then

            i = (j-1)/(ngrid_poly-1) + 1

            pmfc_poly(i) = pmfc_poly(1) + sum

         end if

!        /*   save old mass weighted coordinates */
         r1(:) = r2(:)

!        /*   save old forces   */
         f1(:) = f2(:)

      end do

      do m = 1, npoly

         entropy_poly(m) = &
     &     ( ekinvir_poly(m) + epot_poly(m) - pmfc_poly(m) ) &
     &     / temperature

      end do

      return
      end





!***********************************************************************
      subroutine pmfc_polymers_cons_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons
      use afed_variables, only : fictmass_afed

      use common_variables, only : temperature

      use polymers_variables, only : pmfc_poly, &
     &    ekinvir_poly, epot_poly, entropy_poly, rc_arc, pmfc_arc, &
     &    narc, npoly, ngrid_poly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s

!     /*   real numbers   */
      real(8) :: dx, fx, sum, drc2

!     /*   integers   */
      integer :: i, j, m

!     /*   mass weighted coordinates   */
      real(8), dimension(ncons) :: r1, r2, f1, f2

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_poly_cons( 0 )

!-----------------------------------------------------------------------
!     /*   initialize spline - s:f                                    */
!-----------------------------------------------------------------------

      call spline_init_frc_poly_cons

!-----------------------------------------------------------------------
!     /*   calculate free energy                                      */
!-----------------------------------------------------------------------

!     /*   free energy at the reactant is zero   */
      pmfc_poly(1) = epot_poly(1)

!     /*   reaction coordinate   */
      rc_arc(1) = 0.d0

!     /*   reaction coordinate   */
      pmfc_arc(1) = 0.d0

!     /*   parameter s at the reactant is zero  */
      s = 0.d0

!     /*   free energy difference   */
      sum = 0.d0

!     /*   get old mass weighted coordinates and old force  */
      call spline_crd_frc_poly_cons( r1, f1, s )

!     /*   scan from the reactant to the product   */
      do j = 2, narc

!        /*   parameter s   */
         s = dble(j-1) / dble(narc-1)

!        /*   get new mass weighted coordinates and new force  */
         call spline_crd_frc_poly_cons( r2, f2, s )

!        /*   measure of reaction coordinate   */
         drc2 = 0.d0

!        /*   loop of constraints   */
         do i = 1, ncons

!           /*   displacement    */
            dx = ( r2(i) - r1(i) ) / sqrt( fictmass_afed(i) )

!           /*   measure of reaction coordinate   */
            drc2 = drc2 + dx*dx

!           /*   gradient   */
            fx = 0.5d0 * ( f2(i) + f1(i) )

!           /*   reversible work   */
            sum = sum - fx * dx

!        /*   loop of constraints   */
         end do

!        /*   reaction coordinate   */
         rc_arc(j) = rc_arc(j-1) + sqrt(drc2)

!        /*   reversible work   */
         pmfc_arc(j) = pmfc_arc(1) + sum

         if ( mod(j-1,ngrid_poly-1) .eq. 0 ) then

            i = (j-1)/(ngrid_poly-1) + 1

            pmfc_poly(i) = pmfc_poly(1) + sum

         end if

!        /*   save old mass weighted coordinates */
         r1(:) = r2(:)

!        /*   save old forces   */
         f1(:) = f2(:)

      end do

      do m = 1, npoly

         entropy_poly(m) = &
     &     ( ekinvir_poly(m) + epot_poly(m) - pmfc_poly(m) ) &
     &     / temperature

      end do

      return
      end





!***********************************************************************
      subroutine pmfc_polymers_cons_revised_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

      use common_variables, only : temperature

      use polymers_variables, only : pmfc_poly, scons_poly, fcons_poly, &
     &    ekinvir_poly, epot_poly, entropy_poly, npoly

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   real numbers   */
      real(8) :: dx, fx, sum

!     /*   integers   */
      integer :: i, j, m

!-----------------------------------------------------------------------
!     /*   calculate free energy                                      */
!-----------------------------------------------------------------------

!     /*   initialize   */
      pmfc_poly(:) = 0.d0

!     /*   free energy at the reactant is zero   */
      pmfc_poly(1) = epot_poly(1)

!     /*   free energy difference   */
      sum = 0.d0

!     /*   calculate arc length by linear approximation   */

      do j = 2, npoly

         do i = 1, ncons

            dx = ( scons_poly(i,j) - scons_poly(i,j-1) )

            fx = 0.5d0 * ( fcons_poly(i,j-1) + fcons_poly(i,j) )

            sum = sum - fx * dx

         end do

         pmfc_poly(j) = pmfc_poly(1) + sum

      end do

      do m = 1, npoly

         entropy_poly(m) = &
     &     ( ekinvir_poly(m) + epot_poly(m) - pmfc_poly(m) ) &
     &     / temperature

      end do

      return
      end
