!***********************************************************************
      subroutine setup_pimd_nvt_qtst_MPI
!***********************************************************************
!=======================================================================
!
!     set up pimd in nvt ensemble (massive nose-hoover chains)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ibath_start, ipos_start, iounit, nstep, iatom_qtst

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   atom fixed in z direction                                  */
!-----------------------------------------------------------------------

      call read_int1_MPI ( iatom_qtst, '<iatom_qtst>', 10, iounit )

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams_MPI

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix_MPI

!-----------------------------------------------------------------------
!     /*   thermostat masses for path integral MD                     */
!-----------------------------------------------------------------------

      call setup_qmass_pimd

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_pimd_MPI

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid_MPI
         call init_mode_MPI
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position_MPI( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position_MPI( 2 )
      else
         call error_handling_MPI &
     &      ( 1, 'subroutine setup_pimd_nvt_mnhc_MPI', 34 )
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity_qtst_MPI
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity_MPI( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity_MPI( 2 )
      else
         call error_handling_MPI &
     &      ( 1, 'subroutine setup_pimd_nvt_mnhc_MPI', 34 )
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

!!!      call correct_velocity_MPI

!-----------------------------------------------------------------------
!     /*   set up thermostat positions and velocities                 */
!-----------------------------------------------------------------------

!     /*   for initial start   */

      if ( ibath_start .le. 0 ) then
!        /*   Maxwell distribution of thermostat velocities   */
         call init_bath_mnhc_MPI

!     /*   for restart   */

      else if ( ibath_start .eq. 1 ) then
!        /*   read in thermostat position and momentum   */
         call restart_bath_mnhc_MPI( 1 )
      else if ( ibath_start .eq. 2 ) then
!        /*   read in thermostat position and momentum   */
         call restart_bath_mnhc_MPI( 1 )
      else
         call error_handling_MPI &
     &      ( 1, 'subroutine setup_pimd_nvt_mnhc_MPI', 34 )
      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nvt_mnhc_MPI

         call my_mpi_finalize_2

         stop

      end if

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

      call setup_cons_MPI

      return
      end





!***********************************************************************
      subroutine init_velocity_qtst_MPI
!***********************************************************************
!=======================================================================
!
!    Translational velocities from Maxwell-Boltzmann distribution.
!    The distribution is determined by temperature and mass.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   vsigma: standard deviation of Maxwell distribution         */
!-----------------------------------------------------------------------

      call nm_velocity_qtst_MPI

!-----------------------------------------------------------------------
!     /*   remove net linear and angular momentum                     */
!-----------------------------------------------------------------------

!!!      call correct_velocity_MPI

!-----------------------------------------------------------------------
!     /*   scale velocity:  centroid and non-centroid                 */
!-----------------------------------------------------------------------

!!!      call scale_velocity

!-----------------------------------------------------------------------
!     /*   normal modes to Cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity_MPI ( 0 )

      return
      end





!***********************************************************************
      subroutine nm_velocity_qtst_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, beta, vux, vuy, vuz, natom, nbead, iatom_qtst, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         do j = 1, natom
            vsigma = dsqrt(1.d0/beta/fictmass(j,1))
            if ( j .eq. iatom_qtst ) then
               vux(j,1) = vsigma*gasdev()
               vuy(j,1) = vsigma*gasdev()
               vuz(j,1) = 0.d0
            else
               vux(j,1) = 0.d0
               vuy(j,1) = 0.d0
               vuz(j,1) = 0.d0
            end if
         end do

         do i = 2, nbead
         do j = 1, natom
            vsigma = dsqrt(1.d0/beta/fictmass(j,i))
            vux(j,i) = vsigma*gasdev()
            vuy(j,i) = vsigma*gasdev()
            vuz(j,i) = vsigma*gasdev()
         end do
         end do

      end if

      call my_mpi_bcast_real_2 ( vux, natom, nbead )
      call my_mpi_bcast_real_2 ( vuy, natom, nbead )
      call my_mpi_bcast_real_2 ( vuz, natom, nbead )

      return
      end





!***********************************************************************
      subroutine pimdcycle_nvt_qtst_MPI
!***********************************************************************
!=======================================================================
!
!     the molecular dynamics cycle:  thermostat type III.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nvt_mnhc_qtst_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update thermostats attached to centroids  */
         call update_mnhc_cent_qtst

!        /*   update the velocities by interatomic forces   */
         call update_vel_qtst

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_MPI

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_MPI

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel_qtst

!        /*   update thermostats attached to centroids  */
         call update_mnhc_cent_qtst

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nvt_mnhc_qtst_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pimd_nvt_mnhc_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine update_vel_qtst
!***********************************************************************
!=======================================================================
!
!     update normal mode velocities by interatomic forces
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, fux, fuy, fuz, dt, fictmass, natom, nbead, &
     &   amu_mass_earth, iatom_qtst

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      real(8) :: fm, hm

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do i = 1, natom

         fm = fictmass(i,1)
         hm = fictmass(i,1) * amu_mass_earth

         if ( i .eq. iatom_qtst ) then
            vux(i,1) = vux(i,1) + 0.5d0*dt*fux(i,1)/fm
            vuy(i,1) = vuy(i,1) + 0.5d0*dt*fuy(i,1)/fm
            vuz(i,1) = vuz(i,1) + 0.5d0*dt*fuz(i,1)/hm
         else
            vux(i,1) = vux(i,1) + 0.5d0*dt*fux(i,1)/hm
            vuy(i,1) = vuy(i,1) + 0.5d0*dt*fuy(i,1)/hm
            vuz(i,1) = vuz(i,1) + 0.5d0*dt*fuz(i,1)/hm
         end if

      end do

      do j = 2, nbead
      do i = 1, natom

         fm = fictmass(i,j)

         vux(i,j) = vux(i,j) + 0.5d0*dt*fux(i,j)/fm
         vuy(i,j) = vuy(i,j) + 0.5d0*dt*fuy(i,j)/fm
         vuz(i,j) = vuz(i,j) + 0.5d0*dt*fuz(i,j)/fm

      end do
      end do

      return
      end





!***********************************************************************
      subroutine update_mnhc_cent_qtst
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, ysweight, qmass_cent, dt, &
     &   fxbath_cent, fybath_cent, fzbath_cent, gkt, vxbath_cent, &
     &   vybath_cent, vzbath_cent, xbath_cent, ybath_cent, zbath_cent, &
     &   nys, nnhc, ncolor, natom, amu_mass_earth, iatom_qtst

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, inhc, j, l

      real(8) :: dt_ys, dkinx, dkiny, dkinz, scalex, scaley, scalez, &
     &           vxfact, vyfact, vzfact, pvxfact, pvyfact, pvzfact, &
     &           fm, hm

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = 1, ncolor
         do j = 1, natom

!           /*   calculate total kinetic energy of the system   */

            fm = fictmass(j,1)
            hm = fictmass(j,1) * amu_mass_earth

            if ( j .eq. iatom_qtst ) then
               dkinx = fm*vux(j,1)*vux(j,1)
               dkiny = fm*vuy(j,1)*vuy(j,1)
               dkinz = hm*vuz(j,1)*vuz(j,1)
            else
               dkinx = hm*vux(j,1)*vux(j,1)
               dkiny = hm*vuy(j,1)*vuy(j,1)
               dkinz = hm*vuz(j,1)*vuz(j,1)
            end if

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) = (dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) = (dkinz - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fybath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vybath_cent(j,i-1,l) &
     &           *vybath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fzbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vzbath_cent(j,i-1,l) &
     &           *vzbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_cent(j,nnhc-inhc,l) = &
     &            vybath_cent(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_cent(j,nnhc-inhc,l) = &
     &            vzbath_cent(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cent(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
               ybath_cent(j,inhc,l) = ybath_cent(j,inhc,l) &
     &            + 0.5d0*vybath_cent(j,inhc,l)*dt_ys
               zbath_cent(j,inhc,l) = zbath_cent(j,inhc,l) &
     &            + 0.5d0*vzbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cent(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys
               vybath_cent(j,inhc,l) = &
     &            vybath_cent(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,inhc,l)*vyfact*dt_ys
               vzbath_cent(j,inhc,l) = &
     &            vzbath_cent(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,inhc,l)*vzfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fybath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vybath_cent(j,inhc,l) &
     &            *vybath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fzbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vzbath_cent(j,inhc,l) &
     &            *vzbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex
            vuy(j,1) = vuy(j,1)*scaley
            vuz(j,1) = vuz(j,1)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = ncolor, 1, -1
         do j = natom, 1, -1

!           /*   calculate total kinetic energy of the system   */

            fm = fictmass(j,1)
            hm = fictmass(j,1) * amu_mass_earth

            if ( j .eq. iatom_qtst ) then
               dkinx = fm*vux(j,1)*vux(j,1)
               dkiny = fm*vuy(j,1)*vuy(j,1)
               dkinz = hm*vuz(j,1)*vuz(j,1)
            else
               dkinx = hm*vux(j,1)*vux(j,1)
               dkiny = hm*vuy(j,1)*vuy(j,1)
               dkinz = hm*vuz(j,1)*vuz(j,1)
            end if

            scalex = 1.d0
            scaley = 1.d0
            scalez = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) = (dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) = (dkinz - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fybath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vybath_cent(j,i-1,l) &
     &           *vybath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
              fzbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vzbath_cent(j,i-1,l) &
     &           *vzbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vyfact &
     &            = exp(-0.125d0*vybath_cent(j,nnhc-inhc+1,l)*dt_ys)
               vzfact &
     &            = exp(-0.125d0*vzbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys
               vybath_cent(j,nnhc-inhc,l) = &
     &            vybath_cent(j,nnhc-inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,nnhc-inhc,l)*vyfact*dt_ys
               vzbath_cent(j,nnhc-inhc,l) = &
     &            vzbath_cent(j,nnhc-inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,nnhc-inhc,l)*vzfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)
            pvyfact = exp(-0.5d0*vybath_cent(j,1,l)*dt_ys)
            pvzfact = exp(-0.5d0*vzbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact
            scaley = scaley*pvyfact
            scalez = scalez*pvzfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)
            fybath_cent(j,1,l) &
     &         = (scaley*scaley*dkiny - gkt)/qmass_cent(1,l)
            fzbath_cent(j,1,l) &
     &         = (scalez*scalez*dkinz - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
               ybath_cent(j,inhc,l) = ybath_cent(j,inhc,l) &
     &            + 0.5d0*vybath_cent(j,inhc,l)*dt_ys
               zbath_cent(j,inhc,l) = zbath_cent(j,inhc,l) &
     &            + 0.5d0*vzbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)
               vyfact = exp(-0.125d0*vybath_cent(j,inhc+1,l)*dt_ys)
               vzfact = exp(-0.125d0*vzbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys
               vybath_cent(j,inhc,l) = &
     &            vybath_cent(j,inhc,l)*vyfact*vyfact &
     &            + 0.25d0*fybath_cent(j,inhc,l)*vyfact*dt_ys
               vzbath_cent(j,inhc,l) = &
     &            vzbath_cent(j,inhc,l)*vzfact*vzfact &
     &            + 0.25d0*fzbath_cent(j,inhc,l)*vzfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fybath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vybath_cent(j,inhc,l) &
     &            *vybath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)
               fzbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vzbath_cent(j,inhc,l) &
     &            *vzbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys
            vybath_cent(j,nnhc,l) = vybath_cent(j,nnhc,l) &
     &         + 0.25d0*fybath_cent(j,nnhc,l)*dt_ys
            vzbath_cent(j,nnhc,l) = vzbath_cent(j,nnhc,l) &
     &         + 0.25d0*fzbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex
            vuy(j,1) = vuy(j,1)*scaley
            vuz(j,1) = vuz(j,1)*scalez

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine standard_nvt_mnhc_qtst_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!     thermostat type III.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, gkt, &
     &   hamiltonian_sys, omega_p2, dnmmass, ux, uy, uz, qmass_cent, &
     &   xbath_cent, ybath_cent, zbath_cent, qmass, ebath_cent, &
     &   vxbath_cent, vybath_cent, vzbath_cent, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath,  ebath_mode, ndof, iprint_std, &
     &   nbead, natom, ncolor, nnhc, iatom_qtst

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, m

      real(8) :: factqk

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init_MPI( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy_qtst

      if ( iatom_qtst .eq. 0 ) then
         if ( (ndof-3*natom) .eq. 0 ) then
            temp = 0.d0
         else
            temp = 2.d0*ekin/dble(ndof-3*natom)/boltz
         end if
      else
         if ( (ndof-3*natom) .eq. 0 ) then
            temp = 0.d0
         else
            temp = 2.d0*ekin/dble(ndof-3*natom+2)/boltz
         end if
      end if

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, natom
         ebath_cent = ebath_cent &
     &   + 0.5d0*qmass_cent(i,m)*vxbath_cent(j,i,m)*vxbath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vybath_cent(j,i,m)*vybath_cent(j,i,m) &
     &   + 0.5d0*qmass_cent(i,m)*vzbath_cent(j,i,m)*vzbath_cent(j,i,m) &
     &   + gkt*xbath_cent(j,i,m) &
     &   + gkt*ybath_cent(j,i,m) &
     &   + gkt*zbath_cent(j,i,m)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

      do i = 2, nbead
         do j = 1, nnhc
         do k = 1, natom
            ebath_mode = ebath_mode &
     &           + 0.5d0*qmass(i)*vxbath(k,j,i)*vxbath(k,j,i) &
     &           + 0.5d0*qmass(i)*vybath(k,j,i)*vybath(k,j,i) &
     &           + 0.5d0*qmass(i)*vzbath(k,j,i)*vzbath(k,j,i) &
     &           + gkt*xbath(k,j,i) &
     &           + gkt*ybath(k,j,i) &
     &           + gkt*zbath(k,j,i)
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output_MPI

      return
      end





!***********************************************************************
      subroutine kinetic_energy_qtst
!***********************************************************************
!=======================================================================
!
!     calculate ekin = fictitious kinetic energy of the system
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ekin, fictmass, vux, vuy, vuz, ekin, natom, nbead, &
     &   amu_mass_earth, iatom_qtst

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom
      real(8) :: fm, hm

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do iatom = 1, natom

         fm = fictmass(iatom,1)
         hm = fictmass(iatom,1) * amu_mass_earth

         if ( iatom .eq. iatom_qtst ) then
            ekin = ekin &
     &        + fm*vux(iatom,1)*vux(iatom,1) &
     &        + fm*vuy(iatom,1)*vuy(iatom,1) &
     &        + hm*vuz(iatom,1)*vuz(iatom,1)
         else
            ekin = ekin &
     &        + hm*vux(iatom,1)*vux(iatom,1) &
     &        + hm*vuy(iatom,1)*vuy(iatom,1) &
     &        + hm*vuz(iatom,1)*vuz(iatom,1)
         end if

      end do

      do imode = 2, nbead
      do iatom = 1, natom

         fm = fictmass(iatom,imode)

         ekin = ekin &
     &      + fm*vux(iatom,imode)*vux(iatom,imode) &
     &      + fm*vuy(iatom,imode)*vuy(iatom,imode) &
     &      + fm*vuz(iatom,imode)*vuz(iatom,imode)

      end do
      end do

      ekin = 0.5d0*ekin

      return
      end

