!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 8, 2026 by M. Shiga
!      Description:     main routine of PIMD
!
!///////////////////////////////////////////////////////////////////////
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
!      PIMD version 2.7.1
!
!      Author:        M. Shiga
!
!      Last Updated:  Jan 8, 2026.
!
!      Copyright(C) 2016-2026  M. Shiga  All rights reserved.
!
!      Licensed under the Apache License, Version 2.0 (the "License");
!      you may not use this file except in compliance with the License.
!      You may obtain a copy of the License at
!
!      http://www.apache.org/licenses/LICENSE-2.0
!
!      Unless required by applicable law or agreed to in writing,
!      software distributed under the License is distributed on an
!      "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
!      either express or implied. See the License for the specific
!      language governing permissions and limitations under the License.
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!***********************************************************************
      program pimd_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   method, nph_type, npt_type, ntt_type, nth_type, ensemble, &
     &   ipotential, irem_type, iorder_hmc, bath_type, nbead, myrank, &
     &   ends_string, equation_om, afed_type, tamd_type, logmfd_type, &
     &   qmmm_embedding, code, box, xmpi

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   name of the code                                           */
!-----------------------------------------------------------------------

      code = 'PIMD_MPI'

!-----------------------------------------------------------------------
!     /*  initialize MPI                                              */
!-----------------------------------------------------------------------

      call my_mpi_init_2

!-----------------------------------------------------------------------
!     /*   print titles                                               */
!-----------------------------------------------------------------------

      call print_titles_MPI

!-----------------------------------------------------------------------
!     /*   set parameters                                             */
!-----------------------------------------------------------------------

      call setparams_MPI

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      call setallocation

!-----------------------------------------------------------------------
!     /*   prepare condition                                          */
!-----------------------------------------------------------------------

      call setcondition_MPI

!-----------------------------------------------------------------------
!     /*   initialize external programs                               */
!-----------------------------------------------------------------------

      call init_extprograms_MPI

!-----------------------------------------------------------------------
!     /*  set up the calculation                                      */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         call setup_geometry_MPI

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         call setup_testforce_MPI

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         call setup_testvirial_MPI

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         call setup_geometry_MPI

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         call setup_geometry_MPI

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         call setup_omopt_MPI

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_geoopt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &         'Error - GEOOPT not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         call setup_nma_MPI

!         if      ( nbead .eq. 1 ) then
!
!            call setup_nma_MPI
!
!         else
!
!            if ( myrank .eq. 0 ) then
!               write( 6, '(a)' )
!     &         'Error - NMA not supported for nbead > 1.'
!               write( 6, '(a)' )
!            end if
!
!            call error_handling_MPI( 1, 'program pimd_MPI', 16 )
!
!         end if

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_geoopt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &         'Error - PHONON not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_sd_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &         'Error - SD not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_boxopt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &         'Error - BOXOPT not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== full optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_fullopt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &         'Error - FULLOPT not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_elastic_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - ELASTIC not supported for nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if      ( bath_type(1:5) .eq. 'NONE ' ) then

               call setup_md_nve_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - MD, NVE ensemble. <bath_type> must be' &
     &               // ' NONE.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI( 1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_md_nvt_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - MD, NVT ensemble. <bath_type> must be' &
     &               // ' MNHC.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI( 1, 'program pimd_MPI', 16 )

            end if

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - MD is supported for NVT and NVE' &
     &            // ' ensembles only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI( 1, 'program pimd_MPI', 16 )

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if      ( nbead .eq. 1 ) then

               call setup_pimd_nve_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NVE ensemble not supported for' &
     &               // ' nbead > 1.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI( 1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               call setup_pimd_nvt_nhc_MPI

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               call setup_pimd_nvt_nhcs_MPI

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_pimd_nvt_mnhc_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NVT ensemble. <bath_type> must be' &
     &               // ' either NHC, NHCS or MNHC.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call setup_pimd_nph_MPI

               do i = 1,   3
               do j = i+1, 3

                  if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                     if ( myrank .eq. 0 ) then
                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <nph_type> = PPHEX.'
                        write( 6, '(a)' )
                     end if

                     call error_handling_MPI &
     &                  ( 1, 'program pimd_MPI', 16 )

                  end if

               end do
               end do

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call setup_pimd_nph_MPI

               do i = 1,   3
               do j = i+1, 3

                  if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                     if ( myrank .eq. 0 ) then
                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <nph_type> = PPHEX.'
                        write( 6, '(a)' )
                     end if

                     call error_handling_MPI &
     &                  ( 1, 'program pimd_MPI', 16 )

                  end if

               end do
               end do

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call setup_pimd_nph_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NPH ensemble. <nph_type> must be' &
     &               // ' CUBIC1, CUBIC2, PPHEX.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call setup_pimd_nth_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NTH ensemble.' &
     &               // ' <nth_type> must be PPHEX.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pimd_npt_MPI

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        if ( myrank .eq. 0 ) then
                           write( 6, '(a)' ) &
     &                        'Warning - The box is not cubic.' // &
     &                        ' Set <npt_type> = PPHEX.'
                           write( 6, '(a)' )
                        end if

                        call error_handling_MPI &
     &                     ( 1, 'program pimd_MPI', 16 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pimd_npt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIMD, NPT ensemble.' &
     &                  // ' <npt_type> must be CUBIC2 or PPHEX.'
                     write( 6, '(a)' )

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

                  end if

               end if

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NPT ensemble. <bath_type> must be' &
     &               // ' MNHC.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_pimd_ntt_MPI

            else

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pimd_ntt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIMD, NTT ensemble.' &
     &                  // ' <npt_type> must be PPHEX.'
                     write( 6, '(a)' )

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

                  end if

               end if

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIMD, NTT ensemble. <bath_type> must be' &
     &               // ' MNHC.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:5) .eq. 'QTST ' ) then

            call setup_pimd_nvt_qtst_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - PIMD must be either NVT, NPH, PTH,' &
     &            // ' NPT or NTT ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - PIHMC, NVE ensemble not supported.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call setup_pihmc_nvt_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call setup_pihmc_nvt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NVT ensemble (DUAL).' &
     &                  // ' <iorder_hmc> must be either 2 or 4.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call setup_pihmc_nvt_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call setup_pihmc_nvt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NVT ensemble.' &
     &                  // ' <iorder_hmc> must be either 2 or 4.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - PIHMC, NVH ensemble not supported.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - PIHMC, NTH ensemble not supported.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call setup_pihmc_npt_MPI

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        if ( myrank .eq. 0 ) then
                           write( 6, '(a)' ) &
     &                        'Warning - The box is not cubic.' // &
     &                        ' Set <npt_type> = PPHEX.'
                           write( 6, '(a)' )
                        end if

                        call error_handling_MPI &
     &                     ( 1, 'program pimd_MPI', 16 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pihmc_npt_MPI

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        if ( myrank .eq. 0 ) then
                           write( 6, '(a)' ) &
     &                        'Warning - The box is not cubic.' // &
     &                        ' Set <npt_type> = PPHEX.'
                           write( 6, '(a)' )
                        end if

                        call error_handling_MPI &
     &                     ( 1, 'program pimd_MPI', 16 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_npt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NPT ensemble (2nd).' &
     &                  // ' <npt_type> must be CUBIC1, CUBIC2, PPHEX.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call setup_pihmc_npt_MPI

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        if ( myrank .eq. 0 ) then
                           write( 6, '(a)' ) &
     &                        'Warning - The box is not cubic.' // &
     &                        ' Set <npt_type> = PPHEX.'
                           write( 6, '(a)' )
                        end if

                        call error_handling_MPI &
     &                     ( 1, 'program pimd_MPI', 16 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pihmc_npt_MPI

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        if ( myrank .eq. 0 ) then
                           write( 6, '(a)' ) &
     &                        'Warning - The box is not cubic.' // &
     &                        ' Set <npt_type> = PPHEX.'
                           write( 6, '(a)' )
                        end if

                        call error_handling_MPI &
     &                     ( 1, 'program pimd_MPI', 16 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_npt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NPT ensemble (4th).' &
     &                  // ' <npt_type> must be CUBIC1, CUBIC2, PPHEX.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NPT ensemble.' &
     &               // ' <iorder_hmc> must be either 2 or 4.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_ntt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NTT ensemble (2nd).' &
     &                  // ' <ntt_type> must be PPHEX.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_ntt_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - PIHMC, NTT ensemble (4th).' &
     &                  // ' <ntt_type> must be PPHEX.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NTT ensemble.' &
     &               // ' <iorder_hmc> must be either 2 or 4.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - PIHMC must be either NVT, NPT,' &
     &            // ' or NTT ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_cmd_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - CMD must be NVE ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_rpmd_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_rpmd_nvt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - RPMD must be NVE or NVT ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_trpmd_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_trpmd_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - TRPMD must be NVE or NVT ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_pibcmd_MPI

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_pibcmd_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - BCMD must be NVT or NVE ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         call setup_dvr_MPI

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call setup_multi_me_MPI

               else

                  write( 6, '(a)' ) &
     &               'Error - For MTS, set <ipotential> =' // &
     &               ' QMMM and <qmmm_embedding> = ME.'
                  write( 6, '(a)' )

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else

                write( 6, '(a)' ) &
     &             'Error - MTS, <bath_type> must be MNHC.'
                write( 6, '(a)' )

                call error_handling_MPI(1, 'program pimd_MPI', 16 )

             end if

         else

            write( 6, '(a)' ) &
     &         'Error - MTS is supported for NVT ensemble only.'
            write( 6, '(a)' )

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call setup_meta_dual_MPI

            else

               if (  bath_type(1:5) .eq. 'MNHC ' ) then

                  call setup_meta_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - MTD, <bath_type> must be MNHC.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            end if

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - MTD is supported for NVT ensemble only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if (  bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_tass_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - TASS, <bath_type> must be MNHC.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - TASS is supported for NVT ensemble only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call setup_rehmc_MPI

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call setup_rehmc_MPI

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call setup_rehmc_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - REHMC, <irem_type> must be either' &
     &               // ' T, TX or HX for NVT.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if ( npt_type(1:6) .ne. 'PPHEX ' ) then

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - REHMC, NPT ensemble. <npt_type> must be' &
     &               // ' PPHEX.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            else if ( irem_type(1:3) .eq. 'T  ' ) then

               call setup_rehmc_npt_MPI

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call setup_rehmc_npt_MPI

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - REHMC, <irem_type> must be either' &
     &               // ' T or TX for NPT.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI(1, 'program pimd_MPI', 16 )

            end if

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - REHMC is supported for ' // &
     &            'NVT and NPT ensembles only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_remc_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - REMC is supported for ' // &
     &            'NVT ensemble only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         if      ( nbead .eq. 1 ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - String method needs nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or. &
     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or. &
     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or. &
     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then

            call setup_string_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - <ends_string> must be FREE,' // &
     &            ' FIXED, FREEFIXED or FIXEDFREE.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         if      ( nbead .eq. 1 ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - OM method needs nbead > 1.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or. &
     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or. &
     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or. &
     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then

            continue

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - <ends_string> must be FREE,' // &
     &            ' FIXED, FREEFIXED or FIXEDFREE.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

         if ( ( equation_om(1:11) .eq. 'OVERDAMPED '  ) .or. &
     &        ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) ) then

            continue

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - <equation_om> must be' // &
     &            ' OVERDAMPED or UNDERDAMPED.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

         call setup_omopt_MPI

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         if      ( nbead .eq. 1 ) then
!
!            if ( myrank .eq. 0 ) then
!               write( 6, '(a)' )
!     &            'Error - OM method needs nbead > 1.'
!               write( 6, '(a)' )
!            end if
!
!            call error_handling_MPI(1, 'program pimd_MPI', 16 )
!
!         end if
!
!         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or.
!     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or.
!     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or.
!     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then
!
!            continue
!
!         else
!
!            if ( myrank .eq. 0 ) then
!               write( 6, '(a)' )
!     &            'Error - <ends_string> must be FREE,' //
!     &            ' FIXED, FREEFIXED or FIXEDFREE.'
!               write( 6, '(a)' )
!            end if
!
!            call error_handling_MPI(1, 'program pimd_MPI', 16 )
!
!         end if
!
!         if ( ( equation_om(1:11) .eq. 'OVERDAMPED '  ) .or.
!     &        ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) ) then
!
!            continue
!
!         else
!
!            if ( myrank .eq. 0 ) then
!               write( 6, '(a)' )
!     &            'Error - <equation_om> must be' //
!     &            ' OVERDAMPED or UNDERDAMPED.'
!               write( 6, '(a)' )
!            end if
!
!            call error_handling_MPI(1, 'program pimd_MPI', 16 )
!
!         end if
!
!         call setup_om_MPI

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_rotor_nve_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_rotor_nvt_MPI

         else

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - ROTOR is supported for' &
     &            // ' NVE and NVT ensemble only.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         end if

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call setup_tfs_MPI

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call setup_mfe_MPI

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call setup_rptfs_MPI

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call setup_rpmfe_MPI

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

         if      ( ensemble(1:4) .ne. 'NVT ' ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - AFED supported only for NVT ensemble.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else if ( bath_type(1:5) .ne. 'MNHC ' ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - AFED supported only for MNHC thermostat.'
               write( 6, '(a)' )
            end if

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else

            if ( ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &           ( afed_type(1:8) .eq. 'HESSIAN ' ) .or. &
     &           ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &           ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &           ( afed_type(1:7) .eq. 'ASCENT '  ) .or. &
     &           ( afed_type(1:5) .eq. 'AUTO '    ) ) then

               call setup_adescent_afed_MPI

            else if ( afed_type(1:5) .eq. 'TAMD '    ) then

               if ( ( tamd_type(1:4) .eq. 'NVE ' ) .or. &
     &              ( tamd_type(1:4) .eq. 'NVT ' ) .or. &
     &              ( tamd_type(1:4) .eq. 'VS  ' ) ) then

                  call setup_tamd_afed_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - TAMD type is incorrect.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI( 1, 'program pimd_MPI', 16 )

               end if

            else if ( afed_type(1:7) .eq. 'LOGMFD '  ) then

               if ( ( logmfd_type(1:4) .eq. 'NVE ' ) .or. &
     &              ( logmfd_type(1:4) .eq. 'NVT ' ) .or. &
     &              ( logmfd_type(1:4) .eq. 'VS  ' ) ) then

                  call setup_logmfd_afed_MPI

               else

                  if ( myrank .eq. 0 ) then
                     write( 6, '(a)' ) &
     &                  'Error - LOGMFD type is incorrect.'
                     write( 6, '(a)' )
                  end if

                  call error_handling_MPI( 1, 'program pimd_MPI', 16 )

               end if

            else

               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               'Error - AFED type is incorrect.'
                  write( 6, '(a)' )
               end if

               call error_handling_MPI( 1, 'program pimd_MPI', 16 )

            end if

         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call setup_tps_MPI

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call setup_gad_MPI

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call setup_shs_MPI

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         continue

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call setup_pimd_1d_MPI

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call setup_pihmc_1d_MPI

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call setup_pibcmd_1d_MPI

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call setup_rpmd_1d_MPI

!     ==== one dimensional rpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call setup_trpmd_1d_MPI

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call setup_cmd_1d_MPI

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call setup_matsubara_1d_MPI

!     ==== otherwise ====

      else

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Error in <method>.'
            write( 6, '(a)' )
         end if

         call error_handling_MPI(1, 'program pimd_MPI', 16 )

      end if

!-----------------------------------------------------------------------
!     /*   print subtitles                                            */
!-----------------------------------------------------------------------

      call print_subtitles_MPI

!-----------------------------------------------------------------------
!     /*   integrate the equation of motion                           */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         call static_MPI

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         if ( xmpi(1:3) .eq. 'OFF' ) then
            call testforce_MPI
         else
            call testforce_XMPI
         end if

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         if ( xmpi(1:3) .eq. 'OFF' ) then
            call testvirial_MPI
         else
            call testvirial_XMPI
         end if

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         call testewald_MPI

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         call testewpol_MPI

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         call testforce_om_MPI

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         if ( xmpi(1:3) .eq. 'OFF' ) then
            call geooptcycle_MPI
         else
            call geooptcycle_XMPI
         end if

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         call nma_MPI

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         call phonon_MPI

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         call sdcycle_MPI

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         call boxoptcycle_MPI

!     ==== box optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         call fulloptcycle_MPI

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         call elastic_MPI

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call mdcycle_nve_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call mdcycle_nvt_MPI

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if ( xmpi(1:3) .eq. 'OFF' ) then
               call pimdcycle_nve_MPI
            else
               call error_handling_MPI(1, 'program pimd_MPI', 16 )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               if ( xmpi(1:3) .eq. 'OFF' ) then
                  call pimdcycle_nvt_nhc_MPI
               else
                  call error_handling_MPI(1, 'program pimd_MPI', 16 )
               end if

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               if ( xmpi(1:3) .eq. 'OFF' ) then
                  call pimdcycle_nvt_nhcs_MPI
               else
                  call error_handling_MPI(1, 'program pimd_MPI', 16 )
               end if

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( xmpi(1:3) .eq. 'OFF' ) then
                  call pimdcycle_nvt_mnhc_MPI
               else if ( xmpi(1:2) .eq. 'ON' ) then
                  call pimdcycle_nvt_mnhc_XMPI
               else
                  call error_handling_MPI(1, 'program pimd_MPI', 16 )
               end if

            end if

         else if ( xmpi(1:2) .eq. 'ON' ) then

            call error_handling_MPI(1, 'program pimd_MPI', 16 )

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call pimdcycle_nph_c1_MPI

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call pimdcycle_nph_c1_MPI

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_nph_pp_MPI

            end if

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if      ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_nth_pp_MPI

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

               call pimdcycle_npt_c2_MPI

            else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_npt_pp_MPI

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_ntt_pp_MPI

            end if

         else if ( ensemble(1:5) .eq. 'QTST ' ) then

            call pimdcycle_nvt_qtst_MPI

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call pihmccycle_second_dual_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call pihmccycle_fourth_dual_MPI

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call pihmccycle_second_nvt_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call pihmccycle_fourth_nvt_MPI

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_c1_MPI
                  else
                     call pihmccycle_second_npt_c1_MPI
                  end if

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_c2_MPI
                  else
                     call pihmccycle_second_npt_c2_MPI
                  end if

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_pp_MPI
              else
                     call pihmccycle_second_npt_pp_MPI
                  end if

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_c1_MPI
                  else
                     call pihmccycle_fourth_npt_c1_MPI
                  end if

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_c2_MPI
                  else
                     call pihmccycle_fourth_npt_c2_MPI
                  end if

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_pp_MPI
                  else
                     call pihmccycle_fourth_npt_pp_MPI
                  end if

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call pihmccycle_second_ntt_pp_MPI

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call pihmccycle_fourth_ntt_pp_MPI

               end if

            end if

         end if

!     ====  cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if ( xmpi(1:3) .eq. 'OFF' ) then
               call cmdcycle_MPI
            else
               call cmdcycle_XMPI
            end if

         end if

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

!           /*   analytical integration of bead springs   */
            if ( xmpi(1:3) .eq. 'OFF' ) then
               call rpmdcycle_nve_MPI
            else if ( xmpi(1:2) .eq. 'ON' ) then
               call rpmdcycle_nve_XMPI
            else if ( xmpi(1:5) .eq. 'DEBUG' ) then
               call rpmdcycle_nve_debug_XMPI
            else
               call error_handling_MPI(1, 'program pimd_MPI', 16 )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( xmpi(1:3) .eq. 'OFF' ) then
               call rpmdcycle_nvt_MPI
            else
               call error_handling_MPI(1, 'program pimd_MPI', 16 )
            end if

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if ( ensemble(1:4) .eq. 'NVE ' ) then

            call trpmdcycle_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call trpmdcycle_MPI

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

!           /*   pibcmd   */
            if ( xmpi(1:3) .eq. 'OFF' ) then
               call pibcmdcycle_MPI
            else
               call pibcmdcycle_XMPI
            end if

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

!           /*   pibcmd   */
            if ( xmpi(1:3) .eq. 'OFF' ) then
               call pibcmdcycle_MPI
            else
               call pibcmdcycle_XMPI
            end if

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         call dvrcycle_MPI

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call multicycle_me_MPI

               end if

            end if

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call metacycle_dual_MPI

            else

               if ( bath_type(1:5) .eq. 'MNHC ' ) then

                  call metacycle_MPI

               end if

            end if

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call tasscycle_MPI

            end if

         end if

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_t_dual_MPI
               else
                  call rehmccycle_t_MPI
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_tx_dual_MPI
               else
                  call rehmccycle_tx_MPI
               end if

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call rehmccycle_hx_MPI

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_t_dual_npt_MPI
               else
                  call rehmccycle_t_npt_MPI
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_tx_dual_npt_MPI
               else
                  call rehmccycle_tx_npt_MPI
               end if

            end if

         end if

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call remccycle_t_dual_MPI
               else
                  call remccycle_t_MPI
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call remccycle_tx_dual_MPI
               else
                  call remccycle_tx_MPI
               end if

            end if

         end if

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         call stringcycle_MPI

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         call omoptcycle_MPI

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         call omcycle_MPI

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call rotor_nve_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call rotor_nvt_MPI

         end if

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call tfscycle_MPI

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call mfecycle_MPI

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call rptfscycle_MPI

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call rpmfecycle_MPI

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

!        //   free energy gradient with constant constraint
         if      ( afed_type(1:5) .eq. 'GRAD ' ) then

            call afedcycle_grad_MPI

!        //   free energy hessian with constant constraint
         else if ( afed_type(1:8) .eq. 'HESSIAN ' ) then

            call afedcycle_hessian_MPI

!        //   ascent search for free energy saddle points
         else if ( afed_type(1:7) .eq. 'ASCENT ' ) then

             call afedcycle_ascent_MPI

!        //   descent search for free energy minimum points
         else if ( afed_type(1:8) .eq. 'DESCENT ' ) then

            call afedcycle_descent_MPI

!        //   automated search for free energy stationary points
         else if ( afed_type(1:5) .eq. 'AUTO ' ) then

            call afedcycle_auto_MPI

!        //   convergence test
         else if ( afed_type(1:5) .eq. 'TEST ' ) then

            call afedcycle_test_MPI

!        //   temperature accelerated molecular dynamics
         else if ( afed_type(1:5) .eq. 'TAMD ' ) then

            if      ( tamd_type(1:4) .eq. 'NVE ' ) then

               call afedcycle_tamd_nve_MPI

            else if ( tamd_type(1:4) .eq. 'NVT ' ) then

               call afedcycle_tamd_nvt_MPI

            else if ( tamd_type(1:4) .eq. 'VS  ' ) then

               call afedcycle_tamd_vs_MPI

            end if

!        //   logarithmic mean force dynamics
         else if ( afed_type(1:7) .eq. 'LOGMFD ' ) then

            if      ( logmfd_type(1:4) .eq. 'NVE ' ) then

               call afedcycle_logmfd_nve_MPI

            else if ( logmfd_type(1:4) .eq. 'NVT ' ) then

               call afedcycle_logmfd_nvt_MPI

            else if ( logmfd_type(1:4) .eq. 'VS  ' ) then

               call afedcycle_logmfd_vs_MPI

            end if

!        //   none
         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call tpscycle_MPI

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call gadcycle_MPI

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call shscycle_MPI

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         call scancycle_MPI

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call pimdcycle_1d_MPI

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call pihmccycle_1d_MPI

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call pibcmdcycle_1d_MPI

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call rpmdcycle_1d_MPI

!     ==== one dimensional trpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call trpmdcycle_1d_MPI

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call cmdcycle_1d_MPI

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call matsubaracycle_1d_MPI

!     ==== otherwise ====

      else

         call error_handling_MPI(1, 'program pimd_MPI', 16 )

      end if

!-----------------------------------------------------------------------
!     /*  finalize and prepare for the next run                       */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         if ( nbead .eq. 1 ) then

            call backup_geoopt_MPI

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini.'
            write ( 6, '(a)' )
         end if

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )
         continue

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         if ( nbead .eq. 1 ) then

            call backup_geoopt_MPI

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini.'
            write ( 6, '(a)' )
         end if

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         if ( nbead .eq. 1 ) then

            call backup_boxopt_MPI

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini box.ini.'
            write ( 6, '(a)' )
         end if

!     ==== full optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         if ( nbead .eq. 1 ) then

            call backup_boxopt_MPI

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini box.ini.'
            write ( 6, '(a)' )
         end if

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         continue

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_md_nve_MPI

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_md_nvt_MPI

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if ( xmpi(1:3) .eq. 'OFF' ) then
               call backup_pimd_nve_MPI
            else
               call backup_pimd_nve_XMPI
            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               call backup_pimd_nvt_nhc_MPI

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               call backup_pimd_nvt_nhcs_MPI

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( xmpi(1:3) .eq. 'OFF' ) then
                  call backup_pimd_nvt_mnhc_MPI
               else
                  call backup_pimd_nvt_mnhc_XMPI
               endif

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call backup_pimd_nph_MPI

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call backup_pimd_nph_MPI

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_nph_MPI

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_nth_MPI

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

               call backup_pimd_npt_MPI

            else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_npt_MPI

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_ntt_MPI

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:5) .eq. 'QTST ' ) then

            call backup_pimd_nvt_mnhc_MPI

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if     ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call backup_pihmc_second_nvt_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call backup_pihmc_fourth_nvt_MPI

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call backup_pihmc_second_nvt_MPI

               else if ( iorder_hmc .eq. 4 ) then

                  call backup_pihmc_fourth_nvt_MPI

               end if

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call backup_pihmc_second_npt_MPI

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call backup_pihmc_second_npt_MPI

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_second_npt_MPI

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call backup_pihmc_fourth_npt_MPI

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call backup_pihmc_fourth_npt_MPI

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_fourth_npt_MPI

               end if

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_second_ntt_MPI

               else

                  call error_handling_MPI(1, 'program pimd_MPI', 16 )

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_fourth_ntt_MPI

               end if

            end if

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'box.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if( xmpi(1:3) .eq. 'OFF' ) then
               call backup_cmd_MPI
            else
               call backup_cmd_XMPI
            endif

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if( xmpi(1:3) .eq. 'OFF' ) then
               call backup_pimd_nve_MPI
            else
               call backup_pimd_nve_XMPI
            endif

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_cmd_MPI

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_trpmd_MPI

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_trpmd_MPI

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if( xmpi(1:3) .eq. 'OFF' ) then
               call backup_pibcmd_MPI
            else
               call backup_pibcmd_XMPI
            endif

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

            if( xmpi(1:3) .eq. 'OFF' ) then
               call backup_pibcmd_MPI
            else
               call backup_pibcmd_XMPI
            endif

            if ( myrank .eq. 0 ) then
               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &           'Restart files: geometry.ini, step.ini, ' // &
     &           'bath.ini, averages.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         continue

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call backup_multi_me_MPI

                  if ( myrank .eq. 0 ) then
                     write ( 6, '(a)' )
                     write ( 6, '(a)' ) &
     &                  'Restart files: geometry.ini, step.ini, ' // &
     &                  'bath.ini, averages.ini.'
                     write ( 6, '(a)' )
                  end if

               end if

            end if

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call backup_meta_dual_MPI

               if ( myrank .eq. 0 ) then
                  write ( 6, '(a)' )
                  write ( 6, '(a)' ) &
     &               'Restart files: geometry.ini, step.ini, ' // &
     &               'hills.ini, cv.ini.'
                  write ( 6, '(a)' )
                end if

            else

               if ( bath_type(1:5) .eq. 'MNHC ' ) then

                  call backup_meta_MPI

                  if ( myrank .eq. 0 ) then
                     write ( 6, '(a)' )
                     write ( 6, '(a)' ) &
     &                  'Restart files: geometry.ini, step.ini, ' // &
     &                  'bath.ini, hills.ini, cv.ini.'
                     write ( 6, '(a)' )
                  end if

               end if

            end if

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call backup_tass_MPI

            end if

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'bath.ini, hills.ini, cv.ini.'
            write ( 6, '(a)' )
         end if

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call backup_rehmc_MPI

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call backup_rehmc_MPI

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call backup_rehmc_MPI

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call backup_rehmc_npt_MPI

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call backup_rehmc_npt_MPI

            end if

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'averages.ini.'
            write ( 6, '(a)' )
         end if

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         call backup_remc_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'averages.ini.'
           write ( 6, '(a)' )
         end if

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         call backup_string_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini.'
            write ( 6, '(a)' )
         end if

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         call backup_omopt_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: string.ini, geometry.ini, step.ini.'
            write ( 6, '(a)' )
         end if

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         call backup_om_MPI
!
!         if ( myrank .eq. 0 ) then
!            write ( 6, '(a)' )
!            write ( 6, '(a)' )
!     &         'Restart files: string.ini, geometry.ini, ' //
!     &         'step.ini, bath.ini.'
!            write ( 6, '(a)' )
!         end if

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_rotor_nve_MPI

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_rotor_nvt_MPI

         end if

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: rotor.ini, step.ini, ' // &
     &         'averages.ini.'
            write ( 6, '(a)' )
         end if

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call backup_tfs_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'cstate.ini, averages.ini.'
            write ( 6, '(a)' )
         end if

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call backup_mfe_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'cstate.ini.'
            write ( 6, '(a)' )
         end if

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call backup_rptfs_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'cstate.ini, averages.ini.'
            write ( 6, '(a)' )
         end if

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call backup_rpmfe_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'cstate.ini, averages.ini.'
            write ( 6, '(a)' )
         end if

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

         if    ( ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &           ( afed_type(1:8) .eq. 'HESSIAN ' ) .or. &
     &           ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &           ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &           ( afed_type(1:7) .eq. 'ASCENT '  ) ) then

            call backup_adescent_afed_MPI

            if ( myrank .eq. 0 ) then
!              write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &            'Restart files: geometry.ini, afed.ini, '
               write ( 6, '(a)' )
            end if

         else if ( afed_type(1:5) .eq. 'AUTO ' ) then

            call backup_adescent_afed_MPI

            if ( myrank .eq. 0 ) then
!              write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &            'Restart files: geometry.ini, afed.ini, ' // &
     &            'auto.ini.'
               write ( 6, '(a)' )
            end if

         else if ( afed_type(1:5) .eq. 'TAMD ' ) then

            if      ( tamd_type(1:4) .eq. 'NVE ' ) then

               call backup_tamd_nve_afed_MPI

            else if ( tamd_type(1:4) .eq. 'NVT ' ) then

               call backup_tamd_nvt_afed_MPI

            else if ( tamd_type(1:4) .eq. 'VS  ' ) then

               call backup_tamd_vs_afed_MPI

            end if

            if ( myrank .eq. 0 ) then
!              write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &            'Restart files: geometry.ini, afed.ini.'
               write ( 6, '(a)' )
            end if

         else if ( afed_type(1:7) .eq. 'LOGMFD ' ) then

            if      ( logmfd_type(1:4) .eq. 'NVE ' ) then

               call backup_logmfd_nve_afed_MPI

            else if ( logmfd_type(1:4) .eq. 'NVT ' ) then

               call backup_logmfd_nvt_afed_MPI

            else if ( logmfd_type(1:4) .eq. 'VS  ' ) then

               call backup_logmfd_vs_afed_MPI

            end if

            if ( myrank .eq. 0 ) then
!              write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &            'Restart files: geometry.ini, afed.ini.'
               write ( 6, '(a)' )
            end if

         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call backup_tps_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call backup_gad_MPI

         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'gad.ini.'
         write ( 6, '(a)' )

         end if

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call backup_shs_MPI

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call backup_pimd_nvt_mnhc_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call backup_pihmc_second_nvt_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call backup_pibcmd_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call backup_pimd_nve_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional trpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call backup_trpmd_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call backup_cmd_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call backup_pimd_nve_MPI

         if ( myrank .eq. 0 ) write ( 6, '(a)' )

!     ==== otherwise ====

      else

         call error_handling_MPI(1, 'program pimd_MPI', 16 )

      end if

!-----------------------------------------------------------------------
!     /*   finalize external programs                                 */
!-----------------------------------------------------------------------

      call finalize_extprograms_MPI

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) write(6,'(a)') 'Normal termination of pimd.'
      if ( myrank .eq. 0 ) write(6,'(a)')

!-----------------------------------------------------------------------
!     /*  finalize MPI                                                */
!-----------------------------------------------------------------------

      call my_mpi_finalize_2

      stop
      end

