!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Feb 1, 2025 by M. Shiga
!      Description:     main routine of PIMD
!
!///////////////////////////////////////////////////////////////////////
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
!      PIMD version 2.7.1
!
!      Author:        M. Shiga
!
!      Last Updated:  Jan 8, 2026.
!
!      Copyright(C) 2016-2026  M. Shiga  All rights reserved.
!
!      Licensed under the Apache License, Version 2.0 (the "License");
!      you may not use this file except in compliance with the License.
!      You may obtain a copy of the License at
!
!      http://www.apache.org/licenses/LICENSE-2.0
!
!      Unless required by applicable law or agreed to in writing,
!      software distributed under the License is distributed on an
!      "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
!      either express or implied. See the License for the specific
!      language governing permissions and limitations under the License.
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!***********************************************************************
      program pimd
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   method, nph_type, npt_type, ntt_type, nth_type, ensemble, &
     &   ipotential, irem_type, iorder_hmc, bath_type, nbead, box, &
     &   ends_string, equation_om, afed_type, tamd_type, logmfd_type, &
     &   qmmm_embedding, code

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   name of the code                                           */
!-----------------------------------------------------------------------

      code = 'PIMD'

!-----------------------------------------------------------------------
!     /*   print titles                                               */
!-----------------------------------------------------------------------

      call print_titles

!-----------------------------------------------------------------------
!     /*   set parameters                                             */
!-----------------------------------------------------------------------

      call setparams

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      call setallocation

!-----------------------------------------------------------------------
!     /*   prepare condition                                          */
!-----------------------------------------------------------------------

      call setcondition

!-----------------------------------------------------------------------
!     /*   initialize external programs                               */
!-----------------------------------------------------------------------

      call init_extprograms

!-----------------------------------------------------------------------
!     /*  set up the calculation                                      */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         call setup_geometry

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         call setup_testforce

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         call setup_testvirial

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         call setup_geometry

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         call setup_geometry

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         call setup_omopt

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_geoopt

         else

            write( 6, '(a)' ) &
     &         'Error - GEOOPT not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_nma

         else

            write( 6, '(a)' ) &
     &         'Error - NMA not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_geoopt

         else

            write( 6, '(a)' ) &
     &         'Error - PHONON not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_sd

         else

            write( 6, '(a)' ) &
     &         'Error - SD not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_boxopt

         else

            write( 6, '(a)' ) &
     &         'Error - BOXOPT not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== full optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_fullopt

         else

            write( 6, '(a)' ) &
     &         'Error - FULLOPT not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         if      ( nbead .eq. 1 ) then

            call setup_elastic

         else

            write( 6, '(a)' ) &
     &         'Error - ELASTIC not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if      ( bath_type(1:5) .eq. 'NONE ' ) then

               call setup_md_nve

            else

               write( 6, '(a)' ) &
     &            'Error - MD, NVE ensemble. <bath_type> must be' &
     &            // ' NONE.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_md_nvt

            else

               write( 6, '(a)' ) &
     &            'Error - MD, NVT ensemble. <bath_type> must be' &
     &            // ' MNHC.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - MD is supported for NVT and NVE' &
     &         // ' ensembles only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            if      ( nbead .eq. 1 ) then

               call setup_pimd_nve

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NVE ensemble not supported for' &
     &            // ' nbead > 1.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               call setup_pimd_nvt_nhc

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               call setup_pimd_nvt_nhcs

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_pimd_nvt_mnhc

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NVT ensemble. <bath_type> must be' &
     &            // ' either NHC, NHCS or MNHC.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call setup_pimd_nph

               do i = 1,   3
               do j = i+1, 3

                  if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                     write( 6, '(a)' ) &
     &                  'Warning - The box is not cubic.' // &
     &                  ' Set <nph_type> = PPHEX.'
                     write( 6, '(a)' )

                     call error_handling( 1, 'program pimd', 12 )

                  end if

               end do
               end do

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call setup_pimd_nph

               do i = 1,   3
               do j = i+1, 3

                  if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                     write( 6, '(a)' ) &
     &                  'Warning - The box is not cubic.' // &
     &                  ' Set <nph_type> = PPHEX.'
                     write( 6, '(a)' )

                     call error_handling( 1, 'program pimd', 12 )

                  end if

               end do
               end do

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call setup_pimd_nph

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NPH ensemble. <nph_type> must be' &
     &            // ' either CUBIC1, CUBIC2 or PPHEX.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call setup_pimd_nth

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NTH ensemble.' &
     &            // ' <nth_type> must be PPHEX.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pimd_npt

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <nph_type> = PPHEX.'
                        write( 6, '(a)' )

                        call error_handling( 1, 'program pimd', 12 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pimd_npt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIMD, NPT ensemble.' &
     &               // ' <npt_type> must be CUBIC2 or PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NPT ensemble. <bath_type> must be' &
     &            // ' MNHC.'
               write( 6, '(a)' )

               call error_handling(1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pimd_ntt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIMD, NTT ensemble.' &
     &               // ' <ntt_type> must be PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               write( 6, '(a)' ) &
     &            'Error - PIMD, NTT ensemble. <bath_type> must be' &
     &            // ' MNHC.'
               write( 6, '(a)' )

               call error_handling(1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - PIMD must be either NVT, NPH, PTH,' &
     &         // ' NPT or NTT ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            write( 6, '(a)' ) &
     &         'Error - PIHMC, NVE ensemble not supported.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call setup_pihmc_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call setup_pihmc_nvt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NVT ensemble (DUAL).' &
     &               // ' <iorder_hmc> must be either 2 or 4.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call setup_pihmc_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call setup_pihmc_nvt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NVT ensemble.' &
     &               // ' <iorder_hmc> must be either 2 or 4.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            write( 6, '(a)' ) &
     &         'Error - PIHMC, NPH ensemble not supported.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            write( 6, '(a)' ) &
     &         'Error - PIHMC, NTH ensemble not supported.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call setup_pihmc_npt

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <npt_type> = PPHEX.'
                        write( 6, '(a)' )

                        call error_handling( 1, 'program pimd', 12 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pihmc_npt

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <npt_type> = PPHEX.'
                        write( 6, '(a)' )

                        call error_handling( 1, 'program pimd', 12 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_npt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NPT ensemble (2nd).' &
     &               // ' <npt_type> must be CUBIC1, CUBIC2, PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call setup_pihmc_npt

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <npt_type> = PPHEX.'
                        write( 6, '(a)' )

                        call error_handling( 1, 'program pimd', 12 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call setup_pihmc_npt

                  do i = 1,   3
                  do j = i+1, 3

                     if ( abs(box(i,j))+abs(box(j,i)) .gt. 1.d-4 ) then

                        write( 6, '(a)' ) &
     &                     'Warning - The box is not cubic.' // &
     &                     ' Set <npt_type> = PPHEX.'
                        write( 6, '(a)' )

                        call error_handling( 1, 'program pimd', 12 )

                     end if

                  end do
                  end do

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_npt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NPT ensemble (4th).' &
     &               // ' <npt_type> must be CUBIC1, CUBIC2, PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               write( 6, '(a)' ) &
     &            'Error - PIHMC, NPT ensemble.' &
     &            // ' <iorder_hmc> must be either 2 or 4.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_ntt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NTT ensemble (2nd).' &
     &               // ' <ntt_type> must be PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call setup_pihmc_ntt

               else

                  write( 6, '(a)' ) &
     &               'Error - PIHMC, NTT ensemble (4th).' &
     &                // ' <ntt_type> must be PPHEX.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               write( 6, '(a)' ) &
     &            'Error - PIHMC, NTT ensemble.' &
     &            // ' <iorder_hmc> must be either 2 or 4.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - PIHMC must be either NVT, NPT,' &
     &         // ' or NTT ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_cmd

         else

            write( 6, '(a)' ) &
     &         'Error - CMD must be NVE ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_rpmd

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_rpmd_nvt

         else

            write( 6, '(a)' ) &
     &         'Error - RPMD must be NVE or NVT ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_trpmd

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_trpmd

         else

            write( 6, '(a)' ) &
     &         'Error - TRPMD must be NVE or NVT ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_pibcmd

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

            call setup_pibcmd

         else

            write( 6, '(a)' ) &
     &         'Error - BCMD must be NVT or NVE ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         call setup_dvr

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call setup_multi_me

               else

                  write( 6, '(a)' ) &
     &               'Error - For MTS, set <ipotential> =' // &
     &               ' QMMM and <qmmm_embedding> = ME.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

                end if

             else

                write( 6, '(a)' ) &
     &             'Error - MTS, <bath_type> must be MNHC.'
                write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - MTS is supported for NVT ensemble only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call setup_meta_dual

            else

               if ( bath_type(1:5) .eq. 'MNHC ' ) then

                  call setup_meta

               else

                  write( 6, '(a)' ) &
     &               'Error - MTD, <bath_type> must be MNHC.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - MTD is supported for NVT ensemble only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call setup_tass

            else

               write( 6, '(a)' ) &
     &            'Error - TASS, <bath_type> must be MNHC.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - TASS is supported for NVT ensemble only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call setup_rehmc

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call setup_rehmc

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call setup_rehmc

            else

               write( 6, '(a)' ) &
     &            'Error - REHMC, <irem_type> must be either' &
     &            // ' T, TX or HX for NVT.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if ( npt_type(1:6) .ne. 'PPHEX ' ) then

               write( 6, '(a)' ) &
     &            'Error - REHMC, NPT ensemble. <npt_type> must be' &
     &            // ' PPHEX.'
               write( 6, '(a)' )

               call error_handling(1, 'program pimd', 12 )

            else if ( irem_type(1:3) .eq. 'T  ' ) then

               call setup_rehmc_npt

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call setup_rehmc_npt

            else

               write( 6, '(a)' ) &
     &            'Error - REHMC, <irem_type> must be either' &
     &            // ' T or TX for NPT.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - REHMC is supported for ' // &
     &         'NVT and NPT ensembles only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            call setup_remc

         else

            write( 6, '(a)' ) &
     &         'Error - REMC is supported for ' // &
     &         'NVT ensemble only.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         if      ( nbead .eq. 1 ) then

            write( 6, '(a)' ) &
     &         'Error - String method needs nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or. &
     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or. &
     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or. &
     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then

            call setup_string

         else

            write( 6, '(a)' ) &
     &         'Error - <ends_string> must be FREE,' // &
     &         ' FIXED, FREEFIXED or FIXEDFREE.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         if      ( nbead .eq. 1 ) then

            write( 6, '(a)' ) &
     &         'Error - OM method needs nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or. &
     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or. &
     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or. &
     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then

            continue

         else

            write( 6, '(a)' ) &
     &         'Error - <ends_string> must be FREE,' // &
     &         ' FIXED, FREEFIXED or FIXEDFREE.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

         if ( ( equation_om(1:11) .eq. 'OVERDAMPED '  ) .or. &
     &        ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) ) then

            continue

         else

            write( 6, '(a)' ) &
     &         'Error - <equation_om> must be' // &
     &         ' OVERDAMPED or UNDERDAMPED.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

         call setup_omopt

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         if      ( nbead .eq. 1 ) then
!
!            write( 6, '(a)' )
!     &         'Error - OM method needs nbead > 1.'
!            write( 6, '(a)' )
!
!            call error_handling( 1, 'program pimd', 12 )
!
!         end if
!
!         if ( ( ends_string(1:5) .eq. 'FREE '       ) .or.
!     &        ( ends_string(1:6) .eq. 'FIXED '      ) .or.
!     &        ( ends_string(1:10) .eq. 'FREEFIXED ' ) .or.
!     &        ( ends_string(1:10) .eq. 'FIXEDFREE ' ) ) then
!
!            continue
!
!         else
!
!            write( 6, '(a)' )
!     &         'Error - <ends_string> must be FREE,' //
!     &         ' FIXED, FREEFIXED or FIXEDFREE.'
!            write( 6, '(a)' )
!
!            call error_handling( 1, 'program pimd', 12 )
!
!         end if
!
!         if ( ( equation_om(1:11) .eq. 'OVERDAMPED '  ) .or.
!     &        ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) ) then
!
!            continue
!
!         else
!
!            write( 6, '(a)' )
!     &         'Error - <equation_om> must be' //
!     &         ' OVERDAMPED or UNDERDAMPED.'
!            write( 6, '(a)' )
!
!            call error_handling( 1, 'program pimd', 12 )
!
!         end if
!
!         call setup_om

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( nbead .eq. 1 ) then

            if      ( ensemble(1:4) .eq. 'NVE ' ) then

               call setup_rotor_nve

            else if ( ensemble(1:4) .eq. 'NVT ' ) then

               call setup_rotor_nvt

            else

               write( 6, '(a)' ) &
     &            'Error - ROTOR is supported for' &
     &            // ' NVE and NVT ensemble only.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         else

            write( 6, '(a)' ) &
     &         'Error - ROTOR not supported for nbead > 1.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         end if

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call setup_tfs

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call setup_mfe

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call setup_rptfs

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call setup_rpmfe

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

         if      ( ensemble(1:4) .ne. 'NVT ' ) then

            write( 6, '(a)' ) &
     &         'Error - AFED supported only for NVT ensemble.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         else if ( bath_type(1:5) .ne. 'MNHC ' ) then

            write( 6, '(a)' ) &
     &         'Error - AFED supported only for MNHC thermostat.'
            write( 6, '(a)' )

            call error_handling( 1, 'program pimd', 12 )

         else

            if ( ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &           ( afed_type(1:8) .eq. 'HESSIAN ' ) .or. &
     &           ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &           ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &           ( afed_type(1:7) .eq. 'ASCENT '  ) .or. &
     &           ( afed_type(1:5) .eq. 'AUTO '    ) ) then

               call setup_adescent_afed

            else if ( afed_type(1:5) .eq. 'TAMD '    ) then

               if ( ( tamd_type(1:4) .eq. 'NVE ' ) .or. &
     &              ( tamd_type(1:4) .eq. 'NVT ' ) .or. &
     &              ( tamd_type(1:4) .eq. 'VS  ' ) ) then

                  call setup_tamd_afed

               else

                  write( 6, '(a)' ) &
     &               'Error - TAMD type is incorrect.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else if ( afed_type(1:7) .eq. 'LOGMFD '  ) then

               if ( ( logmfd_type(1:4) .eq. 'NVE ' ) .or. &
     &              ( logmfd_type(1:4) .eq. 'NVT ' ) .or. &
     &              ( logmfd_type(1:4) .eq. 'VS  ' ) ) then

                  call setup_logmfd_afed

               else

                  write( 6, '(a)' ) &
     &               'Error - LOGMFD type is incorrect.'
                  write( 6, '(a)' )

                  call error_handling( 1, 'program pimd', 12 )

               end if

            else

               write( 6, '(a)' ) &
     &            'Error - AFED type is incorrect.'
               write( 6, '(a)' )

               call error_handling( 1, 'program pimd', 12 )

            end if

         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call setup_tps

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call setup_gad

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call setup_shs

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         continue

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call setup_pimd_1d

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call setup_pihmc_1d

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call setup_pibcmd_1d

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call setup_rpmd_1d

!     ==== one dimensional trpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call setup_trpmd_1d

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call setup_cmd_1d

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call setup_matsubara_1d

!     ==== otherwise ====

      else

         write( 6, '(a)' ) 'Error in <method>.'
         write( 6, '(a)' )

         call error_handling( 1, 'program pimd', 12 )

      end if

!-----------------------------------------------------------------------
!     /*   print subtitles                                            */
!-----------------------------------------------------------------------

      call print_subtitles

!-----------------------------------------------------------------------
!     /*   integrate the equation of motion                           */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         call static

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         call testforce

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         call testvirial

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         call testewald

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         call testewpol

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         call testforce_om

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         call geooptcycle

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         call nma

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         call phonon

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         call sdcycle

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         call boxoptcycle

!     ==== full optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         call fulloptcycle

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         call elastic

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call mdcycle_nve

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call mdcycle_nvt

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call pimdcycle_nve

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               call pimdcycle_nvt_nhc

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               call pimdcycle_nvt_nhcs

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call pimdcycle_nvt_mnhc

            end if

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call pimdcycle_nph_c1

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call pimdcycle_nph_c2

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_nph_pp

            end if

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if      ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_nth_pp

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

               call pimdcycle_npt_c2

            else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_npt_pp

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

               call pimdcycle_ntt_pp

            end if

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call error_handling( 1, 'program pimd', 12 )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call pihmccycle_second_dual_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call pihmccycle_fourth_dual_nvt

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call pihmccycle_second_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call pihmccycle_fourth_nvt

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_c1
                  else
                     call pihmccycle_second_npt_c1
                  end if

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_c2
                  else
                     call pihmccycle_second_npt_c2
                  end if

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_second_dual_npt_pp
                  else
                     call pihmccycle_second_npt_pp
                  end if

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_c1
                  else
                     call pihmccycle_fourth_npt_c1
                  end if

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_c2
                  else
                     call pihmccycle_fourth_npt_c2
                  end if

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                     call pihmccycle_fourth_dual_npt_pp
                  else
                     call pihmccycle_fourth_npt_pp
                  end if

               end if

            end if

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call pihmccycle_second_ntt_pp

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call pihmccycle_fourth_ntt_pp

               end if

            end if

         end if

!     ====  cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call cmdcycle

         end if

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

!           /*   analytical integration of bead springs   */
            call rpmdcycle_nve

         else if  ( ensemble(1:4) .eq. 'NVT ' ) then

            call rpmdcycle_nvt

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if  ( ensemble(1:4) .eq. 'NVT ' ) then

            call trpmdcycle

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

            call trpmdcycle

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

!           /*   pibcmd   */
            call pibcmdcycle
!cc            call pibcmdcycle_gillespie

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

!           /*   pibcmd   */
            call pibcmdcycle
!cc            call pibcmdcycle_gillespie

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         call dvrcycle

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call multicycle_me

               end if

            end if

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call metacycle_dual

            else

               if ( bath_type(1:5) .eq. 'MNHC ' ) then

                  call metacycle

               end if

            end if

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call tasscycle

            end if

         end if

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_t_dual
               else
                  call rehmccycle_t
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_tx_dual
               else
                  call rehmccycle_tx
               end if

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call rehmccycle_hx

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_t_dual_npt
               else
                  call rehmccycle_t_npt
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call rehmccycle_tx_dual_npt
               else
                  call rehmccycle_tx_npt
               end if

            end if

         end if

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call remccycle_t_dual
               else
                  call remccycle_t
               end if

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               if      ( ipotential(1:5) .eq. 'DUAL ' ) then
                  call remccycle_tx_dual
               else
                  call remccycle_tx
               end if

            end if

         end if

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         call stringcycle

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         call omoptcycle

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         call omcycle

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call rotor_nve

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call rotor_nvt

         end if

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call tfscycle

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call mfecycle

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call rptfscycle

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call rpmfecycle

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

!        //   free energy gradient with constant constraint
         if      ( afed_type(1:5) .eq. 'GRAD ' ) then

            call afedcycle_grad

!        //   free energy hessian with constant constraint
         else if ( afed_type(1:8) .eq. 'HESSIAN ' ) then

            call afedcycle_hessian

!        //   ascent search for free energy saddle points
         else if ( afed_type(1:7) .eq. 'ASCENT ' ) then

             call afedcycle_ascent

!        //   descent search for free energy minimum points
         else if ( afed_type(1:8) .eq. 'DESCENT ' ) then

            call afedcycle_descent

!        //   automated search for free energy stationary points
         else if ( afed_type(1:5) .eq. 'AUTO ' ) then

            call afedcycle_auto

!        //   convergence test
         else if ( afed_type(1:5) .eq. 'TEST ' ) then

            call afedcycle_test

!        //   temperature accelerated molecular dynamics
         else if ( afed_type(1:5) .eq. 'TAMD ' ) then

            if      ( tamd_type(1:4) .eq. 'NVE ' ) then

               call afedcycle_tamd_nve

            else if ( tamd_type(1:4) .eq. 'NVT ' ) then

               call afedcycle_tamd_nvt

            else if ( tamd_type(1:4) .eq. 'VS  ' ) then

               call afedcycle_tamd_vs

            end if

!        //   logarithmic mean force dynamics
         else if ( afed_type(1:7) .eq. 'LOGMFD ' ) then

            if      ( logmfd_type(1:4) .eq. 'NVE ' ) then

               call afedcycle_logmfd_nve

            else if ( logmfd_type(1:4) .eq. 'NVT ' ) then

               call afedcycle_logmfd_nvt

            else if ( logmfd_type(1:4) .eq. 'VS  ' ) then

               call afedcycle_logmfd_vs

            end if

!        //   none
         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call tpscycle

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call gadcycle

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call shscycle

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         call scancycle

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call pimdcycle_1d

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call pihmccycle_1d

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call pibcmdcycle_1d

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call rpmdcycle_1d

!     ==== one dimensional trpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call trpmdcycle_1d

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call cmdcycle_1d

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call matsubaracycle_1d

!     ==== otherwise ====

      else

         call error_handling( 1, 'program pimd', 12 )

      end if

!-----------------------------------------------------------------------
!     /*  finalize and prepare for the next run                       */
!-----------------------------------------------------------------------

!     ==== static ====

      if      ( method(1:7) .eq. 'STATIC ' ) then

         write ( 6, '(a)' )
         continue

!     ==== test forces ====

      else if ( method(1:10) .eq. 'TESTFORCE ' ) then

         write ( 6, '(a)' )
         continue

!     ==== test virial ====

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         write ( 6, '(a)' )
         continue

!     ==== test ewald sum ====

      else if ( method(1:10) .eq. 'TESTEWALD ' ) then

         write ( 6, '(a)' )
         continue

!     ==== test ewald dipole sum ====

      else if ( method(1:10) .eq. 'TESTEWPOL ' ) then

         write ( 6, '(a)' )
         continue

!     ==== test OM forces ====

      else if ( method(1:7) .eq. 'TESTOM ' ) then

         write ( 6, '(a)' )
         continue

!     ==== geometry optimization ====

      else if ( method(1:7) .eq. 'GEOOPT ' ) then

         if ( nbead .eq. 1 ) then

            call backup_geoopt

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini.'
         write ( 6, '(a)' )

!     ==== normal mode analysis ====

      else if ( method(1:4) .eq. 'NMA ' ) then

         if      ( nbead .eq. 1 ) then

            write ( 6, '(a)' )
            continue

         end if

!     ==== phonon calculation ====

      else if ( method(1:7) .eq. 'PHONON ' ) then

         if      ( nbead .eq. 1 ) then

            write ( 6, '(a)' )
            continue

         end if

!     ==== steepest decent ====

      else if ( method(1:3) .eq. 'SD ' ) then

         if      ( nbead .eq. 1 ) then

            call backup_geoopt

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini.'
         write ( 6, '(a)' )

!     ==== box optimization ====

      else if ( method(1:7) .eq. 'BOXOPT ' ) then

         if      ( nbead .eq. 1 ) then

               call backup_boxopt

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, box.ini.'
         write ( 6, '(a)' )

!     ==== full optimization ====

      else if ( method(1:8) .eq. 'FULLOPT ' ) then

         if      ( nbead .eq. 1 ) then

            call backup_boxopt

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, box.ini.'
         write ( 6, '(a)' )

!     ==== elastic constants ====

      else if ( method(1:8) .eq. 'ELASTIC ' ) then

         continue

!     ==== classical md ====

      else if ( method(1:3) .eq. 'MD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_md_nve

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_md_nvt

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== pimd ====

      else if ( method(1:5) .eq. 'PIMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_pimd_nve

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'NHC  ' ) then

               call backup_pimd_nvt_nhc

            else if ( bath_type(1:5) .eq. 'NHCS ' ) then

               call backup_pimd_nvt_nhcs

            else if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call backup_pimd_nvt_mnhc

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NPH ' ) then

            if      ( nph_type(1:7) .eq. 'CUBIC1 ' ) then

               call backup_pimd_nph

            else if ( nph_type(1:7) .eq. 'CUBIC2 ' ) then

               call backup_pimd_nph

            else if ( nph_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_nph

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'box.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NTH ' ) then

            if ( nth_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_nth

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'box.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

               call backup_pimd_npt

            else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_npt

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'box.ini, bath.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

               call backup_pimd_ntt

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'box.ini, bath.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== pihmc ====

      else if ( method(1:6) .eq. 'PIHMC ' ) then

         if     ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               if ( iorder_hmc .eq. 2 ) then

                  call backup_pihmc_second_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call backup_pihmc_fourth_nvt

               end if

            else

               if      ( iorder_hmc .eq. 2 ) then

                  call backup_pihmc_second_nvt

               else if ( iorder_hmc .eq. 4 ) then

                  call backup_pihmc_fourth_nvt

               end if

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call backup_pihmc_second_npt

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call backup_pihmc_second_npt

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_second_npt

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if      ( npt_type(1:7) .eq. 'CUBIC1 ' ) then

                  call backup_pihmc_fourth_npt

               else if ( npt_type(1:7) .eq. 'CUBIC2 ' ) then

                  call backup_pihmc_fourth_npt

               else if ( npt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_fourth_npt

               end if

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'box.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NTT ' ) then

            if      ( iorder_hmc .eq. 2 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_second_ntt

               end if

            else if ( iorder_hmc .eq. 4 ) then

               if ( ntt_type(1:7) .eq. 'PPHEX  ' ) then

                  call backup_pihmc_fourth_ntt

               end if

            end if

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'box.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== cmd ====

      else if ( method(1:4) .eq. 'CMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_cmd

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'bath.ini, averages.ini.'
         write ( 6, '(a)' )

!     ==== rpmd ====

      else if ( method(1:5) .eq. 'RPMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_pimd_nve

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, step.ini, ' // &
     &         'averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_cmd

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== trpmd ====

      else if ( method(1:6) .eq. 'TRPMD ' ) then

         if ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_trpmd

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_trpmd

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== pibcmd ====

      else if ( method(1:5) .eq. 'BCMD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_pibcmd

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         else if ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_pibcmd

            write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &        'Restart files: geometry.ini, step.ini, ' // &
     &        'bath.ini, averages.ini.'
            write ( 6, '(a)' )

         end if

!     ==== dvr ====

      else if ( method(1:4) .eq. 'DVR ' ) then

         continue

!     ==== multiple time scale pimd ====

      else if ( method(1:4) .eq. 'MTS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( bath_type(1:5) .eq. 'MNHC ' ) then

               if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &              ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

                  call backup_multi_me

                  write ( 6, '(a)' )
                  write ( 6, '(a)' ) &
     &               'Restart files: geometry.ini, step.ini, ' // &
     &               'bath.ini, averages.ini.'
                  write ( 6, '(a)' )

               end if

            end if

         end if

!     ==== metadynamics ====

      else if ( method(1:4) .eq. 'MTD ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( ipotential(1:5) .eq. 'DUAL ' ) then

               call backup_meta_dual

               write ( 6, '(a)' )
               write ( 6, '(a)' ) &
     &            'Restart files: geometry.ini, step.ini, ' // &
     &            'hills.ini, cv.ini.'
               write ( 6, '(a)' )

            else

               if ( bath_type(1:5) .eq. 'MNHC ' ) then

                  call backup_meta

                  write ( 6, '(a)' )
                  write ( 6, '(a)' ) &
     &               'Restart files: geometry.ini, step.ini, ' // &
     &               'bath.ini, hills.ini, cv.ini.'
                  write ( 6, '(a)' )

               end if

            end if

         end if

!     ==== temperature accelerated sliced sampling ====

      else if ( method(1:5) .eq. 'TASS ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if ( bath_type(1:5) .eq. 'MNHC ' ) then

               call backup_tass

            end if

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'bath.ini, hills.ini, cv.ini.'
         write ( 6, '(a)' )

!     ==== rehmc ====

      else if ( method(1:6) .eq. 'REHMC ' ) then

         if      ( ensemble(1:4) .eq. 'NVT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call backup_rehmc

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call backup_rehmc

            else if ( irem_type(1:3) .eq. 'HX ' ) then

               call backup_rehmc

            end if

         else if ( ensemble(1:4) .eq. 'NPT ' ) then

            if      ( irem_type(1:3) .eq. 'T  ' ) then

               call backup_rehmc_npt

            else if ( irem_type(1:3) .eq. 'TX ' ) then

               call backup_rehmc_npt

            end if

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'averages.ini.'
         write ( 6, '(a)' )

!     ==== remc ====

      else if ( method(1:5) .eq. 'REMC ' ) then

         call backup_remc

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'averages.ini.'
         write ( 6, '(a)' )

!     ==== string method ====

      else if ( method(1:7) .eq. 'STRING ' ) then

         call backup_string

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini.'
         write ( 6, '(a)' )

!     ==== optimization of OM action ====

      else if ( method(1:6) .eq. 'OMOPT ' ) then

         call backup_omopt

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: string.ini, geometry.ini, step.ini.'
         write ( 6, '(a)' )

!     ==== OM dynamics ====
!
!      else if ( method(1:3) .eq. 'OM ' ) then
!
!         call backup_om
!
!         write ( 6, '(a)' )
!         write ( 6, '(a)' )
!     &      'Restart files: string.ini, geometry.ini, ' //
!     &      'step.ini, bath.ini.'
!         write ( 6, '(a)' )

!     ==== rigid rotor ====

      else if ( method(1:6) .eq. 'ROTOR ' ) then

         if      ( ensemble(1:4) .eq. 'NVE ' ) then

            call backup_rotor_nve

         else if ( ensemble(1:4) .eq. 'NVT ' ) then

            call backup_rotor_nvt

         end if

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: step.ini, rotor.ini, ' // &
     &      'averages.ini.'
         write ( 6, '(a)' )

!     ==== tully's fewest switch ====

      else if ( method(1:4) .eq. 'TFS ' ) then

         call backup_tfs

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'cstate.ini, averages.ini.'
         write ( 6, '(a)' )

!     ==== mean field ehrenfest ====

      else if ( method(1:4) .eq. 'MFE ' ) then

         call backup_mfe

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'cstate.ini, averages.ini.'
         write ( 6, '(a)' )

!     ==== ring polymer surface hopping ====

      else if ( method(1:6) .eq. 'RPTFS ' ) then

         call backup_rptfs

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'cstate.ini, averages.ini.'
         write ( 6, '(a)' )

!     ==== ring polymer ehrenfest mean field ====

      else if ( method(1:6) .eq. 'RPMFE ' ) then

         call backup_rpmfe

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'cstate.ini, averages.ini.'
         write ( 6, '(a)' )

!     ==== adiabatic free energy dynamics ====

      else if ( method(1:5) .eq. 'AFED ' ) then

         if    ( ( afed_type(1:5) .eq. 'GRAD '    ) .or. &
     &           ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &           ( afed_type(1:5) .eq. 'TEST '    ) .or. &
     &           ( afed_type(1:8) .eq. 'DESCENT ' ) .or. &
     &           ( afed_type(1:7) .eq. 'ASCENT '  ) ) then

            call backup_adescent_afed

!           write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, afed.ini'
            write ( 6, '(a)' )

         else if ( afed_type(1:5) .eq. 'AUTO ' ) then

            call backup_adescent_afed

!           write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, afed.ini, ' // &
     &         'auto.ini.'
            write ( 6, '(a)' )

         else if ( afed_type(1:5) .eq. 'TAMD ' ) then

            if      ( tamd_type(1:4) .eq. 'NVE ' ) then

               call backup_tamd_nve_afed

            else if ( tamd_type(1:4) .eq. 'NVT ' ) then

               call backup_tamd_nvt_afed

            else if ( tamd_type(1:4) .eq. 'VS  ' ) then

               call backup_tamd_vs_afed

            end if

!           write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, afed.ini.'
            write ( 6, '(a)' )

         else if ( afed_type(1:7) .eq. 'LOGMFD ' ) then

            if      ( logmfd_type(1:4) .eq. 'NVE ' ) then

               call backup_logmfd_nve_afed

            else if ( logmfd_type(1:4) .eq. 'NVT ' ) then

               call backup_logmfd_nvt_afed

            else if ( logmfd_type(1:4) .eq. 'VS  ' ) then

               call backup_logmfd_vs_afed

            end if

!           write ( 6, '(a)' )
            write ( 6, '(a)' ) &
     &         'Restart files: geometry.ini, afed.ini.'
            write ( 6, '(a)' )

         end if

!     ==== transition path sampling ====

      else if ( method(1:4) .eq. 'TPS ' ) then

         call backup_tps

         write ( 6, '(a)' )

!     ==== gad ====

      else if ( method(1:4) .eq. 'GAD ' ) then

         call backup_gad

         write ( 6, '(a)' )
         write ( 6, '(a)' ) &
     &      'Restart files: geometry.ini, step.ini, ' // &
     &      'gad.ini.'
         write ( 6, '(a)' )

!     ==== shs ====

      else if ( method(1:4) .eq. 'SHS ' ) then

         call backup_shs

!     ==== scan ====

      else if ( method(1:5) .eq. 'SCAN ' ) then

         write ( 6, '(a)' )

!     ==== one dimensional pimd ====

      else if ( method(1:8) .eq. 'PIMD-1D ' ) then

         call backup_pimd_nvt_mnhc

         write ( 6, '(a)' )

!     ==== one dimensional pihmc ====

      else if ( method(1:7) .eq. 'HMC-1D ' ) then

         call backup_pihmc_second_nvt

         write ( 6, '(a)' )

!     ==== one dimensional bcmd ====

      else if ( method(1:8) .eq. 'BCMD-1D ' ) then

         call backup_pibcmd

         write ( 6, '(a)' )

!     ==== one dimensional rpmd ====

      else if ( method(1:8) .eq. 'RPMD-1D ' ) then

         call backup_pimd_nve

         write ( 6, '(a)' )

!     ==== one dimensional trpmd ====

      else if ( method(1:9) .eq. 'TRPMD-1D ' ) then

         call backup_trpmd

         write ( 6, '(a)' )

!     ==== one dimensional cmd ====

      else if ( method(1:7) .eq. 'CMD-1D ' ) then

         call backup_cmd

         write ( 6, '(a)' )

!     ==== one dimensional matsubara dynamics ====

      else if ( method(1:13) .eq. 'MATSUBARA-1D ' ) then

         call backup_pimd_nve

         write ( 6, '(a)' )

!     ==== otherwise ====

      else

         call error_handling( 1, 'program pimd', 12 )

      end if

!-----------------------------------------------------------------------
!     /*   finalize external programs                                 */
!-----------------------------------------------------------------------

      call finalize_extprograms

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      write(6,'(a)') 'Normal termination of pimd.'
      write(6,'(a)')

      stop
      end

