!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 18, 2020 by M. Shiga
!      Description:     path integral hybrid Monte Carlo (dual level)
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine pihmccycle_second_dual_nvt
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc

      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_second_hmc_nvt

!     /*   save data   */
      call save_second_hmc_nvt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_second_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   restore low level forces  */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   update all the normal mode coordinates   */
            call update_posvel_anal

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_second_hmc_nvt

!        /*   judge accept or reject   */
         call judge_second_hmc_nvt

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_second_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_second_nvt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_fourth_dual_nvt
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc

      use hmc_variables, only : &
     &   jstep

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_fourth_hmc_nvt

!     /*   save data   */
      call save_fourth_hmc_nvt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_fourth_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   restore low level forces  */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   update all the normal mode coordinates   */
            call update_posvel_anal

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_fourth_hmc_nvt

!        /*   judge accept or reject   */
         call judge_fourth_hmc_nvt

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_fourth_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_fourth_nvt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_second_dual_npt_c1
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_second_hmc_npt_c1

!     /*   save data   */
      call save_second_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_second_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_c1

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_c1

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c1

!              /*   update centroid position   */
               call update_pos_cent_nph_c1

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c1

!              /*   update volume   */
               call update_pos_box_nph_c1

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_c1

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_second_hmc_npt_c1

!        /*   judge accept or reject   */
         call judge_second_hmc_npt_c1

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_second_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_second_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_second_dual_npt_c2
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_second_hmc_npt_c2

!     /*   save data   */
      call save_second_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_second_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_c2

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_c2

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c2

!              /*   update centroid position   */
               call update_pos_cent_nph_c2

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c2

!              /*   update volume   */
               call update_pos_box_nph_c2

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_c2

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_second_hmc_npt_c2

!        /*   judge accept or reject   */
         call judge_second_hmc_npt_c2

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_second_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_second_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_second_dual_npt_pp
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_second_hmc_npt_pp

!     /*   save data   */
      call save_second_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_second_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_pp

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_pp

!              /*   update centroid velocities   */
               call update_vel_cent_nph_pp

!              /*   update centroid position   */
               call update_pos_cent_nph_pp

!              /*   update centroid velocities   */
               call update_vel_cent_nph_pp

!              /*   update volume   */
               call update_pos_box_nph_pp

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_pp

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_second_hmc_npt_pp

!        /*   judge accept or reject   */
         call judge_second_hmc_npt_pp

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_second_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_second_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_fourth_dual_npt_c1
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_fourth_hmc_npt_c1

!     /*   save data   */
      call save_fourth_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_fourth_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_c1

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_c1

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c1

!              /*   update centroid position   */
               call update_pos_cent_nph_c1

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c1

!              /*   update volume   */
               call update_pos_box_nph_c1

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_c1

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_fourth_hmc_npt_c1

!        /*   judge accept or reject   */
         call judge_fourth_hmc_npt_c1

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_fourth_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_fourth_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_fourth_dual_npt_c2
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_fourth_hmc_npt_c2

!     /*   save data   */
      call save_fourth_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_fourth_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_c2

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_c2

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c2

!              /*   update centroid position   */
               call update_pos_cent_nph_c2

!              /*   update centroid velocities   */
               call update_vel_cent_nph_c2

!              /*   update volume   */
               call update_pos_box_nph_c2

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_c2

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_fourth_hmc_npt_c2

!        /*   judge accept or reject   */
         call judge_fourth_hmc_npt_c2

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_fourth_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_fourth_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pihmccycle_fourth_dual_npt_pp
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc, &
     &   iref, nref

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce_dual_lo

!     /*   get interatomic forces   */
      call getforce_dual_hi

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_fourth_hmc_npt_pp

!     /*   save data   */
      call save_fourth_hmc_npt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_fourth_hmc( 1 )
      call analysis_slhmc

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   get interatomic forces   */
         call getforce_dual_lo

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        /*   cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update volume velocity   */
            call update_vel_box_nph_pp

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update non-centroid position and velocity  */
            call update_mode_nph

            do iref = 1, nref

!              /*   update volume   */
               call update_pos_box_nph_pp

!              /*   update centroid velocities   */
               call update_vel_cent_nph_pp

!              /*   update centroid position   */
               call update_pos_cent_nph_pp

!              /*   update centroid velocities   */
               call update_vel_cent_nph_pp

!              /*   update volume   */
               call update_pos_box_nph_pp

            end do

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce_dual_lo

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update centroid and non-centroid velocities   */
            call update_vel

!           /*   update volume velocity   */
            call update_vel_box_nph_pp

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi

!        /*   energy   */
         call getenergy_fourth_hmc_npt_pp

!        /*   judge accept or reject   */
         call judge_fourth_hmc_npt_pp

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_fourth_hmc( 2 )
         call analysis_slhmc

!        /*   output restart   */
         call backup_pihmc_fourth_npt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine getforce_dual_lo
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use dual_variables, only : idual_hi, idual_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initial setup starts   */
      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag   */
         call search_tag ( '<dual_potential>', 16, iounit, ierr )

!        /*   dual:  high and low levels   */
         read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling &
     &      ( ierr, 'subroutine getforce_dual_lo', 27 )

!        /*   setup complete   */
         iset = 1

!     /*   initial setup ends   */
      end if

!-----------------------------------------------------------------------
!     /*   choose between new or old routine                          */
!-----------------------------------------------------------------------

!     /*   same potentials   */
      if ( idual_hi(1:len_trim(idual_hi)) .eq. &
     &     idual_lo(1:len_trim(idual_lo)) ) then

!        /*   use old routine   */
         call getforce_dual_lo_old

!     /*   different potentials   */
      else

!        /*   use new routine   */
         call getforce_dual_lo_new

!     /*   end of if statement   */
      end if

      return
      end





!***********************************************************************
      subroutine getforce_dual_lo_new
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir, potential, ipotential

      use dual_variables, only : &
     &   pot_low, potential_low, fx_low, fy_low, fz_low, &
     &   vir_low, dipx_low, dipy_low, dipz_low, idual_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=80) :: ipotential_save

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual

!-----------------------------------------------------------------------
!     /*   force calculation                                          */
!-----------------------------------------------------------------------

      ipotential_save = ipotential

      ipotential = idual_lo

      call getforce

      ipotential = ipotential_save

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_low(:)     = pot(:)

      potential_low  = potential

      fx_low(:,:)    = fx(:,:)
      fy_low(:,:)    = fy(:,:)
      fz_low(:,:)    = fz(:,:)

      vir_low(:,:)   = vir(:,:)

      dipx_low(:)    = dipx(:)
      dipy_low(:)    = dipy(:)
      dipz_low(:)    = dipz(:)

      return
      end





!***********************************************************************
      subroutine getforce_dual_lo_old
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir, ux, uy, uz, x, y, z, &
     &   potential, natom, nbead, mbox, iounit, pimd_command

      use dual_variables, only : &
     &   x_lo, y_lo, z_lo, e_lo, fx_lo, fy_lo, fz_lo, dipx_lo, dipy_lo, &
     &   dipz_lo, pot_low, potential_low, fx_low, fy_low, fz_low, &
     &   vir_low, dipx_low, dipy_low, dipz_low, scr_dir_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp, xi, yi, zi

      character(len=80) :: char_file
      character(len=80) :: char_dir
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do j = 1, nbead

         call int3_to_char( j, char_num )

!-----------------------------------------------------------------------
!        /*   x_lo, y_lo, z_lo                                        */
!-----------------------------------------------------------------------

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            x_lo(i) = xi
            y_lo(i) = yi
            z_lo(i) = zi

         end do

!-----------------------------------------------------------------------
!        /*   make geometry.ini                                       */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_lo) // '/' // char_num // &
     &               '/geometry.ini'

         call iogeometry( 1, char_file, len(char_file), iounit, &
     &                       x_lo, y_lo, z_lo, natom, 1 )

!-----------------------------------------------------------------------
!        /*   execute at subdirectories                               */
!-----------------------------------------------------------------------

         char_dir = trim(scr_dir_lo) // '/' // char_num

         call system ('cd ' // char_dir // '; ' // trim(pimd_command) &
     &                 // ' > ./monitor.out; cd ../../')

!-----------------------------------------------------------------------
!        /*   read forces.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_lo) // '/' // char_num // &
     &               '/forces.out'

         call ioforce ( 2, char_file, len(char_file), iounit, &
     &                  e_lo, fx_lo, fy_lo, fz_lo, natom, 1 )

!-----------------------------------------------------------------------
!        /*   read dipole.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_lo) // '/' // char_num // &
     &               '/dipole.out'

         call iodipole( 2, char_file, len(char_file), iounit, &
     &                  dipx_lo, dipy_lo, dipz_lo, 1 )

!-----------------------------------------------------------------------
!        /*    sum up energies                                        */
!-----------------------------------------------------------------------

         pot(j) = e_lo(1)

!-----------------------------------------------------------------------
!        /*    sum up forces                                          */
!-----------------------------------------------------------------------

         do i = 1, natom

            fx(i,j) = fx_lo(i)
            fy(i,j) = fy_lo(i)
            fz(i,j) = fz_lo(i)

         end do

!-----------------------------------------------------------------------
!        /*   dipole                                                  */
!-----------------------------------------------------------------------

         dipx(j) = dipx_lo(1)
         dipy(j) = dipy_lo(1)
         dipz(j) = dipz_lo(1)

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         vir(1,1) = vir(1,1) + fx(i,j) * ( ux(i,1) - x(i,j) )
         vir(1,2) = vir(1,2) + fx(i,j) * ( uy(i,1) - y(i,j) )
         vir(1,3) = vir(1,3) + fx(i,j) * ( uz(i,1) - z(i,j) )
         vir(2,1) = vir(2,1) + fy(i,j) * ( ux(i,1) - x(i,j) )
         vir(2,2) = vir(2,2) + fy(i,j) * ( uy(i,1) - y(i,j) )
         vir(2,3) = vir(2,3) + fy(i,j) * ( uz(i,1) - z(i,j) )
         vir(3,1) = vir(3,1) + fz(i,j) * ( ux(i,1) - x(i,j) )
         vir(3,2) = vir(3,2) + fz(i,j) * ( uy(i,1) - y(i,j) )
         vir(3,3) = vir(3,3) + fz(i,j) * ( uz(i,1) - z(i,j) )
      end do
      end do

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_low(:)     = pot(:)

      potential_low  = potential

      fx_low(:,:)    = fx(:,:)
      fy_low(:,:)    = fy(:,:)
      fz_low(:,:)    = fz(:,:)

      vir_low(:,:)   = vir(:,:)

      dipx_low(:)    = dipx(:)
      dipy_low(:)    = dipy(:)
      dipz_low(:)    = dipz(:)

      return
      end





!***********************************************************************
      subroutine getforce_dual_hi
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use dual_variables, only : idual_hi, idual_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   initial setup starts   */
      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag   */
         call search_tag ( '<dual_potential>', 16, iounit, ierr )

!        /*   dual:  high and low levels   */
         read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   file close   */
         close( iounit )

!        /*   stop on error   */
         call error_handling &
     &      ( ierr, 'subroutine getforce_dual_hi', 27 )

!        /*   setup complete   */
         iset = 1

!     /*   initial setup ends   */
      end if

!-----------------------------------------------------------------------
!     /*   choose between new or old routine                          */
!-----------------------------------------------------------------------

!     /*   same potentials   */
      if ( idual_hi(1:len_trim(idual_hi)) .eq. &
     &     idual_lo(1:len_trim(idual_lo)) ) then

!        /*   use old routine   */
         call getforce_dual_hi_old

!     /*   different potentials   */
      else

!        /*   use new routine   */
         call getforce_dual_hi_new

!     /*   end of if statement   */
      end if

      return
      end





!***********************************************************************
      subroutine getforce_dual_hi_new
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir, potential, ipotential, &
     &   istep, nbead, iounit_dual, x, y, z

      use dual_variables, only : &
     &   pot_high, potential_high, fx_high, fy_high, fz_high, vir_high, &
     &   dipx_high, dipy_high, dipz_high, pot_low, potential_low, &
     &   iprint_dual, idual_hi, x_trial, y_trial, z_trial

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      character(len=80) :: ipotential_save

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual

!-----------------------------------------------------------------------
!     /*   force calculation                                          */
!-----------------------------------------------------------------------

      ipotential_save = ipotential

      ipotential = idual_hi

      call getforce

      ipotential = ipotential_save

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_high(:)    = pot(:)

      potential_high = potential

      x_trial(:,:)    = x(:,:)
      y_trial(:,:)    = y(:,:)
      z_trial(:,:)    = z(:,:)

      fx_high(:,:)   = fx(:,:)
      fy_high(:,:)   = fy(:,:)
      fz_high(:,:)   = fz(:,:)

      vir_high(:,:)  = vir(:,:)

      dipx_high(:)   = dipx(:)
      dipy_high(:)   = dipy(:)
      dipz_high(:)   = dipz(:)

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_dual) .eq. 0 ) then

         do i = 1, nbead

            write ( iounit_dual, '(i8,4f16.8)' ) &
     &         istep, pot_high(i), pot_low(i), &
     &         potential_high, potential_low

         end do

         flush( iounit_dual )

      end if

      return
      end





!***********************************************************************
      subroutine getforce_dual_hi_old
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, dipx, dipy, dipz, pot, vir, ux, uy, uz, x, y, z, &
     &   potential, natom, nbead, mbox, iounit, istep, &
     &   iounit_dual, pimd_command

      use dual_variables, only : &
     &   x_hi, y_hi, z_hi, e_hi, fx_hi, fy_hi, fz_hi, dipx_hi, dipy_hi, &
     &   dipz_hi, pot_high, potential_high, fx_high, fy_high, fz_high, &
     &   vir_high, dipx_high, dipy_high, dipz_high, pot_low, &
     &   potential_low, scr_dir_hi, iprint_dual

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp, xi, yi, zi

      character(len=80) :: char_file
      character(len=80) :: char_dir
      character(len=3)  :: char_num

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_dual

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do j = 1, nbead

         call int3_to_char( j, char_num )

!-----------------------------------------------------------------------
!        /*   x_hi, y_hi, z_hi                                        */
!-----------------------------------------------------------------------

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            x_hi(i) = xi
            y_hi(i) = yi
            z_hi(i) = zi

         end do

!-----------------------------------------------------------------------
!        /*   make geometry.ini                                       */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_hi) // '/' // char_num // &
     &               '/geometry.ini'

         call iogeometry( 1, char_file, len(char_file), iounit, &
     &                       x_hi, y_hi, z_hi, natom, 1 )

!-----------------------------------------------------------------------
!        /*   execute at subdirectories                               */
!-----------------------------------------------------------------------

         char_dir = trim(scr_dir_hi) // '/' // char_num

         call system ('cd ' // char_dir // '; ' // trim(pimd_command) &
     &                 // ' > ./monitor.out; cd ../../')

!-----------------------------------------------------------------------
!        /*   read forces.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_hi) // '/' // char_num // &
     &               '/forces.out'

         call ioforce ( 2, char_file, len(char_file), iounit, &
     &                  e_hi, fx_hi, fy_hi, fz_hi, natom, 1 )

!-----------------------------------------------------------------------
!        /*   read dipole.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(scr_dir_hi) // '/' // char_num // &
     &               '/dipole.out'

         call iodipole( 2, char_file, len(char_file), iounit, &
     &                  dipx_hi, dipy_hi, dipz_hi, 1 )

!-----------------------------------------------------------------------
!        /*    sum up energies                                        */
!-----------------------------------------------------------------------

         pot(j) = e_hi(1)

!-----------------------------------------------------------------------
!        /*    sum up forces                                          */
!-----------------------------------------------------------------------

         do i = 1, natom

            fx(i,j) = fx_hi(i)
            fy(i,j) = fy_hi(i)
            fz(i,j) = fz_hi(i)

         end do

!-----------------------------------------------------------------------
!        /*   dipole                                                  */
!-----------------------------------------------------------------------

         dipx(j) = dipx_hi(1)
         dipy(j) = dipy_hi(1)
         dipz(j) = dipz_hi(1)

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         vir(1,1) = vir(1,1) + fx(i,j) * ( ux(i,1) - x(i,j) )
         vir(1,2) = vir(1,2) + fx(i,j) * ( uy(i,1) - y(i,j) )
         vir(1,3) = vir(1,3) + fx(i,j) * ( uz(i,1) - z(i,j) )
         vir(2,1) = vir(2,1) + fy(i,j) * ( ux(i,1) - x(i,j) )
         vir(2,2) = vir(2,2) + fy(i,j) * ( uy(i,1) - y(i,j) )
         vir(2,3) = vir(2,3) + fy(i,j) * ( uz(i,1) - z(i,j) )
         vir(3,1) = vir(3,1) + fz(i,j) * ( ux(i,1) - x(i,j) )
         vir(3,2) = vir(3,2) + fz(i,j) * ( uy(i,1) - y(i,j) )
         vir(3,3) = vir(3,3) + fz(i,j) * ( uz(i,1) - z(i,j) )
      end do
      end do

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force

!-----------------------------------------------------------------------
!     /*   save                                                       */
!-----------------------------------------------------------------------

      pot_high(:)    = pot(:)

      potential_high = potential

      fx_high(:,:)   = fx(:,:)
      fy_high(:,:)   = fy(:,:)
      fz_high(:,:)   = fz(:,:)

      vir_high(:,:)  = vir(:,:)

      dipx_high(:)   = dipx(:)
      dipy_high(:)   = dipy(:)
      dipz_high(:)   = dipz(:)

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_dual) .eq. 0 ) then

         do i = 1, nbead

            write ( iounit_dual, '(i8,4f16.8)' ) &
     &         istep, pot_high(i), pot_low(i), &
     &         potential_high, potential_low

         end do

      end if

      return
      end





!***********************************************************************
      subroutine getforce_dual_restore
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, potential, fx, fy, fz, vir, dipx, dipy, dipz

      use dual_variables, only : &
     &   pot_low, potential_low, fx_low, fy_low, fz_low, vir_low, &
     &   dipx_low, dipy_low, dipz_low

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   restore                                                    */
!-----------------------------------------------------------------------

      pot(:)     = pot_low(:)

      potential  = potential_low

      fx(:,:)    = fx_low(:,:)
      fy(:,:)    = fy_low(:,:)
      fz(:,:)    = fz_low(:,:)

      vir(:,:)   = vir_low(:,:)

      dipx(:)    = dipx_low(:)
      dipy(:)    = dipy_low(:)
      dipz(:)    = dipz_low(:)

      return
      end





!***********************************************************************
      subroutine analysis_slhmc
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, potential, iounit, istep

      use hmc_variables, only : &
     &   potential_cor

      use dual_variables, only : &
     &   params_dual, iprint_dual

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: v_high, v_low, weight, dlog_weight
      integer :: iprint_slhmc
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         open ( iounit, file = 'slhmc.out', access='append' )

         write( iounit, '(a)' ) &
     &      '----------------------------------------' // &
     &      '----------------'
         write( iounit, '(a)' ) &
     &      '    step          V_high           V_low' // &
     &      '          weight'
         write( iounit, '(a)' ) &
     &      '----------------------------------------' // &
     &      '----------------'

         close( iounit )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   print interval                                             */
!-----------------------------------------------------------------------

      iprint_slhmc = iprint_dual

!-----------------------------------------------------------------------
!     /*   energies and weight                                        */
!-----------------------------------------------------------------------

!     //   high level
      v_high = potential

!     //   low level
      v_low  = potential - potential_cor

!     //   log of weight
      dlog_weight = beta * (params_dual(1)-1.d0) * potential_cor
      dlog_weight = min( 75.d0, dlog_weight )

!     //   weight
      weight = exp( dlog_weight )

!-----------------------------------------------------------------------
!     /*   print energies and weight                                  */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_slhmc) .eq. 0 ) then

         open ( iounit, file = 'slhmc.out', access='append' )

         write ( iounit, '(i8,2f16.8,e16.8)' ) &
     &      istep, v_high, v_low, weight

         close ( iounit )

      end if

      return
      end






!***********************************************************************
      subroutine justtrained_slhmc
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use hmc_variables, only : &
     &   potential_cor, hamiltonian_hmc, hamiltonian_hmc_save

      use dual_variables, only : &
     &   params_dual, justtrained_dual, potential_low, &
     &   potential_low_save, potential_high_save

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   update low-level potential, hamiltonian after training     */
!-----------------------------------------------------------------------

!     /*   if just trained   */
      if ( justtrained_dual .eq. 1 ) then

!        /*   old correction   */
         potential_cor = potential_high_save - potential_low_save

!        /*   subtract old correction to hamiltonian_hmc   */
         hamiltonian_hmc = hamiltonian_hmc &
     &      + ( 1.d0 - params_dual(1) ) * potential_cor

!        /*   add new correction to hamiltonian_hmc   */
         call correct_dual_hamiltonian

!        /*   save new potential and hamiltonian   */
         potential_low_save  = potential_low
         hamiltonian_hmc_save  = hamiltonian_hmc

!        /*   reset flag   */
         justtrained_dual = 0

!     /*   if just trained   */
      end if

      return
      end
