!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    May 24, 2022 by M. Shiga
!      Description:     path integral hybrid Monte Carlo (test)
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine setup_pimd_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup centroid md   */
      call setup_pimd_nvt_mnhc

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine setup_pihmc_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup path integral hybrid monte carlo   */
      call setup_pihmc_nvt

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine setup_pibcmd_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup brownian chain molecular dynamics   */
      call setup_pibcmd

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine setup_rpmd_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup ring polymer md   */
      call setup_rpmd

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine setup_trpmd_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup ring polymer md   */
      call setup_rpmd

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine setup_cmd_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup centroid md   */
      call setup_cmd

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity( 0 )

      return
      end





!***********************************************************************
      subroutine pihmccycle_1d
!***********************************************************************
!=======================================================================
!
!     path integral hybrid monte carlo cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   iexit, istep, istep_start, istep_end, nstep, istep_hmc

!     /*   shared variables   */
      use hmc_variables, only : &
     &   jstep

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce

!     /*   cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   energy   */
      call getenergy_second_hmc_nvt

!     /*   save data   */
      call save_second_hmc_nvt

!     /*   adjust HMC step   */
      call adjust_step_hmc

!     /*   calculate the hamiltonian and temperature   */
      call standard_hmc

!     /*   do some analysis   */
      call analysis_second_hmc( 1 )

!     /*   start main cycle   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   start molecular dynamics cycle   */
         do jstep = 1, istep_hmc

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   update all the normal mode coordinates   */
            call update_posvel_anal

!           /*   set y and z position to zero   */
            call resetposition_1d

!           /*   normal mode position -> cartesian position   */
            call nm_trans( 0 )

!           /*   get interatomic forces   */
            call getforce

!           /*   cartesian force -> normal mode force   */
            call nm_trans_force( 1 )

!           /*   update the velocities by interatomic forces   */
            call update_vel

!           /*   set y and z velocity to zero   */
            call resetvelocity_1d

!           /*   normal mode velocity -> cartesian velocity   */
            call nm_trans_velocity( 0 )

!        /*   end of molecular dynamics cycle   */
         end do

!        /*   energy   */
         call getenergy_second_hmc_nvt

!        /*   judge accept or reject   */
         call judge_second_hmc_1d

!        /*   adjust HMC step   */
         call adjust_step_hmc

!        /*   calculate the hamiltonian and temperature   */
         call standard_hmc

!        /*   do some analysis   */
         call analysis_second_hmc( 2 )

!        /*   output restart   */
         call backup_pihmc_second_nvt

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!     /*   end of main cycle   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_1d
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans(0)

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_0

!     /*   do some analysis   */
      call analysis ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         call nm_velocity_mode_1d

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   set y and z position to zero   */
         call resetposition_1d

!        /*   normal mode position -> Cartesian position   */
         call nm_trans(0)

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_0

!        /*   do some analysis   */
         call analysis ( 2 )

!        /*   output restart   */
         call backup_pibcmd

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine rpmdcycle_1d
!***********************************************************************
!=======================================================================
!
!     The RPMD molecular dynamics cycle (analytical version).
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans(0)

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_0

!     /*   do some analysis   */
      call analysis ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   set y and z position to zero   */
         call resetposition_1d

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode position -> Cartesian position   */
         call nm_trans(0)

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_0

!        /*   do some analysis   */
         call analysis ( 2 )

!        /*   output restart   */
         call backup_pimd_nve

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine trpmdcycle_1d
!***********************************************************************
!=======================================================================
!
!     The thermostatted RPMD molecular dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans( 0 )

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nve_0

!     /*   do some analysis   */
      call analysis ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd( 1 )

!        /*   set y and z position to zero   */
         call resetposition_1d

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode position -> Cartesian position   */
         call nm_trans( 0 )

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force( 1 )

!        /*   get harmonic force  */
         call getforce_ref

!        /*   update the velocities by interatomic forces   */
         call update_posvel_trpmd( 2 )

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_nve_0

!        /*   do some analysis   */
         call analysis ( 2 )

!        /*   output restart   */
         call backup_pimd_nve

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine cmdcycle_1d
!***********************************************************************
!-----------------------------------------------------------------------
!
!     The CMD molecular dynamics cycle.
!
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans(0)

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_cmd_0

!     /*   do some analysis   */
      call analysis ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_1d

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   set y and z velocity to zero   */
            call resetvelocity_1d

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   set y and z position to zero   */
            call resetposition_1d

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_1d

!           /*   set y and z velocity to zero   */
            call resetvelocity_1d

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans(0)

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_cmd_0

!        /*   do some analysis   */
         call analysis ( 2 )

!        /*   output restart   */
         call backup_cmd

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine resetposition_1d
!***********************************************************************

!     //   shared variables
      use common_variables, only : uy, uz, ndof, natom

!     //   initialize local variables
      implicit none

      uy(:,:) = 0.d0
      uz(:,:) = 0.d0

      ndof = natom

      return
      end





!***********************************************************************
      subroutine resetvelocity_1d
!***********************************************************************

!     //   shared variables
      use common_variables, only : vuy, vuz, ndof, natom, nbead

!     //   initialize local variables
      implicit none

      vuy(:,:) = 0.d0
      vuz(:,:) = 0.d0

      ndof = natom*nbead

      return
      end





!***********************************************************************
      subroutine judge_second_hmc_1d
!***********************************************************************
!=======================================================================
!
!     metropolis step for hybrid monte carlo.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, box, beta, iboundary

!     /*   shared variables   */
      use hmc_variables, only : &
     &   hamiltonian_hmc, hamiltonian_hmc_save, ratio, bfactor, &
     &   x_hmc_last, y_hmc_last, z_hmc_last, pot_hmc_last, box_hmc_last, &
     &   fx_hmc_last, fy_hmc_last, fz_hmc_last, box_save, &
     &   naccept, nreject

!     /*   local variables   */
      implicit none

!     /*   local variables   */
      real(8) :: ranf1, randomno, det3

!     /*   option   */
      integer :: ioption = 0

!-----------------------------------------------------------------------

      x_hmc_last(:,:) = x(:,:)
      y_hmc_last(:,:) = y(:,:)
      z_hmc_last(:,:) = z(:,:)

      pot_hmc_last(:) = pot(:)

      fx_hmc_last(:,:) = fx(:,:)
      fy_hmc_last(:,:) = fy(:,:)
      fz_hmc_last(:,:) = fz(:,:)

      box_hmc_last(:,:) = box(:,:)

!-----------------------------------------------------------------------

      bfactor = beta *( hamiltonian_hmc - hamiltonian_hmc_save )

      if ( ioption .eq. 1 ) then
      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         bfactor = bfactor - 2.d0 * log( det3(box) / det3(box_save) )
      end if
      end if

      if ( bfactor .lt. 75.d0 ) then

         if ( bfactor .le. 0.d0 ) then

!           /*   accepted   */
            naccept = naccept + 1

         else

            randomno = ranf1()

            if ( exp(-bfactor) .gt. randomno ) then

!              /*   accepted   */
               naccept = naccept + 1

            else

!              /*   rejected   */
               nreject = nreject + 1

!              /*   recover saved data   */
               call recover_second_hmc_nvt

            end if

         end if

      else

!        /*   rejected   */
         nreject = nreject + 1

!        /*   recover saved data   */
         call recover_second_hmc_nvt

      end if

!     /*   acceptance ratio   */
      ratio = dble(naccept) / dble(naccept+nreject)

!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call nm_velocity

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d

!     /*   normal mode velocity -> cartesian velocity   */
      call nm_trans_velocity( 0 )

!     /*   energy   */
      call getenergy_second_hmc_nvt

!     /*   save data   */
      call save_second_hmc_nvt

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine nm_velocity_mode_1d
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, beta, vux, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: gasdev, vsigma, ekin_cor

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      ekin_cor = 0.d0

      do i = 2, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i)

         vsigma = sqrt(1.d0/beta/fictmass(j,i))

         vux(j,i) = vsigma*gasdev()

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end





!***********************************************************************
      subroutine update_mnhc_mode_1d
!***********************************************************************
!=======================================================================
!
!     integrate Nose-Hoover chain thermostat attached to non-centroid
!     normal modes.
!
!     reference:  G. J. Martyna et al., Mol. Phys., 87, 1117 (1996).
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, fxbath, gkt, qmass, vxbath, xbath, dt_ref, &
     &   ysweight, nys, natom, nbead, nnhc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, j, k

      real(8) :: dt_ys, dkinx, scalex, vxfact, pvxfact

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 2, nbead

      do j = 1, natom

      do iys = 1, nys

         dt_ys = dt_ref*ysweight(iys)

!-----------------------------------------------------------------------
!        /*   kinetic energy for each atom                            */
!-----------------------------------------------------------------------

         dkinx = fictmass(j,k)*vux(j,k)*vux(j,k)

         scalex = 1.d0

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (dkinx - gkt)/qmass(k)

         do i = 2, nnhc
            fxbath(j,i,k) &
     &      = (qmass(k)*vxbath(j,i-1,k)*vxbath(j,i-1,k) - gkt)/qmass(k)
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,nnhc-i+1,k)*dt_ys)

            vxbath(j,nnhc-i,k) = vxbath(j,nnhc-i,k)*vxfact*vxfact &
     &                         + 0.25d0*fxbath(j,nnhc-i,k)*vxfact*dt_ys

         end do

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         pvxfact = exp(-0.5d0*vxbath(j,1,k)*dt_ys)

         scalex = scalex*pvxfact

!-----------------------------------------------------------------------
!        /*   update thermostat forces                                */
!-----------------------------------------------------------------------

         fxbath(j,1,k) = (scalex*scalex*dkinx - gkt)/qmass(k)

!-----------------------------------------------------------------------
!        /*   update thermostat positions                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc
            xbath(j,i,k) = xbath(j,i,k) + 0.5d0*vxbath(j,i,k)*dt_ys
         end do

!-----------------------------------------------------------------------
!       /*   update thermostat velocities                             */
!-----------------------------------------------------------------------

         do i = 1, nnhc-1

            vxfact = exp(-0.125d0*vxbath(j,i+1,k)*dt_ys)

            vxbath(j,i,k) = vxbath(j,i,k)*vxfact*vxfact &
     &                    + 0.25d0*fxbath(j,i,k)*vxfact*dt_ys

            fxbath(j,i+1,k) &
     &      = (qmass(k)*vxbath(j,i,k)*vxbath(j,i,k) - gkt)/qmass(k)

         end do

         vxbath(j,nnhc,k) = vxbath(j,nnhc,k) &
     &                    + 0.25d0*fxbath(j,nnhc,k)*dt_ys

!-----------------------------------------------------------------------
!        /*   update atomic velocities                                */
!-----------------------------------------------------------------------

         vux(j,k) = vux(j,k)*scalex

      end do

      end do
      end do

      return
      end





!***********************************************************************
      subroutine pimdcycle_1d
!***********************************************************************
!=======================================================================
!
!     the molecular dynamics cycle:  thermostat type III.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans(0)

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_nvt_mnhc

!     /*   do some analysis   */
      call analysis ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update thermostats attached to centroids  */
         call update_mnhc_cent_1d

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_1d

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*  update thermostats attached to non-centroid modes  */
            call update_mnhc_mode_1d

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans(0)

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update thermostats attached to centroids  */
         call update_mnhc_cent_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_nvt_mnhc

!        /*   do some analysis   */
         call analysis ( 2 )

!        /*   output restart   */
         call backup_pimd_nvt_mnhc

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine update_mnhc_cent_1d
!***********************************************************************
!=======================================================================
!
!     Update Nose-Hoover chain thermostat attached to centroid.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, ysweight, qmass_cent, dt, fxbath_cent, gkt, &
     &   vxbath_cent, xbath_cent, nys, nnhc, ncolor, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iys, inhc, j, l

      real(8) :: dt_ys, dkinx, scalex, vxfact, pvxfact

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do iys = 1, nys

!-----------------------------------------------------------------------
!     /*   step size                                                  */
!-----------------------------------------------------------------------

      dt_ys = dt*ysweight(iys) /2.d0

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = 1, ncolor
         do j = 1, natom

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,1)*vux(j,1)*vux(j,1)

            scalex = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!        /*   massive nhc start                                       */
!-----------------------------------------------------------------------

         do l = ncolor, 1, -1
         do j = natom, 1, -1

!           /*   calculate total kinetic energy of the system   */

            dkinx = fictmass(j,1)*vux(j,1)*vux(j,1)

            scalex = 1.d0

            fxbath_cent(j,1,l) = (dkinx - gkt)/qmass_cent(1,l)

            do i = 2, nnhc
              fxbath_cent(j,i,l) = &
     &           (qmass_cent(i-1,l)*vxbath_cent(j,i-1,l) &
     &           *vxbath_cent(j,i-1,l) - gkt)/qmass_cent(i,l)
            end do

!           /*   update the thermostat velocities   */

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys

            do inhc = 1, nnhc-1

               vxfact &
     &            = exp(-0.125d0*vxbath_cent(j,nnhc-inhc+1,l)*dt_ys)

               vxbath_cent(j,nnhc-inhc,l) = &
     &            vxbath_cent(j,nnhc-inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,nnhc-inhc,l)*vxfact*dt_ys

            end do

!           /*   update the particle velocities   */

            pvxfact = exp(-0.5d0*vxbath_cent(j,1,l)*dt_ys)

            scalex = scalex*pvxfact

!           /*   update the force   */

            fxbath_cent(j,1,l) &
     &         = (scalex*scalex*dkinx - gkt)/qmass_cent(1,l)

!           /*   update the thermostat positions   */
            do inhc = 1, nnhc
               xbath_cent(j,inhc,l) = xbath_cent(j,inhc,l) &
     &            + 0.5d0*vxbath_cent(j,inhc,l)*dt_ys
            end do

!          /*   update the thermostat velocities   */

            do inhc = 1, nnhc-1

               vxfact = exp(-0.125d0*vxbath_cent(j,inhc+1,l)*dt_ys)

               vxbath_cent(j,inhc,l) = &
     &            vxbath_cent(j,inhc,l)*vxfact*vxfact &
     &            + 0.25d0*fxbath_cent(j,inhc,l)*vxfact*dt_ys

               fxbath_cent(j,inhc+1,l) = &
     &            (qmass_cent(inhc,l)*vxbath_cent(j,inhc,l) &
     &            *vxbath_cent(j,inhc,l) - gkt)/qmass_cent(inhc+1,l)

            end do

            vxbath_cent(j,nnhc,l) = vxbath_cent(j,nnhc,l) &
     &         + 0.25d0*fxbath_cent(j,nnhc,l)*dt_ys

!           /*   update the paricle velocities   */

            vux(j,1) = vux(j,1)*scalex

!-----------------------------------------------------------------------
!     /*   massive nhc end                                            */
!-----------------------------------------------------------------------

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end

