!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 29, 2025 by M. Shiga
!      Description:     brownian chain molecular dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine pibcmdcycle_MPI
!***********************************************************************

!     /*   shared variables   */
      use common_variables, only : integrator_bcmd
      use cons_variables, only : ncons

!     /*   local variables   */
      implicit none

!     /*   constraints   */
      if ( ncons .ge. 1 ) then

         call pibcmdcycle_cons_MPI

!     /*   integrator: respa   */
      else if ( integrator_bcmd(1:6) .eq. 'RESPA ' ) then

         call pibcmdcycle_anal_MPI

!     /*   integrator: euler-maruyama   */
      else if ( integrator_bcmd(1:6) .eq. 'EULER ' ) then

         call pibcmdcycle_euler_MPI

!     /*   integrator: gillespie   */
      else if ( integrator_bcmd(1:10) .eq. 'GILLESPIE ' ) then

         call pibcmdcycle_gillespie_MPI

!     /*   integrator: platen   */
      else if ( integrator_bcmd(1:7) .eq. 'PLATEN ' ) then

         call pibcmdcycle_platen_MPI

!     /*   integrator   */
      end if

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_anal_MPI
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, kickstep_bcmd

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         if ( mod(istep,kickstep_bcmd) .eq. 0 ) then
            call nm_velocity_mode_MPI
         end if

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update all the normal mode coordinates   */
         call update_posvel_anal

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pibcmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_euler_MPI
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref, &
     &   kickstep_bcmd

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_0_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         if ( mod(istep,kickstep_bcmd) .eq. 0 ) then
            call nm_velocity_mode_MPI
         end if

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI( 0 )

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_0_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pibcmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_cons_MPI
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref, &
     &   kickstep_bcmd

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   get constrained force  */
      call getforce_ref_cons_cent_MPI

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_cons_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         if ( mod(istep,kickstep_bcmd) .eq. 0 ) then
            call nm_velocity_mode_MPI
         end if

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   start multiple time step cycle   */
         do iref = 1, nref

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

!           /*   update all the normal mode coordinates   */
            call update_pos

!           /*   get harmonic forces   */
            call getforce_ref

!           /*   get constrained force  */
            call getforce_ref_cons_cent_MPI

!           /*   update the velocities by harmonic forces   */
            call update_vel_ref

         end do

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI( 0 )

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_cons_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pibcmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_gillespie_MPI
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, kickstep_bcmd

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI(1)

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_gillespie_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         if ( mod(istep,kickstep_bcmd) .eq. 0 ) then
            call nm_velocity_mode_gillespie_MPI
         end if

!        /*   update the velocities by interatomic forces   */
         call update_vel_gillespie

!        /*   update the velocities by interatomic forces   */
         call update_pos_gillespie

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI(0)

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel_gillespie

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_gillespie_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pibcmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine pibcmdcycle_platen_MPI
!***********************************************************************
!=======================================================================
!
!     path integral brownian chain molecular dynamics cycle
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit, iref, nref, &
     &   kickstep_bcmd

!     /*   local variables   */
      implicit none

!     /*   unset respa steps   */
      iref = 1
      nref = 1

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_MPI(0)

!     /*   get interatomic forces   */
      call getforce_MPI

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_MPI( 1 )

!     /*   get harmonic force  */
      call getforce_ref

!     /*   save forces   */
      call update_pibcmd_platen_MPI( 0 )

!     /*   calculate the hamiltonian and temperature   */
      call standard_pibcmd_platen_MPI

!     /*   do some analysis   */
      call analysis_MPI ( 1 )

!     /*   main loop   */
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   random mode velocity   */
         if ( mod(istep,kickstep_bcmd) .eq. 0 ) then
            call nm_velocity_mode_MPI
         end if

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   update the velocities by harmonic forces   */
         call update_vel_ref

!        /*   update all the normal mode coordinates   */
         call update_pos

!        /*   get harmonic forces   */
         call getforce_ref

!        /*   update the velocities by harmonic forces   */
         call update_vel_ref

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_MPI( 0 )

!        /*   get interatomic forces   */
         call getforce_MPI

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_MPI( 1 )

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_MPI( 0 )

!        /*   update by old and new forces, save forces   */
         call update_pibcmd_platen_MPI( 1 )

!        /*   calculate the hamiltonian and temperature   */
         call standard_pibcmd_platen_MPI

!        /*   do some analysis   */
         call analysis_MPI ( 2 )

!        /*   output restart   */
         call backup_pibcmd_MPI

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     /*   main loop   */
      end do

!     /*   current step   */
      istep = istep_end

      return
      end
