!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     phonon calculations
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine phonon_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   setup atoms in the primitive cell                          */
!-----------------------------------------------------------------------

      call setup_phonon_MPI

!-----------------------------------------------------------------------
!     /*   phonon dispersion                                          */
!-----------------------------------------------------------------------

      call phonon_kdisp_MPI

!-----------------------------------------------------------------------
!     /*   k-point sampling                                           */
!-----------------------------------------------------------------------

      call phonon_kdos_MPI

!-----------------------------------------------------------------------
!     /*   phonon density of states                                   */
!-----------------------------------------------------------------------

      call phonon_dos_MPI

!-----------------------------------------------------------------------
!     /*   phonon energies                                            */
!-----------------------------------------------------------------------

      call phonon_energy_MPI

      return
      end





!***********************************************************************
      subroutine setup_phonon_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, box, hessian, natom, iounit, iounit_std, &
     &   myrank

      use phonon_variables, only : &
     &   gx_disp, gy_disp, gz_disp, pdos_min, pdos_max, dpdos, pdos, &
     &   tempmax_phonon, tempmin_phonon, tempstep_phonon, eigval_prim, &
     &   eigvec_prim, x_prim, y_prim, z_prim, box_prim, boxinv_prim, &
     &   hess_prim, npdos, ndisp, joption_phonon, i_prim, lmax_phonon, &
     &   labc_phonon, na_prim, nb_prim, nc_prim, nabc, natom_prim

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, ii, ierr, itest

!     /*   integers   */
      integer :: iimin = 0

!     /*   real numbers   */
      real(8) :: s1, s2, s3, dx, dy, dz, d2

!     /*   real numbers   */
      real(8) :: d2min = 0.d0

!     /*   real numbers   */
      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   primitive cell                                             */
!-----------------------------------------------------------------------

!     /*   master process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<cells_phonon>', 14, iounit, ierr )

!     /*   read integers   */
      read( iounit, *, iostat=ierr ) na_prim, nb_prim, nc_prim

!     /*   file close   */
      close( iounit )

!     /*   if not found, use default data   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<cells_phonon>', 14, iounit, ierr )

!        /*   read integers   */
         read( iounit, *, iostat=ierr ) na_prim, nb_prim, nc_prim

!        /*   file close   */
         close ( iounit )

!     /*   if not found, use default data   */
      end if

!     /*   master process   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine setup_phonon_MPI', 27 )

!     /*   communication   */
      call my_mpi_bcast_int_0( na_prim )
      call my_mpi_bcast_int_0( nb_prim )
      call my_mpi_bcast_int_0( nc_prim )

!     /*   number of primitive cells in the box   */
      nabc = na_prim * nb_prim * nc_prim

!-----------------------------------------------------------------------
!     /*   number of k-point sampling points                          */
!-----------------------------------------------------------------------

!     /*   restart file   */
      call read_int1_MPI ( ndisp, '<kdisp_phonon>', 14, iounit )

!     /*   memory allocation   */
      if ( .not. allocated( gx_disp ) ) &
     &   allocate( gx_disp(ndisp) )
      if ( .not. allocated( gy_disp ) ) &
     &   allocate( gy_disp(ndisp) )
      if ( .not. allocated( gz_disp ) ) &
     &   allocate( gz_disp(ndisp) )

!     /*   master process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<kdisp_phonon>', 14, iounit, ierr )

!     /*   read integer   */
      read( iounit, *, iostat=ierr ) ndisp

!     /*   read k-points   */

      if ( ierr .eq. 0 ) then
          do i = 1, ndisp
            read( iounit, *, iostat=ierr ) &
     &         gx_disp(i), gy_disp(i), gz_disp(i)
         end do
      end if

!     /*   file close   */
      close( iounit )

!     /*   if not found, use default data   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<kdisp_phonon>', 14, iounit, ierr )

!        /*   read integer   */
         read( iounit, *, iostat=ierr ) ndisp

!        /*   read k-points   */

         if ( ierr .eq. 0 ) then
            do i = 1, ndisp
               read( iounit, *, iostat=ierr ) &
     &            gx_disp(i), gy_disp(i), gz_disp(i)
            end do
         end if

!        /*   file close   */
         close ( iounit )

!     /*   if not found, use default data   */
      end if

!     /*   master process   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine setup_phonon_MPI', 27 )

!     /*   communication   */
      call my_mpi_bcast_real_1( gx_disp, ndisp )
      call my_mpi_bcast_real_1( gy_disp, ndisp )
      call my_mpi_bcast_real_1( gz_disp, ndisp )

!-----------------------------------------------------------------------
!     /*   number of k-point sampling points                          */
!-----------------------------------------------------------------------

!     /*   master process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<kdos_phonon>', 13, iounit, ierr )

!     /*   read integers   */
      read( iounit, *, iostat=ierr ) &
     &   lmax_phonon(1), lmax_phonon(2), lmax_phonon(3)

!     /*   file close   */
      close( iounit )

!     /*   if not found, use default data   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<kdos_phonon>', 13, iounit, ierr )

!        /*   read integers   */
         read( iounit, *, iostat=ierr ) &
     &      lmax_phonon(1), lmax_phonon(2), lmax_phonon(3)

!        /*   file close   */
         close ( iounit )

!     /*   if not found, use default data   */
      end if

!     /*   master process   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine setup_phonon_MPI', 27 )

!     /*   communication   */
      call my_mpi_bcast_int_1( lmax_phonon, 3 )

!     /*   total amount of k-point sampling   */
      labc_phonon = lmax_phonon(1)*lmax_phonon(2)*lmax_phonon(3)

!-----------------------------------------------------------------------
!     /*   phonon density of states                                   */
!-----------------------------------------------------------------------

!     /*   master process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<dosrange_phonon>', 17, iounit, ierr )

!     /*   read integer   */
      read( iounit, *, iostat=ierr ) pdos_min, pdos_max, dpdos

!     /*   file close   */
      close( iounit )

!     /*   if not found, use default data   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<dosrange_phonon>', 17, iounit, ierr )

!        /*   read integer   */
         read( iounit, *, iostat=ierr ) pdos_min, pdos_max, dpdos

!        /*   file close   */
         close ( iounit )

!     /*   if not found, use default data   */
      end if

!     /*   master process   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine setup_phonon_MPI', 27 )

!     /*   communication   */
      call my_mpi_bcast_real_0( pdos_min )
      call my_mpi_bcast_real_0( pdos_max )
      call my_mpi_bcast_real_0( dpdos )

!-----------------------------------------------------------------------
!     /*   set temperatures for phonon calculation                    */
!-----------------------------------------------------------------------

!     /*   master process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<temprange_phonon>', 18, iounit, ierr )

!     /*   read integers   */
      read( iounit, *, iostat=ierr ) &
     &   tempmin_phonon, tempmax_phonon, tempstep_phonon

!     /*   file close   */
      close( iounit )

!     /*   if not found, use default data   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<temprange_phonon>', 18, iounit, ierr )

!        /*   read integers   */
         read( iounit, *, iostat=ierr ) &
     &      tempmin_phonon, tempmax_phonon, tempstep_phonon

!        /*   file close   */
         close ( iounit )

!     /*   if not found, use default data   */
      end if

!     /*   master process   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine setup_phonon_MPI', 27 )

!     /*   communication   */
      call my_mpi_bcast_real_0( tempmin_phonon )
      call my_mpi_bcast_real_0( tempmax_phonon )
      call my_mpi_bcast_real_0( tempstep_phonon )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      npdos = nint ( ( pdos_max - pdos_min ) / dpdos ) + 1

      if ( .not. allocated( pdos ) ) &
     &   allocate( pdos(npdos) )

      pdos(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   number of atoms in the primitive cell                      */
!-----------------------------------------------------------------------

      natom_prim = natom / ( na_prim*nb_prim*nc_prim )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( hess_prim ) ) &
     &   allocate( hess_prim(3*natom_prim,3*natom_prim) )

      if ( .not. allocated( eigval_prim ) ) &
     &   allocate( eigval_prim(3*natom_prim) )

      if ( .not. allocated( eigvec_prim ) ) &
     &   allocate( eigvec_prim(3*natom_prim,3*natom_prim) )

      if ( .not. allocated( x_prim ) ) &
     &   allocate( x_prim(natom_prim) )
      if ( .not. allocated( y_prim ) ) &
     &   allocate( y_prim(natom_prim) )
      if ( .not. allocated( z_prim ) ) &
     &   allocate( z_prim(natom_prim) )

      if ( .not. allocated( i_prim ) ) &
     &   allocate( i_prim(natom) )

!-----------------------------------------------------------------------
!     /*   lattice vectors of the primitive cell                      */
!-----------------------------------------------------------------------

      box_prim(:,1) = box(:,1) / na_prim
      box_prim(:,2) = box(:,2) / nb_prim
      box_prim(:,3) = box(:,3) / nc_prim

!-----------------------------------------------------------------------
!     /*   inverse matrix of the primitive lattice                    */
!-----------------------------------------------------------------------

      call inv3( box_prim, boxinv_prim )

      j = 0

      do i = 1, natom

         s1 = boxinv_prim(1,1)*(ux(i,1)-ux(1,1)) &
     &      + boxinv_prim(1,2)*(uy(i,1)-uy(1,1)) &
     &      + boxinv_prim(1,3)*(uz(i,1)-uz(1,1))

         s2 = boxinv_prim(2,1)*(ux(i,1)-ux(1,1)) &
     &      + boxinv_prim(2,2)*(uy(i,1)-uy(1,1)) &
     &      + boxinv_prim(2,3)*(uz(i,1)-uz(1,1))

         s3 = boxinv_prim(3,1)*(ux(i,1)-ux(1,1)) &
     &      + boxinv_prim(3,2)*(uy(i,1)-uy(1,1)) &
     &      + boxinv_prim(3,3)*(uz(i,1)-uz(1,1))

         if ( int(s1+tiny_value) .ne. 0 ) cycle
         if ( int(s2+tiny_value) .ne. 0 ) cycle
         if ( int(s3+tiny_value) .ne. 0 ) cycle

         j = j + 1

         x_prim(j) = x(i,1)
         y_prim(j) = y(i,1)
         z_prim(j) = z(i,1)

      end do

      if ( j .ne. natom_prim ) then
         call error_handling_MPI( 1, 'subroutine setup_phonon_MPI', 27 )
      end if

!-----------------------------------------------------------------------
!     /*   print comments                                             */
!-----------------------------------------------------------------------

      call my_mpi_barrier

      if ( myrank .eq. 0 ) then

      write( 6, '(a,i5)' ) &
     &   'Number of atoms in primitive cell:', natom_prim
      write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     /*   coordinates in the primitive cell                          */
!-----------------------------------------------------------------------

      do i     = 1, natom

!        /*   find atom in primitive cell  */

         do ii = 1, natom_prim

            dx = x(i,1) - x_prim(ii)
            dy = y(i,1) - y_prim(ii)
            dz = z(i,1) - z_prim(ii)

            call pbc_atom_prim( dx, dy, dz )

            d2 = dx*dx + dy*dy + dz*dz

            if ( ii .eq. 1 ) then
               iimin  = ii
               d2min  = d2
            else if ( d2 .lt. d2min ) then
               iimin  = ii
               d2min  = d2
            end if

         end do

         if ( d2min .gt. tiny_value ) then

            if ( myrank .eq. 0 ) then

            write( 6, '(a,f10.5)' ) &
     &         'Error - Atomic positions did not match ' // &
     &         'the ones in the primitive cell.'
            write( 6, '(a)' )

            write( iounit_std, '(a,f10.5)' ) &
     &         'Error - Atomic positions did not match ' // &
     &         'the ones in the primitive cell.'
            write( iounit_std, '(a)' )

            end if

            call error_handling_MPI &
     &         ( 1, 'subroutine setup_phonon_MPI', 23 )

         end if

         i_prim(i) = iimin

      end do

!-----------------------------------------------------------------------
!     /*   print comments                                             */
!-----------------------------------------------------------------------

      call my_mpi_barrier

      if ( myrank .eq. 0 ) then

      write( 6, '(a)' ) &
     &   'All atomic positions matched the ones in primitive cell.'
      write( 6, '(a)' )

      end if

!-----------------------------------------------------------------------
!     /*   restart hessian file                                       */
!-----------------------------------------------------------------------

!     /*   check if file called `hessian.out' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'hessian.out', 11, itest )
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0( itest )

!     /*   if the file exists restart   */

      if ( itest .eq. 0 ) then

         joption_phonon = 0

!     /*   if not start from scratch  */

      else

         joption_phonon = 1

      end if

!-----------------------------------------------------------------------
!     /*   print comments                                             */
!-----------------------------------------------------------------------

      call my_mpi_barrier

      if ( myrank .eq. 0 ) then

      if ( joption_phonon .eq. 0 ) then
         write( 6, '(a)' ) 'Hessian is read from hessian.out.'
         write( 6, '(a)' ) 
      else if ( joption_phonon .eq. 1 ) then
         write( 6, '(a)' ) 'Hessian is calculated from scratch.'
         write( 6, '(a)' ) 
      end if

      end if

!-----------------------------------------------------------------------
!     /*   get hessian matrix                                         */
!-----------------------------------------------------------------------

      if ( joption_phonon .eq. 0 ) then

!        /*   memory allocation:  hessian   */
         if ( .not. allocated( hessian ) ) &
     &      allocate( hessian(3*natom,3*natom,1) )

!        /*   read from hessian.out   */
         call restart_hess_MPI ( 1 )

      else

!        /*   compute hessian   */
         call gethess_paral_MPI

      end if

!-----------------------------------------------------------------------
!     /*   symmetrized hessian matrix                                 */
!-----------------------------------------------------------------------

      do i = 1, 3*natom-1
      do j = i+1, 3*natom
         hessian(i,j,1) = 0.5d0*hessian(i,j,1) + 0.5d0*hessian(j,i,1)
         hessian(j,i,1) = hessian(i,j,1)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   project out translation and rotation                       */
!-----------------------------------------------------------------------

      call project_out_phonon

!-----------------------------------------------------------------------
!     /*   print comments                                             */
!-----------------------------------------------------------------------

      call my_mpi_barrier

      if ( myrank .eq. 0 ) then

      write( 6, '(a)' ) &
     &   'Projection and symmetry applied to hessian matrix.'
      write( 6, '(a)' )

      end if

      return
      end





!***********************************************************************
      subroutine phonon_kdos_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, hessian, physmass, pi, speedlight_SI, &
     &   au_time, au_length, natom, iounit, myrank

      use phonon_variables, only : &
     &   boxinv_prim, hess_prim, eigval_prim, eigvec_prim, &
     &   lmax_phonon, i_prim, natom_prim, na_prim, nb_prim, nc_prim

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k, l, ii, kk, ij, kl, n, la, lb, lc, &
     &           iij, kkl, itest

!     /*   real numbers   */
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, gx, gy, gz, gd, &
     &           dx, dy, dz, fla, flb, flc, factor, factor_cminv

!     /*   complex numbers   */
      complex(8) :: zi = dcmplx(0.d0,1.d0)

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   check if file called `phonon_kdos.out' exists              */
!-----------------------------------------------------------------------

      call testfile ( 'phonon_kdos.out', 15, itest )

      if ( itest .eq. 0 ) then
         write( 6, '(a)' ) 'The file phonon_kdos.out already exists.'
         write( 6, '(a)' ) 
         return
      end if

!-----------------------------------------------------------------------
!     /*   reciprocal vectors                                         */
!-----------------------------------------------------------------------

      ax = 2.d0*pi*boxinv_prim(1,1)
      ay = 2.d0*pi*boxinv_prim(1,2)
      az = 2.d0*pi*boxinv_prim(1,3)

      bx = 2.d0*pi*boxinv_prim(2,1)
      by = 2.d0*pi*boxinv_prim(2,2)
      bz = 2.d0*pi*boxinv_prim(2,3)

      cx = 2.d0*pi*boxinv_prim(3,1)
      cy = 2.d0*pi*boxinv_prim(3,2)
      cz = 2.d0*pi*boxinv_prim(3,3)

!-----------------------------------------------------------------------
!     /*   open file                                                  */
!-----------------------------------------------------------------------

      open( iounit, file = 'phonon_kdos.out' )

      write( iounit, '(a)' )

      write( iounit, '(a)' ) &
     &   '======================================================'

      write( iounit, '(a)' ) &
     &   '  kx [/bohr]  ky [/bohr]  kz [/bohr]   num  freq [/cm]'

      write( iounit, '(a)' ) &
     &   '------------------------------------------------------'

      write( 6, '(a)' ) &
     &   '===================================='

      write( 6, '(a)' ) &
     &   '  kx [/bohr]  ky [/bohr]  kz [/bohr]'

      write( 6, '(a)' ) &
     &   '------------------------------------'

!-----------------------------------------------------------------------
!     /*   start loop over wave vectors                               */
!-----------------------------------------------------------------------

      do la = 1, lmax_phonon(1)
      do lb = 1, lmax_phonon(2)
      do lc = 1, lmax_phonon(3)

!-----------------------------------------------------------------------
!     /*   fractional number within the brillouin zone                */
!-----------------------------------------------------------------------

      fla = dble(2*la-lmax_phonon(1)-1) / dble(2*lmax_phonon(1))
      flb = dble(2*lb-lmax_phonon(2)-1) / dble(2*lmax_phonon(2))
      flc = dble(2*lc-lmax_phonon(3)-1) / dble(2*lmax_phonon(3))

!-----------------------------------------------------------------------
!     /*   k-point                                                    */
!-----------------------------------------------------------------------

      gx = ax*fla + bx*flb + cx*flc
      gy = ay*fla + by*flb + cy*flc
      gz = az*fla + bz*flb + cz*flc

!-----------------------------------------------------------------------
!     /*   mass weighted hessian                                      */
!-----------------------------------------------------------------------

      hess_prim(:,:) = dcmplx( 0.d0, 0.d0 )

!     /*   i and k atoms are those in the supercell  */

      do i = 1, natom
      do k = 1, natom

!        /*   ii and kk atoms are those in the primitive cell   */

         ii = i_prim(i)
         kk = i_prim(k)

!        /*   distance between atoms i and k   */

         dx = x(i,1) - x(k,1)
         dy = y(i,1) - y(k,1)
         dz = z(i,1) - z(k,1)

         call pbc_atom_MPI( dx, dy, dz )

!        /*   dot product of g and d   */

         gd = gx*dx + gy*dy + gz*dz

!        /*   mass factor   */

         factor = 1.d0 / sqrt(physmass(i)*physmass(k))

!        /*   xyz components   */

         do j = 1, 3
         do l = 1, 3

!           /*   labels of atoms in the supercell   */

            ij = 3*(i-1) + j
            kl = 3*(k-1) + l

!           /*   labels of atoms in the primitive cell   */

            iij = 3*(ii-1) + j
            kkl = 3*(kk-1) + l

!           /*   hessian of atoms in the primitive cell   */

            hess_prim(iij,kkl) = hess_prim(iij,kkl) &
     &         + hessian(ij,kl,1) * factor * exp( zi*gd )

         end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   symmetrize mass weighted hessian                           */
!-----------------------------------------------------------------------

      do i = 1, natom_prim-1
      do j = i+1, natom_prim

         hess_prim(i,j) = 0.5d0 * hess_prim(i,j) &
     &                  + 0.5d0 * dconjg(hess_prim(j,i))

         hess_prim(j,i) = dconjg(hess_prim(i,j))

      end do
      end do

      hess_prim(:,:) = hess_prim(:,:) / dble(na_prim*nb_prim*nc_prim)

!-----------------------------------------------------------------------
!     /*   diagonalize mass weighted hessian                          */
!-----------------------------------------------------------------------

      n = 3*natom_prim

      call zdiag_MPI ( hess_prim, eigval_prim, eigvec_prim, n )

!-----------------------------------------------------------------------
!     /*   put minus sign for imaginary frequencies                   */
!-----------------------------------------------------------------------

      do i = 1, n

         if ( dreal(eigval_prim(i)) .gt. 0.d0 ) then
            eigval_prim(i) = + sqrt(abs(eigval_prim(i)))
         else
            eigval_prim(i) = - sqrt(abs(eigval_prim(i)))
         end if

      end do

!-----------------------------------------------------------------------
!     /*   turn into wave numbers [cm^-1]                             */
!-----------------------------------------------------------------------

      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

      do i = 1, n
         eigval_prim(i) = eigval_prim(i) * factor_cminv
      end do

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      do i = 1, n
         write( iounit, '(3f12.6,i6,f12.2)' ) &
     &      gx, gy, gz, i, dreal(eigval_prim(i))
      end do

      write( 6, '(3f12.6)' ) gx, gy, gz

!-----------------------------------------------------------------------
!     /*   end of loop over fourier space                             */
!-----------------------------------------------------------------------

      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      close( iounit )

      write( 6, '(a)' )

      return
      end





!***********************************************************************
      subroutine phonon_kdisp_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, hessian, physmass, pi, speedlight_SI, &
     &   au_time, au_length, natom, iounit, myrank

      use phonon_variables, only : &
     &   hess_prim, gx_disp, gy_disp, gz_disp, eigval_prim, eigvec_prim, &
     &   i_prim, ndisp, natom_prim, na_prim, nb_prim, nc_prim

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k, l, ii, kk, ij, kl, n, iij, kkl, idisp, itest

!     /*   real numbers   */
      real(8) :: gx, gy, gz, gd, dx, dy, dz, factor, factor_cminv

!     /*   complex numbers   */
      complex(8) :: zi = dcmplx(0.d0,1.d0)

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   check if file called `phonon_disp.out' exists              */
!-----------------------------------------------------------------------

      call testfile ( 'phonon_disp.out', 15, itest )

      if ( itest .eq. 0 ) then
         write( 6, '(a)' ) 'The file phonon_disp.out already exists.'
         write( 6, '(a)' ) 
         return
      end if

!-----------------------------------------------------------------------
!     /*   open file                                                  */
!-----------------------------------------------------------------------

      if ( ndisp .eq. 0 ) return

      open( iounit, file = 'phonon_kdisp.out' )

      write( iounit, '(a)' )

      write( iounit, '(a)' ) &
     &   '======================================================'

      write( iounit, '(a)' ) &
     &   '  kx [/bohr]  ky [/bohr]  kz [/bohr]   num    [cm**-1]'

      write( iounit, '(a)' ) &
     &   '------------------------------------------------------'

      write( 6, '(a)' ) &
     &   '===================================='

      write( 6, '(a)' ) &
     &   '  kx [/bohr]  ky [/bohr]  kz [/bohr]'

      write( 6, '(a)' ) &
     &   '------------------------------------'

!-----------------------------------------------------------------------
!     /*   start loop over wave vectors                               */
!-----------------------------------------------------------------------

      do idisp = 1, ndisp

!-----------------------------------------------------------------------
!     /*   k-point                                                    */
!-----------------------------------------------------------------------

      gx = gx_disp(idisp)
      gy = gy_disp(idisp)
      gz = gz_disp(idisp)

!-----------------------------------------------------------------------
!     /*   mass weighted hessian                                      */
!-----------------------------------------------------------------------

      hess_prim(:,:) = dcmplx( 0.d0, 0.d0 )

!     /*   i and k atoms are those in the supercell  */

      do i = 1, natom
      do k = 1, natom

!        /*   ii and kk atoms are those in the primitive cell   */

         ii = i_prim(i)
         kk = i_prim(k)

!        /*   distance between atoms i and k   */

         dx = x(i,1) - x(k,1)
         dy = y(i,1) - y(k,1)
         dz = z(i,1) - z(k,1)

         call pbc_atom_MPI( dx, dy, dz )

!        /*   dot product of g and d   */

         gd = gx*dx + gy*dy + gz*dz

!        /*   mass factor   */

         factor = 1.d0 / sqrt(physmass(i)*physmass(k))

!        /*   xyz components   */

         do j = 1, 3
         do l = 1, 3

!           /*   labels of atoms in the supercell   */

            ij = 3*(i-1) + j
            kl = 3*(k-1) + l

!           /*   labels of atoms in the primitive cell   */

            iij = 3*(ii-1) + j
            kkl = 3*(kk-1) + l

!           /*   hessian of atoms in the primitive cell   */

            hess_prim(iij,kkl) = hess_prim(iij,kkl) &
     &         + hessian(ij,kl,1) * factor * exp( zi*gd )

         end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   symmetrize mass weighted hessian                           */
!-----------------------------------------------------------------------

      do i = 1, natom_prim-1
      do j = i+1, natom_prim

         hess_prim(i,j) = 0.5d0 * hess_prim(i,j) &
     &                  + 0.5d0 * dconjg(hess_prim(j,i))

         hess_prim(j,i) = dconjg(hess_prim(i,j))

      end do
      end do

      hess_prim(:,:) = hess_prim(:,:) / dble(na_prim*nb_prim*nc_prim)

!-----------------------------------------------------------------------
!     /*   diagonalize mass weighted hessian                          */
!-----------------------------------------------------------------------

      n = 3*natom_prim

      call zdiag_MPI ( hess_prim, eigval_prim, eigvec_prim, n )

!-----------------------------------------------------------------------
!     /*   put minus sign for imaginary frequencies                   */
!-----------------------------------------------------------------------

      do i = 1, n

         if ( dreal(eigval_prim(i)) .gt. 0.d0 ) then
            eigval_prim(i) = + sqrt(abs(eigval_prim(i)))
         else
            eigval_prim(i) = - sqrt(abs(eigval_prim(i)))
         end if

      end do

!-----------------------------------------------------------------------
!     /*   turn into wave numbers [cm^-1]                             */
!-----------------------------------------------------------------------

      factor_cminv = 1.d0 / (2.d0*pi*speedlight_SI*au_time*100.d0)

      do i = 1, n
         eigval_prim(i) = eigval_prim(i) * factor_cminv
      end do

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

      do i = 1, n
         write( iounit, '(3f12.6,i6,f12.2)' ) &
     &      gx, gy, gz, i, dreal(eigval_prim(i))
      end do

      write( 6, '(3f12.6)' ) gx, gy, gz

!-----------------------------------------------------------------------
!     /*   end of loop over fourier space                             */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      close( iounit )

      return
      end





!***********************************************************************
      subroutine phonon_energy_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, speedlight_SI, au_time, boltz, hbar, iounit, &
     &   iounit_phonon, myrank

      use phonon_variables, only: &
     &   eigval_prim, temperature_phonon, beta_phonon, tempstep_phonon, &
     &   tempmin_phonon, tempmax_phonon, natom_prim, labc_phonon, nabc, &
     &   lmax_phonon

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   unit converstion factor   */
      real(8) :: factor_cminv

!     /*   integers   */
      integer :: i, n, la, lb, lc, idummy, nstep_phonon, istep_phonon

!     /*   real values   */
      real(8) :: ho, bho, ho_half, bho_half, uqtc, fqtc, uq0c, uctc, &
     &           fctc, uc0, fc0, sc0, uq0, fq0, sq0, uct, fct, sct, &
     &           uqt, fqt, sqt, omega, dummy, dreal_eigval_prim

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   open file: phonon energy                                   */
!-----------------------------------------------------------------------

      open( iounit_phonon, file = 'phonon_energy.out' )

!-----------------------------------------------------------------------
!     /*   number of iterations                                       */
!-----------------------------------------------------------------------

      nstep_phonon &
     &   = int((tempmax_phonon-tempmin_phonon)/tempstep_phonon) + 1

!-----------------------------------------------------------------------
!     /*   loop for temperatures                                      */
!-----------------------------------------------------------------------

      temperature_phonon = tempmin_phonon

      do istep_phonon = 1, nstep_phonon

!-----------------------------------------------------------------------
!        /*   temperature and the inverse                             */
!-----------------------------------------------------------------------

         temperature_phonon = temperature_phonon + tempstep_phonon

         beta_phonon = 1.d0 / ( boltz * temperature_phonon )

!-----------------------------------------------------------------------
!        /*   open file: phonon frequency data                        */
!-----------------------------------------------------------------------

         open( iounit, file = 'phonon_kdos.out' )

         read( iounit, * )
         read( iounit, * )
         read( iounit, * )
         read( iounit, * )

!-----------------------------------------------------------------------
!        /*   number of degrees of freedom                            */
!-----------------------------------------------------------------------

         n = 3*natom_prim

!-----------------------------------------------------------------------
!        /*   compute free/internal energies and entropies            */
!-----------------------------------------------------------------------

         uc0 = 0.d0
         fc0 = 0.d0
         sc0 = 0.d0

         uct = 0.d0
         fct = 0.d0
         sct = 0.d0

         uq0 = 0.d0
         fq0 = 0.d0
         sq0 = 0.d0

         uqt = 0.d0
         fqt = 0.d0
         sqt = 0.d0

!-----------------------------------------------------------------------
!        /*   start loop over wave vectors                            */
!-----------------------------------------------------------------------

         do la = 1, lmax_phonon(1)
         do lb = 1, lmax_phonon(2)
         do lc = 1, lmax_phonon(3)

!-----------------------------------------------------------------------
!        /*   read frequencies                                        */
!-----------------------------------------------------------------------

            do i = 1, n

               read( iounit, * ) &
     &            dummy, dummy, dummy, idummy, dreal_eigval_prim

               eigval_prim(i) = dcmplx( dreal_eigval_prim, 0.d0 )

            end do

!-----------------------------------------------------------------------
!           /*   turn back into au                                    */
!-----------------------------------------------------------------------

            factor_cminv = 1.d0 / (2*pi*speedlight_SI*au_time*100)

            do i = 1, n
               eigval_prim(i) = eigval_prim(i) / factor_cminv
            end do

!-----------------------------------------------------------------------
!           /*   compute energy corrections                           */
!-----------------------------------------------------------------------

            uctc = 0.d0
            fctc = 0.d0
            uq0c = 0.d0
            uqtc = 0.d0
            fqtc = 0.d0

            do i = 1, n

               omega = dreal(eigval_prim(i))

               if ( omega .le. 0.d0 ) then

                  uqtc = uqtc + 1.d0/beta_phonon
                  uctc = uctc + 1.d0/beta_phonon

               else

                  ho = hbar*omega

                  ho_half = ho / 2.d0

                  bho = beta_phonon*hbar*omega

                  bho_half = bho / 2.d0

                  uctc = uctc + 1.d0 / beta_phonon
                  fctc = fctc + 1.d0 / beta_phonon * log ( bho )

                  uq0c = uq0c + ho_half

!                 //  first expression
!                 uqtc = uqtc
!                      + ho_half / tanh(bho_half)

!                 //   second expression
                  uqtc = uqtc + ho_half &
     &                 + ho * exp(-bho) / (1.d0 - exp(-bho))

!                 //  first expression
!                 fqtc = fqtc &
!    &                 + 1.d0 / beta_phonon * log(2.d0*sinh(bho_half))

!                 //   second expression
                  fqtc = fqtc + ho_half &
     &                 + 1.d0 / beta_phonon * log ( 1.d0 - exp(-bho) )

               end if

            end do

!-----------------------------------------------------------------------
!           /*   compute free/internal energies and entropies         */
!-----------------------------------------------------------------------

            uct = uct + uctc
            fct = fct + fctc

            uq0 = uq0 + uq0c
            fq0 = fq0 + uq0c

            uqt = uqt + uqtc
            fqt = fqt + fqtc

!-----------------------------------------------------------------------
!           /*   end loop over wave vectors                           */
!-----------------------------------------------------------------------

         end do
         end do
         end do

!-----------------------------------------------------------------------
!        /*   average over wave vectors                               */
!-----------------------------------------------------------------------

         uc0 = uc0 / labc_phonon
         fc0 = fc0 / labc_phonon

         uct = uct / labc_phonon
         fct = fct / labc_phonon

         uq0 = uq0 / labc_phonon
         fq0 = fq0 / labc_phonon

         uqt = uqt / labc_phonon
         fqt = fqt / labc_phonon

!-----------------------------------------------------------------------
!        /*   primitive to supercell                                  */
!-----------------------------------------------------------------------

         uc0 = uc0 * nabc
         fc0 = fc0 * nabc
         sc0 = 0.d0

         uct = uct * nabc
         fct = fct * nabc
         sct = (uct - fct) / temperature_phonon

         uq0 = uq0 * nabc
         fq0 = fq0 * nabc
         sq0 = 0.d0

         uqt = uqt * nabc
         fqt = fqt * nabc
         sqt = (uqt - fqt) / temperature_phonon

!-----------------------------------------------------------------------
!        /*   close file                                              */
!-----------------------------------------------------------------------

         close( iounit )

!-----------------------------------------------------------------------
!        /*   sc0 is minus infinity                                   */
!-----------------------------------------------------------------------

         dummy = sc0

!-----------------------------------------------------------------------
!        /*   print free energies                                     */
!-----------------------------------------------------------------------

         if ( istep_phonon .eq. 1 ) then

            write( iounit_phonon, '(a)' )

            write( iounit_phonon, '(a)' ) &
     &         '========================================' // &
     &         '================================' // &
     &         '================================'

            write( iounit_phonon, '(a)' ) &
     &         '    Temp     Classical U     Classical A' // &
     &         '     Classical S       Quantum U' // &
     &         '       Quantum A       Quantum S'

            write( iounit_phonon, '(a)' ) &
     &         '----------------------------------------' // &
     &         '--------------------------------' // &
     &         '--------------------------------'

            write( iounit_phonon, '(f8.2,2f16.8,a16,3f16.8)' ) &
     &          0.d0, uc0, fc0, '        -infty', uq0, fq0, sq0

            write( iounit_phonon, '(f8.2,6f16.8)' ) &
     &         temperature_phonon, uct, fct, sct, uqt, fqt, sqt

         else

            write( iounit_phonon, '(f8.2,6f16.8)' ) &
     &         temperature_phonon, uct, fct, sct, uqt, fqt, sqt

         end if

!-----------------------------------------------------------------------
!     /*   close file: phonon frequencies                             */
!-----------------------------------------------------------------------

         close( iounit )

!-----------------------------------------------------------------------
!     /*   loop for temperatures                                      */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   close file: phonon energy                                  */
!-----------------------------------------------------------------------

      close( iounit_phonon )

      return
      end





!***********************************************************************
      subroutine phonon_dos_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, iounit, myrank

      use phonon_variables, only : &
     &   pdos, pdos_min, dpdos, npdos, eigval_prim, &
     &   lmax_phonon, labc_phonon, natom_prim, nabc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, n, la, lb, lc, idummy

!     /*   real values   */
      real(8) :: omega, dummy, dreal_eigval_prim

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   open file                                                  */
!-----------------------------------------------------------------------

      open( iounit, file = 'phonon_kdos.out' )

      read( iounit, * )
      read( iounit, * )
      read( iounit, * )
      read( iounit, * )

!-----------------------------------------------------------------------
!     /*   number of degrees of freedom                               */
!-----------------------------------------------------------------------

      n = 3*natom_prim

!-----------------------------------------------------------------------
!     /*   start loop over wave vectors                               */
!-----------------------------------------------------------------------

      do la = 1, lmax_phonon(1)
      do lb = 1, lmax_phonon(2)
      do lc = 1, lmax_phonon(3)

!-----------------------------------------------------------------------
!     /*   read frequencies                                           */
!-----------------------------------------------------------------------

      do i = 1, n
         read( iounit, * ) &
     &      dummy, dummy, dummy, idummy, dreal_eigval_prim
         eigval_prim(i) = dcmplx( dreal_eigval_prim, 0.d0 )
      end do

!-----------------------------------------------------------------------
!     /*   add to bins of frequencies                                 */
!-----------------------------------------------------------------------

      do i = 1, n

         omega = dreal(eigval_prim(i))

         if ( omega .le. 0.d0 ) then

            continue

         else

            j = nint ( ( omega - pdos_min ) / dpdos ) + 1

            if ( ( j .ge. 1 ) .and. ( j .le. npdos ) ) then
               pdos(j) = pdos(j) + 1.d0
            end if

         end if

      end do

!-----------------------------------------------------------------------
!     /*   end loop over wave vectors                                 */
!-----------------------------------------------------------------------

      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   average over wave vectors                                  */
!-----------------------------------------------------------------------

      pdos(:) = pdos(:) / labc_phonon

!-----------------------------------------------------------------------
!     /*   primitive to supercell                                     */
!-----------------------------------------------------------------------

      pdos(:) = pdos(:) * nabc

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      close( iounit )

!-----------------------------------------------------------------------
!     /*   print out                                                  */
!-----------------------------------------------------------------------

!     /*   open file   */
      open( iounit, file = 'phonon_dos.out' )

      write( iounit, '(a)' )
      write( iounit, '(a)') '========================'
      write( iounit, '(a)') '[cm**-1]      phonon dos'
      write( iounit, '(a)') '------------------------'

!     /*   loop of meshes   */
      do j = 1, npdos

!        /*   frequency at the mesh   */
         omega = pdos_min + (j-1)*dpdos

!        /*   print frequency and phonon density of states   */
         write( iounit, '(f8.2,f16.8)' ) omega, pdos(j)

!     /*   loop of meshes   */
      end do

!     /*   close file   */
      close( iounit )

      return
      end

