!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     periodic boundary condition
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine pbc_cent
!***********************************************************************

      use common_variables, only : &
     &   ux, uy, uz, box, boxinv, natom, iboundary, mbox, nbead

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i, j

      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz, xc, yc, zc

!-----------------------------------------------------------------------
!     /*   periodic boundary for path integrals                       */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( iboundary .eq. 1 ) then

         do i = 1, natom

!           /*   shift according to centroids   */

            xc = ux(i,1)
            yc = uy(i,1)
            zc = uz(i,1)

            ac = boxinv(1,1)*xc + boxinv(1,2)*yc + boxinv(1,3)*zc
            bc = boxinv(2,1)*xc + boxinv(2,2)*yc + boxinv(2,3)*zc
            cc = boxinv(3,1)*xc + boxinv(3,2)*yc + boxinv(3,3)*zc

            do j = 1, nbead
               mbox(1,i,j) = mbox(1,i,j) + nint(ac-0.5d0)
               mbox(2,i,j) = mbox(2,i,j) + nint(bc-0.5d0)
               mbox(3,i,j) = mbox(3,i,j) + nint(cc-0.5d0)
            end do

            da = - dble(nint(ac-0.5d0))
            db = - dble(nint(bc-0.5d0))
            dc = - dble(nint(cc-0.5d0))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

!           /*   shift to the range 0 < x < box   */

            ux(i,1) = xc + dx
            uy(i,1) = yc + dy
            uz(i,1) = zc + dz

         end do

      else if ( iboundary .eq. 2 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine pbc_bead
!***********************************************************************

      use common_variables, only : &
     &   ux, uy, uz, box, boxinv, natom, nbead, iboundary, mbox

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i, j

      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz, xc, yc, zc

!-----------------------------------------------------------------------
!     /*   periodic boundary                                          */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( iboundary .eq. 1 ) then

         do j = 1, nbead
         do i = 1, natom

!           /*   shift according to ux, uy, uz   */

            xc = ux(i,j)
            yc = uy(i,j)
            zc = uz(i,j)

            ac = boxinv(1,1)*xc + boxinv(1,2)*yc + boxinv(1,3)*zc
            bc = boxinv(2,1)*xc + boxinv(2,2)*yc + boxinv(2,3)*zc
            cc = boxinv(3,1)*xc + boxinv(3,2)*yc + boxinv(3,3)*zc

            mbox(1,i,j) = mbox(1,i,j) + nint(ac-0.5d0)
            mbox(2,i,j) = mbox(2,i,j) + nint(bc-0.5d0)
            mbox(3,i,j) = mbox(3,i,j) + nint(cc-0.5d0)

            da = - dble(nint(ac-0.5d0))
            db = - dble(nint(bc-0.5d0))
            dc = - dble(nint(cc-0.5d0))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

!           /*   shift to the range 0 < x < box   */

            ux(i,j) = xc + dx
            uy(i,j) = yc + dy
            uz(i,j) = zc + dz

         end do
         end do

      else if ( iboundary .eq. 2 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine pbc_xyz
!***********************************************************************

      use common_variables, only : &
     &   x, y, z, box, boxinv, natom, nbead, iboundary, mbox

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i, j

      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz, xc, yc, zc

!-----------------------------------------------------------------------
!     /*   periodic boundary in cartesian                             */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( iboundary .eq. 1 ) then

         do j = 1, nbead
         do i = 1, natom

!           /*   shift according to x, y, z   */

            xc = x(i,j)
            yc = y(i,j)
            zc = z(i,j)

            ac = boxinv(1,1)*xc + boxinv(1,2)*yc + boxinv(1,3)*zc
            bc = boxinv(2,1)*xc + boxinv(2,2)*yc + boxinv(2,3)*zc
            cc = boxinv(3,1)*xc + boxinv(3,2)*yc + boxinv(3,3)*zc

            mbox(1,i,j) = mbox(1,i,j) + nint(ac-0.5d0)
            mbox(2,i,j) = mbox(2,i,j) + nint(bc-0.5d0)
            mbox(3,i,j) = mbox(3,i,j) + nint(cc-0.5d0)

            da = - dble(nint(ac-0.5d0))
            db = - dble(nint(bc-0.5d0))
            dc = - dble(nint(cc-0.5d0))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

!           /*   shift to the range 0 < x < box   */

            x(i,j) = xc + dx
            y(i,j) = yc + dy
            z(i,j) = zc + dz

         end do
         end do

      else if ( iboundary .eq. 2 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine pbc_cons( rs, ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption
      real(8) :: rs

!-----------------------------------------------------------------------
!     /*   periodic boundary condition                                */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then
         rs = rs - 360.d0*nint(rs/360.d0)
      end if

      return
      end





!***********************************************************************
      subroutine pbc_rehmc_npt
!***********************************************************************

      use common_variables, only : &
     &   x, y, z, box, boxinv, box_bead, boxinv_bead, natom, nbead, &
     &   iboundary, mbox

      implicit none

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i, j

      real(8) :: ac, bc, cc, da, db, dc, dx, dy, dz, xc, yc, zc

!-----------------------------------------------------------------------
!     /*   periodic boundary in cartesian                             */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 0 ) then

         continue

      else if ( iboundary .eq. 1 ) then

         do j = 1, nbead

            box(:,:) = box_bead(:,:,j)
            boxinv(:,:) = boxinv_bead(:,:,j)

         do i = 1, natom

!           /*   shift according to x, y, z   */

            xc = x(i,j)
            yc = y(i,j)
            zc = z(i,j)

            ac = boxinv(1,1)*xc + boxinv(1,2)*yc + boxinv(1,3)*zc
            bc = boxinv(2,1)*xc + boxinv(2,2)*yc + boxinv(2,3)*zc
            cc = boxinv(3,1)*xc + boxinv(3,2)*yc + boxinv(3,3)*zc

            mbox(1,i,j) = mbox(1,i,j) + nint(ac-0.5d0)
            mbox(2,i,j) = mbox(2,i,j) + nint(bc-0.5d0)
            mbox(3,i,j) = mbox(3,i,j) + nint(cc-0.5d0)

            da = - dble(nint(ac-0.5d0))
            db = - dble(nint(bc-0.5d0))
            dc = - dble(nint(cc-0.5d0))

            dx = box(1,1)*da + box(1,2)*db + box(1,3)*dc
            dy = box(2,1)*da + box(2,2)*db + box(2,3)*dc
            dz = box(3,1)*da + box(3,2)*db + box(3,3)*dc

!           /*   shift to the range 0 < x < box   */

            x(i,j) = xc + dx
            y(i,j) = yc + dy
            z(i,j) = zc + dz

         end do
         end do

      else if ( iboundary .eq. 2 ) then

         continue

      end if

      return
      end

