!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, B. Thomsen
!      Last updated:    Jul 30, 2019 by M. Shiga
!      Description:     normal mode analysis
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nma
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, hessian, physmass, pi, speedlight_SI, au_time, hbar, &
     &   au_length, beta, temperature, potential, dt, au_length, pot, &
     &   fx, fy, fz, species, istep, natom, iounit, int_spec, &
     &   iprint_minfo

      use nma_variables, only : &
     &   hess, redmass, eigval, eigvec, nstep_modes

      use mech_variables, only : &
     &   nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, ij, kl, n

      real(8) :: factor, factor_cminv, sum

      integer :: n356, nvib, nimg

      real(8) :: omega_cutoff = 1.d0

      real(8) :: fcor, ecor, ecor_z, ecor_t, ecor_c, omega_cminv, omega

      character(len=8) :: char_num

      real(8) :: bohr2ang = au_length * 1.d+10

      real(8) :: rx, ry, rz, qri, dpot

      character(len=2) :: gversion = '03'

!-----------------------------------------------------------------------
!     /*   setup frozen atoms                                         */
!-----------------------------------------------------------------------

      call force_freeze_setup

!-----------------------------------------------------------------------
!     /*   make hessian matrix                                        */
!-----------------------------------------------------------------------

      call gethess_paral

!-----------------------------------------------------------------------
!     /*   symmetrized hessian matrix of one bead                     */
!-----------------------------------------------------------------------

      do i = 1, 3*natom
      do j = i, 3*natom
         hess(i,j) = 0.5d0 * ( hessian(i,j,1) + hessian(j,i,1) )
         hess(j,i) = hess(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   project out translation and rotation                       */
!-----------------------------------------------------------------------

      if ( nfreeze_mech .le. 0 ) then
         call project_out_nma( 1 )
      end if

!-----------------------------------------------------------------------
!     /*   mass weighted hessian                                      */
!-----------------------------------------------------------------------

      ij = 0

      do i = 1, natom
      do j = 1, 3

         ij = ij + 1

         kl = 0

         do k = 1, natom
         do l = 1, 3

            kl = kl + 1

            factor = 1.d0 / sqrt(physmass(i)*physmass(k))

            hess(ij,kl) = hess(ij,kl) * factor

         end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   diagonalize mass weighted hessian                          */
!-----------------------------------------------------------------------

      n = 3*natom

      call ddiag ( hess, eigval, eigvec, n )

!-----------------------------------------------------------------------
!     /*   print eigenvalues and eigenvectors                         */
!-----------------------------------------------------------------------

      open ( iounit, file = 'eigen.out' )

      do i = 1, n

         write( iounit, '(e24.16)' ) eigval(i)

         do j = 1, n, 3

            write( iounit, '(3e24.16)' ) eigvec(j+0:j+2,i)

         end do

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   print modes                                                */
!-----------------------------------------------------------------------

      call read_int1( nstep_modes, '<nstep_modes>', 13, iounit )

      if ( nstep_modes .gt. 0 ) then

         call system( 'mkdir -p modes' )

         do k = 1, 3*natom

            call int8_to_char( k, char_num )

            open ( iounit, file = 'modes/mode.' // char_num // '.xyz' )

            do istep = -nstep_modes, nstep_modes

               qri = dble(istep) * dt

               dpot =  0.5d0 * eigval(k) * qri * qri

               do j = 1, 3*natom, 3

                  i = (j+2) / 3

                  rx = qri * eigvec(j+0,k) / sqrt(physmass(i))
                  ry = qri * eigvec(j+1,k) / sqrt(physmass(i))
                  rz = qri * eigvec(j+2,k) / sqrt(physmass(i))

                  dpot = dpot - fx(i,1)*rx - fy(i,1)*ry - fz(i,1)*rz

               end do

               write( iounit, '(i8)' )  natom
               write( iounit, '(a,i8,2f16.8)' ) &
     &             "ANGSTROM", istep, qri, pot(1)+dpot

               do j = 1, 3*natom, 3

                  i = (j+2) / 3

                  rx = x(i,1) + qri * eigvec(j+0,k) / sqrt(physmass(i))
                  ry = y(i,1) + qri * eigvec(j+1,k) / sqrt(physmass(i))
                  rz = z(i,1) + qri * eigvec(j+2,k) / sqrt(physmass(i))

                  rx = rx * bohr2ang
                  ry = ry * bohr2ang
                  rz = rz * bohr2ang

                  write( iounit, '(a,3f16.8)' ) &
     &               species(i)(1:4), rx, ry, rz

               end do

            end do

            close( iounit )

         end do

      end if

!-----------------------------------------------------------------------
!     /*   convert eigenvalues to frequencies                         */
!-----------------------------------------------------------------------

      do i = 1, n

         if ( eigval(i) .gt. 0.d0 ) then
            eigval(i) = + sqrt(+eigval(i))
         else
            eigval(i) = - sqrt(-eigval(i))
         end if

      end do

!-----------------------------------------------------------------------
!     /*   turn into wave numbers [cm^-1]                             */
!-----------------------------------------------------------------------

      factor_cminv = 2.d0 * pi * speedlight_SI * au_time * 100.d0

      do i = 1, n
         eigval(i) = eigval(i) / factor_cminv
      end do

!-----------------------------------------------------------------------
!     /*   print a minfo file if requested                            */
!-----------------------------------------------------------------------

      call read_int1 ( iprint_minfo, '<iprint_minfo>', 14, iounit )

      if ( iprint_minfo .ne. -1 ) then
         call print_minfo ( 1, 0 )
      end if

!-----------------------------------------------------------------------
!     /*   normal modes                                               */
!-----------------------------------------------------------------------

!     /*   multiply mass factor   */

      k = 0

      do i = 1, natom

         k = k + 1
         eigvec(k,:) = eigvec(k,:)/sqrt(physmass(i))

         k = k + 1
         eigvec(k,:) = eigvec(k,:)/sqrt(physmass(i))

         k = k + 1
         eigvec(k,:) = eigvec(k,:)/sqrt(physmass(i))

      end do

!     /*   normalize   */

      do j = 1, n

         sum = 0.d0

         do i = 1, n
            sum = sum + eigvec(i,j)*eigvec(i,j)
         end do

         if ( sum .ne. 0.d0 ) then
            redmass(j) = 1.d0/sum/1822.88853d0
         else
            redmass(j) = 0.d0
         end if

         sum = 1.d0/sqrt(sum)

         do i = 1, n
            eigvec(i,j) = eigvec(i,j)*sum
         end do

      end do

!-----------------------------------------------------------------------
!     /*   print wavenumbers                                          */
!-----------------------------------------------------------------------

      open ( iounit, file = 'nma.out', access = 'append' )

      write( iounit, '(a)' ) &
     &   ' Entering Gaussian System'

      write( iounit, '(a)' )

      write( iounit, '(a)' ) &
     &   ' *********************************************'

      if ( gversion(1:2) .eq. 'g98' ) then
         write( iounit, '(a)' ) &
     &      ' Gaussian 98:    '
      else if ( gversion(1:2) .eq. 'g03' ) then
         write( iounit, '(a)' ) &
     &      ' Gaussian 03:    '
      else if ( gversion(1:2) .eq. 'g09' ) then
         write( iounit, '(a)' ) &
     &      ' Gaussian 09:    '
      else if ( gversion(1:2) .eq. 'g16' ) then
         write( iounit, '(a)' ) &
     &      ' Gaussian 16:    '
      end if

      write( iounit, '(a)' )
      write( iounit, '(a)' ) &
     &   ' *********************************************'

      write( iounit, '(a)' )
      write( iounit, '(a)' )

      write( iounit, '(25x,a)' ) 'Standard orientation:'
      write( iounit, '(a)' ) ' ----------------------------' // &
     &               '-----------------------------------------'
      write( iounit, '(a)' ) ' Center     Atomic     Atomic' // &
     &               '              Coordinates (Angstroms)'
      write( iounit, '(a)' ) ' Number     Number      Type ' // &
     &               '             X           Y           Z'
      write( iounit, '(a)' ) ' ----------------------------' // &
     &               '-----------------------------------------'

      do i = 1, natom
         factor = au_length/1.d-10
         write( iounit, '( i5,6x,i5,9x,i5,4x,3f12.6)' ) &
     &   i, int_spec(i), 0, x(i,1)*factor, y(i,1)*factor, z(i,1)*factor
      end do
      write( iounit, '(a)' ) ' ----------------------------' // &
     &               '-----------------------------------------'

      write( iounit, '(a)' )
      write( iounit, '(a)' )
      write( iounit, '(a)' )

      if ( gversion(1:2) .eq. 'g98' ) then
         write( iounit, '(a)' ) &
     &      ' Harmonic frequencies (cm**-1), IR intensities (KM/Mole),'
         write( iounit, '(a)' ) &
     &      ' Raman scattering activities (A**4/AMU),' // &
     &      ' Raman depolarization ratios,'
         write( iounit, '(a)' ) &
     &      ' reduced masses (AMU), force constants (mDyne/A)' // &
     &      ' and normal coordinates:'
      else
         write( iounit, '(a)' ) &
     &      ' Harmonic frequencies (cm**-1), IR intensities' // &
     &      ' (KM/Mole), Raman scattering'
         write( iounit, '(a)' ) &
     &      ' activities (A**4/AMU), depolarization ratios for' // &
     &      ' plane and unpolarized'
         write( iounit, '(a)' ) &
     &      ' incident light, reduced masses (AMU), force constants' // &
     &      ' (mDyne/A),'
         write( iounit, '(a)' ) &
     &      ' and normal coordinates:'
      end if

      do j = 1, n, 3

         write( iounit, '( 14x,i8,15x,i8,15x,i8)' ) j, j+1, j+2
         write( iounit, '(20x,a2,21x,a2,21x,a2)' ) '?A', '?A', '?A'
         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)') &
     &      ' Frequencies --', eigval(j), eigval(j+1), eigval(j+2)
         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)' ) &
     &      ' Red. masses --', redmass(j), redmass(j+1), redmass(j+2)
         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)' ) &
     &      ' Frc consts  --', 0.d0, 0.d0, 0.d0
         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)' ) &
     &      ' IR inten    --', 0.d0, 0.d0, 0.d0
!         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)' ) &
!     &      ' Raman Activ --', 0.d0, 0.d0, 0.d0
!         write( iounit, '(a,f11.4,12x,f11.4,12x,f11.4)' ) &
!     &      ' Depolar     --', 0.d0, 0.d0, 0.d0
         write( iounit, '(a)' ) ' Atom AN      X      Y      Z' // &
     &                        '        X      Y      Z' // &
     &                        '        X      Y      Z'

         k = 1

         do i = 1, natom

            write( iounit, '(2i4,2x,3f7.2,2x,3f7.2,2x,3f7.2)' ) &
     &         i, int_spec(i), &
     &         eigvec(k,j),   eigvec(k+1,j),   eigvec(k+2,j), &
     &         eigvec(k,j+1), eigvec(k+1,j+1), eigvec(k+2,j+1), &
     &         eigvec(k,j+2), eigvec(k+1,j+2), eigvec(k+2,j+2)

            k = k + 3

         end do

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   calculate harmonic correction                              */
!-----------------------------------------------------------------------

!     //   imaginary frequencies
      nimg = 0

!     //   translational and rotational degrees of freedom
      n356 = 0

!     //   zero point correction (quantum)
      ecor_z = 0.d0

!     //   thermal excitation correction (quantum)
      ecor_t = 0.d0

!     //   translational and rotational corrections (classical)
      ecor_c = 0.d0

!     //   free energy correction (quantum)
      fcor = 0.d0

!     //   loop of modes
      do j = 1, 3*natom

!        //   frequency
         omega_cminv = eigval(j)

         if      ( omega_cminv .le. -omega_cutoff ) then

            nimg = nimg + 1

         else if ( abs(omega_cminv) .le. +omega_cutoff ) then

            n356 = n356 + 1

            ecor_c = ecor_c + 0.5d0 / beta

            fcor = fcor +  0.5d0 / beta

         else

            omega = omega_cminv * factor_cminv

            ecor_z = ecor_z + 0.5d0*hbar*omega

!            ecor_t = ecor_t + 0.5d0*hbar*omega
!     &             * ( 1.d0/tanh(0.5d0*beta*hbar*omega) - 1.d0 )

            ecor_t = ecor_t + 0.5d0*hbar*omega &
     &             + hbar*omega * exp(-beta*hbar*omega) &
     &             / (1.d0 - exp(-beta*hbar*omega))

            fcor = fcor + 0.5d0*hbar*omega &
     &           + 1.d0 / beta * log ( 1.d0 - exp(-beta*hbar*omega) )

         end if

!     //   loop of modes
      end do

!     //   vibrational degrees of freedom
      nvib = 3*natom - n356 - nimg

!     //   total energy correction
      ecor = ecor_c + ecor_t

!-----------------------------------------------------------------------
!     /*   print energies                                             */
!-----------------------------------------------------------------------

      write( 6, '(a,i8)' ) &
     &   'Imaginary frequency modes:                 ', nimg

      write( 6, '(a,i8)' ) &
     &   'Translational, rotational, frozen modes:   ', n356

      write( 6, '(a,i8)' ) &
     &   'Real vibrational modes:                    ', nvib

      write( 6, '(a,f16.5)' ) &
     &   'Temperature:                       ', temperature

      write( 6, '(a,f16.5)' ) &
     &   'Potential energy at minimum (au):  ', potential

      write( 6, '(a,f16.5)' ) &
     &   'Quantum vib zero point energy (au):', ecor_z

      write( 6, '(a,f16.5)' ) &
     &   'Thermal vib excitation energy (au):', ecor_t-ecor_z

      write( 6, '(a,f16.5)' ) &
     &   'Thermal trans and rot energy (au): ', ecor_c

      write( 6, '(a,f16.5)' ) &
     &   'Total energy correction (au):      ', ecor

      write( 6, '(a,f16.5)' ) &
     &   'Entropy correction -TS (au):       ', fcor-ecor

      write( 6, '(a,f16.5)' ) &
     &   'Free energy correction (au):       ', fcor

      write( 6, '(a,f16.5)' ) &
     &   'Potential energy (au):             ', potential

      write( 6, '(a,f16.5)' ) &
     &   'Total internal energy (au):        ', potential+ecor

      write( 6, '(a,f16.5)' ) &
     &   'Total free energy (au):            ', potential+fcor

      return
      end
