!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     extensive MPI parallelization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_velocity_mode_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, beta, vux, vuy, vuz, ensemble, &
     &   itrans_start, irot_start, iboundary, natom, nbead, myrank, &
     &   irandom, iounit, myrank

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      integer :: imode = 0

      integer, save :: iset = 0

      real(8) :: gasdev, vsigma, ekin_cor, gasdev_XMPI

      real(8) :: px(natom,nbead), py(natom,nbead), pz(natom,nbead)

!-----------------------------------------------------------------------
!     /*   random number                                              */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   random number   */
         call read_int1_MPI ( irandom, '<irandom>', 9, iounit )

!        /*   randomize seed for all processes   */
         if ( irandom .ne. 0 ) call my_random_seed_XMPI( myrank )

         iset  = 1

      end if

!-----------------------------------------------------------------------
!     /*   ensemble                                                   */
!-----------------------------------------------------------------------

      if ( ensemble(1:3) .eq. 'NVE' ) imode = 2
      if ( ensemble(1:3) .eq. 'NVT' ) imode = 1

!-----------------------------------------------------------------------
!     /*   old kinetic energy                                         */
!-----------------------------------------------------------------------

      ekin_cor = 0.d0

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j) &
     &      - 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &      - 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   sample velocity                                            */
!-----------------------------------------------------------------------

      if ( irandom .eq. 0 ) then

         if ( myrank .eq. 0 ) then

            do j = 1, nbead

               if ( j .lt. imode ) then

                  do i = 1, natom
                     px(i,j) = 0.d0
                     py(i,j) = 0.d0
                     pz(i,j) = 0.d0
                  end do

               else

                  do i = 1, natom
                     px(i,j) = gasdev()
                     py(i,j) = gasdev()
                     pz(i,j) = gasdev()
                  end do

               end if

            end do

         end if

         call my_mpi_bcast_real_2( px, natom, nbead )
         call my_mpi_bcast_real_2( py, natom, nbead )
         call my_mpi_bcast_real_2( pz, natom, nbead )

         do j = jstart_bead, jend_bead

            if ( j .lt. imode ) cycle

            do i = jstart_atom, jend_atom

               vsigma = sqrt( 1.d0/beta/fictmass(i,j) )

               vux(i,j) = vsigma*px(i,j)
               vuy(i,j) = vsigma*py(i,j)
               vuz(i,j) = vsigma*pz(i,j)

            end do

         end do

      else

         do j = jstart_bead, jend_bead

            if ( j .lt. imode ) cycle

            do i = jstart_atom, jend_atom

               vsigma = sqrt( 1.d0/beta/fictmass(i,j) )

               vux(i,j) = vsigma*gasdev_XMPI()
               vuy(i,j) = vsigma*gasdev_XMPI()
               vuz(i,j) = vsigma*gasdev_XMPI()

            end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent_XMPI
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent_XMPI

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent_XMPI
         if ( irot_start .eq. 2 ) call subtract_rotation_cent_XMPI
      end if

!-----------------------------------------------------------------------
!     /*   new kinetic energy                                         */
!-----------------------------------------------------------------------

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j) &
     &      + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j) &
     &      + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)

      end do
      end do

      call my_mpi_allreduce_real_0( ekin_cor )
!      call my_mpi_allreduce_real_0_sub( ekin_cor )
!      call my_mpi_allreduce_real_0_main( ekin_cor )

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end
