!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     randomize normal mode velocities
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_velocity_mode_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, beta, vux, vuy, vuz, ensemble, &
     &   natom, nbead, itrans_start, irot_start, iboundary, myrank, &
     &   myrank_main, myrank_sub, nprocs_main, nprocs_sub

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      integer :: iparal = 0
      integer :: imode = 0

      real(8) :: gasdev, vsigma, ekin_cor

!-----------------------------------------------------------------------
!     /*   ensemble                                                   */
!-----------------------------------------------------------------------

      if ( ensemble(1:3) .eq. 'NVE' ) imode = 2
      if ( ensemble(1:3) .eq. 'NVT' ) imode = 1

!-----------------------------------------------------------------------
!     /*   old kinetic energy                                         */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         ekin_cor = 0.d0

         do i = 1, nbead
         do j = 1, natom

            ekin_cor = ekin_cor &
     &         - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &         - 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &         - 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

         end do
         end do

      end if

      call my_mpi_bcast_real_0 ( ekin_cor )

!-----------------------------------------------------------------------
!     /*   sample velocity                                            */
!-----------------------------------------------------------------------

      if ( iparal .eq. 0 ) then

         if ( myrank .eq. 0 ) then

            do i = imode, nbead
            do j = 1, natom

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do
            end do

         end if

         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )

      else if ( iparal .eq. 1 ) then

         vux(:,:) = 0.d0
         vuy(:,:) = 0.d0
         vuz(:,:) = 0.d0

         do i = imode, nbead

            if ( mod( i-1, nprocs_main ) .ne. myrank_main ) cycle

            do j = 1, natom

               if ( mod( j-1, nprocs_sub ) .ne. myrank_sub ) cycle

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do

         end do

         call my_mpi_allreduce_real_2 ( vux, natom, nbead )
         call my_mpi_allreduce_real_2 ( vuy, natom, nbead )
         call my_mpi_allreduce_real_2 ( vuz, natom, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent
      if ( itrans_start .eq. 3 ) call subtract_velocity_modes
      if ( itrans_start .eq. 4 ) call subtract_velocity_modes
      if ( itrans_start .eq. 5 ) call subtract_velocity_all
      if ( itrans_start .eq. 6 ) call subtract_velocity_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent_MPI
         if ( irot_start .eq. 2 ) call subtract_rotation_cent_MPI
         if ( irot_start .eq. 3 ) call subtract_rotation_modes_MPI
         if ( irot_start .eq. 4 ) call subtract_rotation_modes_MPI
         if ( irot_start .eq. 5 ) call subtract_rotation_mix_MPI
         if ( irot_start .eq. 6 ) call subtract_rotation_mix_MPI
      end if

!-----------------------------------------------------------------------
!     /*   new kinetic energy                                         */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         do i = 1, nbead
         do j = 1, natom

            ekin_cor = ekin_cor &
     &         + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &         + 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &         + 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

         end do
         end do

      end if

      call my_mpi_bcast_real_0 ( ekin_cor )

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end





!***********************************************************************
      subroutine nm_velocity_mode_gillespie_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, vux, vuy, vuz, dt, beta, tau_bcmd, &
     &   natom, nbead, ensemble, itrans_start, irot_start, iboundary, &
     &   myrank, myrank_main, myrank_sub, nprocs_main, nprocs_sub

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      integer :: iparal = 0
      integer :: imode = 0

      real(8) :: gasdev, vsigma, ekin_cor, g3

!-----------------------------------------------------------------------
!     /*   ensemble                                                   */
!-----------------------------------------------------------------------

      if ( ensemble(1:3) .eq. 'NVE' ) imode = 2
      if ( ensemble(1:3) .eq. 'NVT' ) imode = 1

!-----------------------------------------------------------------------
!     /*   old kinetic energy                                         */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         ekin_cor = 0.d0

         do i = 1, nbead
         do j = 1, natom

            ekin_cor = ekin_cor &
     &         - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &         - 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &         - 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

         end do
         end do

      end if

      call my_mpi_bcast_real_0 ( ekin_cor )

!-----------------------------------------------------------------------
!     /*   sample velocity                                            */
!-----------------------------------------------------------------------

      g3 = ( 1.d0 - exp(-2.d0*dt/tau_bcmd) ) / (2.d0*dt/tau_bcmd)

      if ( iparal .eq. 0 ) then

         if ( myrank .eq. 0 ) then

            do i = imode, 1
            do j = 1, natom

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do
            end do

            do i = imode, nbead
            do j = 1, natom

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) * g3 )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do
            end do

         end if

         call my_mpi_bcast_real_2 ( vux, natom, nbead )
         call my_mpi_bcast_real_2 ( vuy, natom, nbead )
         call my_mpi_bcast_real_2 ( vuz, natom, nbead )

      else if ( iparal .eq. 1 ) then

         vux(:,:) = 0.d0
         vuy(:,:) = 0.d0
         vuz(:,:) = 0.d0

         do i = imode, 1

            if ( mod( i-1, nprocs_main ) .ne. myrank_main ) cycle

            do j = 1, natom

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do

         end do

         do i = imode, nbead

            if ( mod( i-1, nprocs_main ) .ne. myrank_main ) cycle

            do j = 1, natom

               if ( mod( j-1, nprocs_sub ) .ne. myrank_sub ) cycle

               vsigma = sqrt( 1.d0/beta/fictmass(j,i) * g3 )

               vux(j,i) = vsigma*gasdev()
               vuy(j,i) = vsigma*gasdev()
               vuz(j,i) = vsigma*gasdev()

            end do

         end do

         call my_mpi_allreduce_real_2 ( vux, natom, nbead )
         call my_mpi_allreduce_real_2 ( vuy, natom, nbead )
         call my_mpi_allreduce_real_2 ( vuz, natom, nbead )

      end if

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent
      if ( itrans_start .eq. 3 ) call subtract_velocity_modes
      if ( itrans_start .eq. 4 ) call subtract_velocity_modes
      if ( itrans_start .eq. 5 ) call subtract_velocity_all
      if ( itrans_start .eq. 6 ) call subtract_velocity_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent_MPI
         if ( irot_start .eq. 2 ) call subtract_rotation_cent_MPI
         if ( irot_start .eq. 3 ) call subtract_rotation_modes_MPI
         if ( irot_start .eq. 4 ) call subtract_rotation_modes_MPI
         if ( irot_start .eq. 5 ) call subtract_rotation_mix_MPI
         if ( irot_start .eq. 6 ) call subtract_rotation_mix_MPI
      end if

!-----------------------------------------------------------------------
!     /*   new kinetic energy                                         */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         do i = 1, nbead
         do j = 1, natom

            ekin_cor = ekin_cor &
     &         + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &         + 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &         + 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

         end do
         end do

      end if

      call my_mpi_bcast_real_0 ( ekin_cor )

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end
