!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     randomize normal mode velocities
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_velocity_mode
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, beta, vux, vuy, vuz, natom, nbead, &
     &   ensemble, itrans_start, irot_start, iboundary

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      integer :: imode = 0

      real(8) :: gasdev, vsigma, ekin_cor

!-----------------------------------------------------------------------
!     /*   ensemble                                                   */
!-----------------------------------------------------------------------

      if ( ensemble(1:3) .eq. 'NVE' ) imode = 2
      if ( ensemble(1:3) .eq. 'NVT' ) imode = 1

!-----------------------------------------------------------------------
!     /*   old kinetic energy                                         */
!-----------------------------------------------------------------------

      ekin_cor = 0.d0

      do i = 1, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   sample velocity                                            */
!-----------------------------------------------------------------------

      do i = imode, nbead
      do j = 1, natom

         vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

         vux(j,i) = vsigma*gasdev()
         vuy(j,i) = vsigma*gasdev()
         vuz(j,i) = vsigma*gasdev()

      end do
      end do

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent
      if ( itrans_start .eq. 3 ) call subtract_velocity_modes
      if ( itrans_start .eq. 4 ) call subtract_velocity_modes
      if ( itrans_start .eq. 5 ) call subtract_velocity_all
      if ( itrans_start .eq. 6 ) call subtract_velocity_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent
         if ( irot_start .eq. 2 ) call subtract_rotation_cent
         if ( irot_start .eq. 3 ) call subtract_rotation_modes
         if ( irot_start .eq. 4 ) call subtract_rotation_modes
         if ( irot_start .eq. 5 ) call subtract_rotation_mix
         if ( irot_start .eq. 6 ) call subtract_rotation_mix
      end if

!-----------------------------------------------------------------------
!     /*   new kinetic energy                                         */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end





!***********************************************************************
      subroutine nm_velocity_mode_gillespie
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian_cor, fictmass, vux, vuy, vuz, dt, beta, tau_bcmd, &
     &   natom, nbead, ensemble, itrans_start, irot_start, iboundary

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      integer :: imode = 0

      real(8) :: gasdev, vsigma, ekin_cor, g3

!-----------------------------------------------------------------------
!     /*   ensemble                                                   */
!-----------------------------------------------------------------------

      if ( ensemble(1:3) .eq. 'NVE' ) imode = 2
      if ( ensemble(1:3) .eq. 'NVT' ) imode = 1

!-----------------------------------------------------------------------
!     /*   old kinetic energy                                         */
!-----------------------------------------------------------------------

      ekin_cor = 0.d0

      do i = 1, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   sample velocity                                            */
!-----------------------------------------------------------------------

      g3 = ( 1.d0 - exp(-2.d0*dt/tau_bcmd) ) / (2.d0*dt/tau_bcmd)

      do i = imode, 1
      do j = 1, natom

         vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

         vux(j,i) = vsigma*gasdev()
         vuy(j,i) = vsigma*gasdev()
         vuz(j,i) = vsigma*gasdev()

      end do
      end do

      do i = imode, nbead
      do j = 1, natom

         vsigma = sqrt( 1.d0/beta/fictmass(j,i) * g3 )

         vux(j,i) = vsigma*gasdev()
         vuy(j,i) = vsigma*gasdev()
         vuz(j,i) = vsigma*gasdev()

      end do
      end do

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent
      if ( itrans_start .eq. 3 ) call subtract_velocity_modes
      if ( itrans_start .eq. 4 ) call subtract_velocity_modes
      if ( itrans_start .eq. 5 ) call subtract_velocity_all
      if ( itrans_start .eq. 6 ) call subtract_velocity_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent
         if ( irot_start .eq. 2 ) call subtract_rotation_cent
         if ( irot_start .eq. 3 ) call subtract_rotation_modes
         if ( irot_start .eq. 4 ) call subtract_rotation_modes
         if ( irot_start .eq. 5 ) call subtract_rotation_mix
         if ( irot_start .eq. 6 ) call subtract_rotation_mix
      end if

!-----------------------------------------------------------------------
!     /*   new kinetic energy                                         */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end
