!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     normal modes in Onsager-Machlup action
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_trans_om(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!
!     itrans = 1 :   backward transformation
!                    u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, tnm, tnminv, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            x(k,i) = 0.d0
            y(k,i) = 0.d0
            z(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            x(k,i) = x(k,i) + tnm(i,j)*ux(k,j)
            y(k,i) = y(k,i) + tnm(i,j)*uy(k,j)
            z(k,i) = z(k,i) + tnm(i,j)*uz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            ux(k,i) = 0.d0
            uy(k,i) = 0.d0
            uz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            ux(k,i) = ux(k,i) + tnminv(i,j)*x(k,j)
            uy(k,i) = uy(k,i) + tnminv(i,j)*y(k,j)
            uz(k,i) = uz(k,i) + tnminv(i,j)*z(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_velocity_om(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!
!     itrans = 1 :   backward transformation
!                    u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, vux, vuy, vuz, tnm, tnminv, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            vx(k,i) = 0.d0
            vy(k,i) = 0.d0
            vz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            vx(k,i) = vx(k,i) + tnm(i,j)*vux(k,j)
            vy(k,i) = vy(k,i) + tnm(i,j)*vuy(k,j)
            vz(k,i) = vz(k,i) + tnm(i,j)*vuz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            vux(k,i) = 0.d0
            vuy(k,i) = 0.d0
            vuz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            vux(k,i) = vux(k,i) + tnminv(i,j)*vx(k,j)
            vuy(k,i) = vuy(k,i) + tnminv(i,j)*vy(k,j)
            vuz(k,i) = vuz(k,i) + tnminv(i,j)*vz(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_om(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   tnm, tnminv, natom, nbead

      use om_variables, only : &
     &   fx_om, fy_om, fz_om, fux_om, fuy_om, fuz_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            fx_om(k,i) = 0.d0
            fy_om(k,i) = 0.d0
            fz_om(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fx_om(k,i) = fx_om(k,i) + fux_om(k,j)*tnminv(j,i)
            fy_om(k,i) = fy_om(k,i) + fuy_om(k,j)*tnminv(j,i)
            fz_om(k,i) = fz_om(k,i) + fuz_om(k,j)*tnminv(j,i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            fux_om(k,i) = 0.d0
            fuy_om(k,i) = 0.d0
            fuz_om(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fux_om(k,i) = fux_om(k,i) + fx_om(k,j)*tnm(j,i)
            fuy_om(k,i) = fuy_om(k,i) + fy_om(k,j)*tnm(j,i)
            fuz_om(k,i) = fuz_om(k,i) + fz_om(k,j)*tnm(j,i)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_ref_om(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   tnm, tnminv, natom, nbead

      use om_variables, only : &
     &   fx_ref_om, fy_ref_om, fz_ref_om, fux_ref_om, fuy_ref_om, &
     &   fuz_ref_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            fx_ref_om(k,i) = 0.d0
            fy_ref_om(k,i) = 0.d0
            fz_ref_om(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fx_ref_om(k,i) = fx_ref_om(k,i) &
     &         + fux_ref_om(k,j)*tnminv(j,i)
            fy_ref_om(k,i) = fy_ref_om(k,i) &
     &         + fuy_ref_om(k,j)*tnminv(j,i)
            fz_ref_om(k,i) = fz_ref_om(k,i) &
     &         + fuz_ref_om(k,j)*tnminv(j,i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            fux_ref_om(k,i) = 0.d0
            fuy_ref_om(k,i) = 0.d0
            fuz_ref_om(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fux_ref_om(k,i) = fux_ref_om(k,i) + fx_ref_om(k,j)*tnm(j,i)
            fuy_ref_om(k,i) = fuy_ref_om(k,i) + fy_ref_om(k,j)*tnm(j,i)
            fuz_ref_om(k,i) = fuz_ref_om(k,i) + fz_ref_om(k,j)*tnm(j,i)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_matrix_om_MPI
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   tnm, tnminv, dnmmass, physmass, pi, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: u2, pm

      real(8), dimension(nbead,nbead) :: a, c, d, ainv, cinv

      real(8), dimension(nbead) :: b

!     /*   ioption = 0: numerical, = 1: analytical   */
      integer :: ioption = 1

!-----------------------------------------------------------------------
!     /*   make a matrix                                              */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do i = 1, nbead
         a(i,i) = 1.d0
      end do

      do i = 2, nbead-1
         a(i,1) = a(i,1) - dble(nbead-i)/dble(nbead-1)
      end do

      do i = 2, nbead-1
         a(i,nbead) = a(i,nbead) - dble(i-1)/dble(nbead-1)
      end do

!-----------------------------------------------------------------------
!     /*   inverse of a matrix                                        */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call inv_MPI( a, ainv, nbead )

      else

         ainv(:,:) = 0.d0

         do i = 1, nbead
            ainv(i,i) = 1.d0
         end do

         do i = 2, nbead-1
            ainv(i,1) = ainv(i,1) + dble(nbead-i)/dble(nbead-1)
         end do

         do i = 2, nbead-1
            ainv(i,nbead) = ainv(i,nbead) + dble(i-1)/dble(nbead-1)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   eigenvector of a matrix                                    */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do j = 1, nbead
         do i = 1, nbead
            d(i,j) = 0.d0
         end do
         end do

         d(1,1) = + 0.d0
         d(2,2) = + 2.d0
         d(2,3) = - 1.d0

         do i = 3, nbead-2
            d(i,i-1) = - 1.d0
            d(i,i)   = + 2.d0
            d(i,i+1) = - 1.d0
         end do

         d(nbead-1,nbead-2) = - 1.d0
         d(nbead-1,nbead-1) = + 2.d0
         d(nbead,nbead)     = + 4.d0

         call ddiag_MPI( d, b, c, nbead )

         do j = 2, nbead-1
            pm = sign(1.d0,c(2,j))
            do i = 2, nbead-1
               c(i,j) = c(i,j) * pm * sqrt(dble(nbead))
            end do
         end do

      else

         c(:,:)         = 0.d0
         c(1,1)         = 1.d0
         c(nbead,nbead) = 1.d0

         do j = 2, nbead-1
            u2 = 0.d0
            do i = 2, nbead-1
               c(i,j) = sin(dble(i-1)*pi*dble(j-1)/dble(nbead-1))
               u2 = u2 + c(i,j)*c(i,j)
            end do
!           /*   1.d0/sqrt(u2) = sqrt(2.d0/dble(nbead-1))   */
            do i = 2, nbead-1
               c(i,j) = c(i,j) / sqrt(u2) * sqrt(dble(nbead))
            end do
         end do

         do i = 1, nbead
            b(i) = 2.d0 - 2.d0*cos(pi*dble(i-1)/dble(nbead-1))
         end do

      end if

!-----------------------------------------------------------------------
!     /*   transform matrix                                           */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, nbead
         tnm(i,j) = 0.d0
         do k = 1, nbead
            tnm(i,j) = tnm(i,j) + ainv(i,k)*c(k,j)
         end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   inverse of transform matrix                                */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         call inv_MPI( tnm, tnminv, nbead )

      else

         cinv(:,:)         = 0.d0
         cinv(1,1)         = 1.d0
         cinv(nbead,nbead) = 1.d0

         do j = 2, nbead-1
         do i = 2, nbead-1
            cinv(i,j) = c(j,i) / dble(nbead)
         end do
         end do

         do i = 1, nbead
         do j = 1, nbead
            tnminv(i,j) = 0.d0
            do k = 1, nbead
               tnminv(i,j) = tnminv(i,j) + cinv(i,k)*a(k,j)
            end do
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   make dnmmass                                               */
!-----------------------------------------------------------------------

!     /*   normal mode mass   */

      do j = 1, natom
         dnmmass(j,1) = physmass(j)
         do i = 2, nbead-1
            dnmmass(j,i) = b(i)*physmass(j)*dble(nbead)
         end do
         dnmmass(j,nbead) = physmass(j)
      end do

      return
      end
