!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     normal mode transformations
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_trans_XMPI( itrans )
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!
!     itrans = 1 :   backward transformation
!                    u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, tnm, tnminv, nbead

      use XMPI_variables, only : &
     &   jstart_atom, jend_atom, jstart_bead, jend_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         call my_mpi_bcast_xyz_XMPI  ( ux, uy, uz, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            x(k,i) = 0.d0
            y(k,i) = 0.d0
            z(k,i) = 0.d0
            do j = 1, nbead
               x(k,i) = x(k,i) + tnm(i,j)*ux(k,j)
               y(k,i) = y(k,i) + tnm(i,j)*uy(k,j)
               z(k,i) = z(k,i) + tnm(i,j)*uz(k,j)
            end do
            end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         call my_mpi_bcast_xyz_XMPI  ( x, y, z, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            ux(k,i) = 0.d0
            uy(k,i) = 0.d0
            uz(k,i) = 0.d0
            do j = 1, nbead
               ux(k,i) = ux(k,i) + tnminv(i,j)*x(k,j)
               uy(k,i) = uy(k,i) + tnminv(i,j)*y(k,j)
               uz(k,i) = uz(k,i) + tnminv(i,j)*z(k,j)
            end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_velocity_XMPI( itrans )
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    vr(i) = vr(i) + sum_j tnm(i,j)*vu(j)
!
!     itrans = 1 :   backward transformation
!                    vu(i) = vu(i) + sum_j tnminv(i,j)*vr(j)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, vux, vuy, vuz, tnm, tnminv, nbead

      use XMPI_variables, only : &
     &   jstart_atom, jend_atom, jstart_bead, jend_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         call my_mpi_bcast_xyz_XMPI  ( vux, vuy, vuz, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            vx(k,i) = 0.d0
            vy(k,i) = 0.d0
            vz(k,i) = 0.d0
            do j = 1, nbead
               vx(k,i) = vx(k,i) + tnm(i,j)*vux(k,j)
               vy(k,i) = vy(k,i) + tnm(i,j)*vuy(k,j)
               vz(k,i) = vz(k,i) + tnm(i,j)*vuz(k,j)
            end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         call my_mpi_bcast_xyz_XMPI  ( vx, vy, vz, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            vux(k,i) = 0.d0
            vuy(k,i) = 0.d0
            vuz(k,i) = 0.d0
            do j = 1, nbead
               vux(k,i) = vux(k,i) + tnminv(i,j)*vx(k,j)
               vuy(k,i) = vuy(k,i) + tnminv(i,j)*vy(k,j)
               vuz(k,i) = vuz(k,i) + tnminv(i,j)*vz(k,j)
            end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_XMPI( itrans )
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, fux, fuy, fuz, tnm, tnminv, nbead

      use XMPI_variables, only : &
     &   jstart_atom, jend_atom, jstart_bead, jend_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         call my_mpi_bcast_xyz_XMPI  ( fux, fuy, fuz, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            fx(k,i) = 0.d0
            fy(k,i) = 0.d0
            fz(k,i) = 0.d0
            do j = 1, nbead
               fx(k,i) = fx(k,i) + fux(k,j)*tnminv(j,i)
               fy(k,i) = fy(k,i) + fuy(k,j)*tnminv(j,i)
               fz(k,i) = fz(k,i) + fuz(k,j)*tnminv(j,i)
            end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         call my_mpi_bcast_xyz_XMPI  ( fx, fy, fz, 2 )

         do k = jstart_atom, jend_atom
         do i = jstart_bead, jend_bead
            fux(k,i) = 0.d0
            fuy(k,i) = 0.d0
            fuz(k,i) = 0.d0
            do j = 1, nbead
               fux(k,i) = fux(k,i) + fx(k,j)*tnm(j,i)
               fuy(k,i) = fuy(k,i) + fy(k,j)*tnm(j,i)
               fuz(k,i) = fuz(k,i) + fz(k,j)*tnm(j,i)
            end do
         end do
         end do

      end if

      return
      end
