!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     normal modes for path integrals
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_trans(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!
!     itrans = 1 :   backward transformation
!                    u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, tnm, tnminv, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            x(k,i) = 0.d0
            y(k,i) = 0.d0
            z(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            x(k,i) = x(k,i) + tnm(i,j)*ux(k,j)
            y(k,i) = y(k,i) + tnm(i,j)*uy(k,j)
            z(k,i) = z(k,i) + tnm(i,j)*uz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            ux(k,i) = 0.d0
            uy(k,i) = 0.d0
            uz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            ux(k,i) = ux(k,i) + tnminv(i,j)*x(k,j)
            uy(k,i) = uy(k,i) + tnminv(i,j)*y(k,j)
            uz(k,i) = uz(k,i) + tnminv(i,j)*z(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_velocity(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    vr(i) = vr(i) + sum_j tnm(i,j)*vu(j)
!
!     itrans = 1 :   backward transformation
!                    vu(i) = vu(i) + sum_j tnminv(i,j)*vr(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, vux, vuy, vuz, tnm, tnminv, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            vx(k,i) = 0.d0
            vy(k,i) = 0.d0
            vz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            vx(k,i) = vx(k,i) + tnm(i,j)*vux(k,j)
            vy(k,i) = vy(k,i) + tnm(i,j)*vuy(k,j)
            vz(k,i) = vz(k,i) + tnm(i,j)*vuz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            vux(k,i) = 0.d0
            vuy(k,i) = 0.d0
            vuz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            vux(k,i) = vux(k,i) + tnminv(i,j)*vx(k,j)
            vuy(k,i) = vuy(k,i) + tnminv(i,j)*vy(k,j)
            vuz(k,i) = vuz(k,i) + tnminv(i,j)*vz(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, fux, fuy, fuz, tnm, tnminv, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            fx(k,i) = 0.d0
            fy(k,i) = 0.d0
            fz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fx(k,i) = fx(k,i) + fux(k,j)*tnminv(j,i)
            fy(k,i) = fy(k,i) + fuy(k,j)*tnminv(j,i)
            fz(k,i) = fz(k,i) + fuz(k,j)*tnminv(j,i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            fux(k,i) = 0.d0
            fuy(k,i) = 0.d0
            fuz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fux(k,i) = fux(k,i) + fx(k,j)*tnm(j,i)
            fuy(k,i) = fuy(k,i) + fy(k,j)*tnm(j,i)
            fuz(k,i) = fuz(k,i) + fz(k,j)*tnm(j,i)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_best(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   tnm, tnminv, natom, nbead

      use best_variables, only : &
     &   fx_best, fy_best, fz_best, fux_best, fuy_best, fuz_best

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            fx_best(k,i) = 0.d0
            fy_best(k,i) = 0.d0
            fz_best(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fx_best(k,i) = fx_best(k,i) + fux_best(k,j)*tnminv(j,i)
            fy_best(k,i) = fy_best(k,i) + fuy_best(k,j)*tnminv(j,i)
            fz_best(k,i) = fz_best(k,i) + fuz_best(k,j)*tnminv(j,i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, nbead
         do k = 1, natom
            fux_best(k,i) = 0.d0
            fuy_best(k,i) = 0.d0
            fuz_best(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, nbead
            fux_best(k,i) = fux_best(k,i) + fx_best(k,j)*tnm(j,i)
            fuy_best(k,i) = fuy_best(k,i) + fy_best(k,j)*tnm(j,i)
            fuz_best(k,i) = fuz_best(k,i) + fz_best(k,j)*tnm(j,i)
         end do
         end do
         end do

      end if

      return
      end

