!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     normal modes for path integrals
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine nm_matrix
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead

      implicit none

!-----------------------------------------------------------------------
!     /*   use different routines for odd and even nbead numbers      */
!-----------------------------------------------------------------------

      if ( nbead .eq. 1 ) then

         call nm_matrix_classical

      else if ( mod(nbead,2) .eq. 0 ) then

         call nm_matrix_even

      else if ( mod(nbead,2) .eq. 1 ) then

         call nm_matrix_odd

      else

         call nm_matrix_diag

      end if

      return
      end





!***********************************************************************
      subroutine nm_matrix_classical
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   u, uinv, tnm, tnminv, dnmmass, physmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      u(1,1)      = 1.d0
      uinv(1,1)   = 1.d0

      tnm(1,1)    = 1.d0
      tnminv(1,1) = 1.d0

      do j = 1, natom
         dnmmass(j,1) = physmass(j)
      end do

      return
      end





!***********************************************************************
      subroutine nm_matrix_even
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   u, uinv, tnm, tnminv, dnmmass, physmass, pi, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: const, di, dj, dp, dum, sqp, sqpinv, dnorm

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dp     = dble(nbead)
      sqp    = sqrt(dp)
      sqpinv = 1.d0/sqrt(dp)
      dnorm  = sqrt(2.d0/dp)

!-----------------------------------------------------------------------
!     /*   u  =  unitary matrix that diagnalizes the spring matrix    */
!-----------------------------------------------------------------------

      dum = -1.d0
      do i = 1, nbead
         u(i,1)     = sqpinv
         u(i,nbead) = dum*sqpinv
         dum = dum*(-1.d0)
      end do

      do i = 1, (nbead-2)/2
         di = dble(i)
         do j = 1, nbead
            dj = dble(j)
            u(j,2*i)   = dnorm*cos(2.d0*pi*di*dj/dp)
            u(j,2*i+1) = dnorm*sin(2.d0*pi*di*dj/dp)
         end do
      end do

      do i = 1, nbead
      do j = 1, nbead
         uinv(j,i) = u(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   make trm and trminv                                        */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, nbead
         tnm(i,j)    = sqp   *u(i,j)
         tnminv(i,j) = sqpinv*uinv(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   real masses of normal modes:                               */
!     /*      dnmmass = eigenvalues of A matrix times nbead.          */
!-----------------------------------------------------------------------

!     /*   centroid   */

      do j = 1, natom
         dnmmass(j,1) = 0.d0
      end do

!     /*   non-centroid   */

      dp = dble(nbead)

      do j = 1, natom
         dnmmass(j,nbead) = 4.d0*dp*physmass(j)
         do i = 1, (nbead-2)/2
            di = dble(i)
            const = 2.d0*(1.d0-cos(2.d0*pi*di/dp))*dp*physmass(j)
            dnmmass(j,2*i)   = const
            dnmmass(j,2*i+1) = const
         end do
      end do

      return
      end





!***********************************************************************
      subroutine nm_matrix_odd
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   u, uinv, tnm, tnminv, dnmmass, physmass, pi, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: const, di, dj, dp, sqp, sqpinv, dnorm

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dp     = dble(nbead)
      sqp    = sqrt(dp)
      sqpinv = 1.d0/sqrt(dp)
      dnorm  = sqrt(2.d0/dp)

!-----------------------------------------------------------------------
!     /*   u  =  unitary matrix that diagnalizes the spring matrix    */
!-----------------------------------------------------------------------

      do i = 1, nbead
         u(i,1)     = sqpinv
      end do

      do i = 1, (nbead-1)/2
         di = dble(i)
         do j = 1, nbead
            dj = dble(j)
            u(j,2*i)   = dnorm*cos(2.d0*pi*di*dj/dp)
            u(j,2*i+1) = dnorm*sin(2.d0*pi*di*dj/dp)
         end do
      end do

      do i = 1, nbead
      do j = 1, nbead
         uinv(j,i) = u(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   make trm and trminv                                        */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, nbead
         tnm(i,j)    = sqp   *u(i,j)
         tnminv(i,j) = sqpinv*uinv(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   real masses of normal modes:                               */
!     /*      dnmmass = eigenvalues of A matrix times nbead.          */
!-----------------------------------------------------------------------

!     /*   centroid   */

      do j = 1, natom
         dnmmass(j,1) = 0.d0
      end do

!     /*   non-centroid   */

      dp = dble(nbead)

      do j = 1, natom
         do i = 1, (nbead-1)/2
            di = dble(i)
            const = 2.d0*(1.d0-cos(2.d0*pi*di/dp))*dp*physmass(j)
            dnmmass(j,2*i)   = const
            dnmmass(j,2*i+1) = const
         end do
      end do

      return
      end





!***********************************************************************
      subroutine nm_matrix_diag
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   u, uinv, tnm, tnminv, dnmmass, physmass, pi, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                         :: i, j

      real(8)                         :: dp, sqp, sqpinv

      real(8), dimension(nbead,nbead) :: a, c

      real(8), dimension(nbead)       :: b

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dp     = dble(nbead)
      sqp    = sqrt(dp)
      sqpinv = 1.d0/sqrt(dp)

!-----------------------------------------------------------------------
!     /*   make the spring matrix A                                   */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead
         a(i,j)        =   0.d0
      end do
      end do

      a(1,nbead)       = - 1.d0
      a(1,1)           = + 2.d0
      a(1,2)           = - 1.d0

      do i = 2, nbead-1
         a(i,i-1)      = - 1.d0
         a(i,i)        = + 2.d0
         a(i,i+1)      = - 1.d0
      end do

      a(nbead,nbead-1) = - 1.d0
      a(nbead,nbead)   = + 2.d0
      a(nbead,1)       = - 1.d0

!-----------------------------------------------------------------------
!     /*   diagonalize A matrix                                       */
!     /*   -->  B is the eigenvalue, C is the eigenvector.            */
!-----------------------------------------------------------------------

      call ddiag ( a, b, c, nbead )

!-----------------------------------------------------------------------
!     /*   u  =  unitary matrix that diagnalizes the spring matrix    */
!-----------------------------------------------------------------------

      do i = 1, nbead
         u(i,1)     = sqpinv
      end do

      do j = 2, nbead
      do i = 1, nbead
         u(i,j)    = c(i,j)
      end do
      end do

      do i = 1, nbead
      do j = 1, nbead
         uinv(j,i) = u(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   make trm and trminv                                        */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, nbead
         tnm(i,j)    = sqp   *u(i,j)
         tnminv(i,j) = sqpinv*uinv(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   real masses of normal modes:                               */
!     /*      dnmmass = eigenvalues of A matrix times nbead.          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 1, nbead
         dnmmass(j,i) = b(i)*dble(nbead)*physmass(j)
      end do
      end do

      return
      end
