!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Oct 29, 2024 by B. Thomsen
!      Description:     Shared Neighbor List Module for PIMD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module neighbor_list_variables
!***********************************************************************

!-----------------------------------------------------------------------
!     Neighbor List Skin
!-----------------------------------------------------------------------

      real(8) :: neighbor_list_skin
      real(8) :: dmax_list_sqr
      real(8) :: rcut_list
      real(8) :: rcut_list_sqr

!-----------------------------------------------------------------------
!     Neighbor List Storage
!-----------------------------------------------------------------------

      integer :: real_list_size

      integer :: list_size_inc = 100

      integer, dimension(:), allocatable :: neighbor_list

      integer, dimension(:), allocatable :: neighbor_list_jx
      integer, dimension(:), allocatable :: neighbor_list_jy
      integer, dimension(:), allocatable :: neighbor_list_jz

!-----------------------------------------------------------------------
!     Neighbor List Navigation
!-----------------------------------------------------------------------

      integer :: max_neighbors = 300

      integer, dimension(:), allocatable :: atom_navigation

!-----------------------------------------------------------------------
!     Local Beads and Atom Indicies
!-----------------------------------------------------------------------

      integer, dimension(:), allocatable :: local_beads
      integer, dimension(:), allocatable :: local_atoms

!-----------------------------------------------------------------------
!     Last positions when neighbor list is updated
!-----------------------------------------------------------------------

      real(8), dimension(:,:), allocatable :: x_list
      real(8), dimension(:,:), allocatable :: y_list
      real(8), dimension(:,:), allocatable :: z_list

!-----------------------------------------------------------------------
!     Bigbox if needed for small box sizes
!-----------------------------------------------------------------------

      real(8), dimension(3,3)::  bigbox
      real(8), dimension(3,3)::  bigboxinv

!-----------------------------------------------------------------------
!     Number of replicated boxes
!-----------------------------------------------------------------------

      integer, dimension(3) :: nbox_list

!***********************************************************************
      end module neighbor_list_variables
!***********************************************************************





!***********************************************************************
      subroutine extend_neighbor_list_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use neighbor_list_variables, only : real_list_size, &
     &   list_size_inc, neighbor_list, neighbor_list_jx, &
     &   neighbor_list_jy, neighbor_list_jz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   Initialize variables
      implicit none

!     //   Initialize variables
      integer, dimension(:), allocatable :: new_list

!-----------------------------------------------------------------------
!     /*   Update the list size and delete the old list               */
!-----------------------------------------------------------------------

!     //   Allocate new list
      allocate(new_list(real_list_size + list_size_inc))

!     //   Copy the data from the old list and add -1 to new elements
      new_list(1:real_list_size) = neighbor_list
      new_list(real_list_size + 1:real_list_size + list_size_inc) = -1

!     //   Swap the new neighbor list to the olds position
!     //   Note that the old data will be deallocated
      call move_alloc(from=new_list, to=neighbor_list)

!     //   Allocate new list
      allocate(new_list(real_list_size + list_size_inc))

!     //   Copy the data from the old list and add -1 to new elements
      new_list(1:real_list_size) = neighbor_list_jx
      new_list(real_list_size + 1:real_list_size + list_size_inc) = -1

!     //   Swap the new neighbor list to the olds position
!     //   Note that the old data will be deallocated
      call move_alloc(from=new_list, to=neighbor_list_jx)

!     //   Allocate new list
      allocate(new_list(real_list_size + list_size_inc))

!     //   Copy the data from the old list and add -1 to new elements
      new_list(1:real_list_size) = neighbor_list_jy
      new_list(real_list_size + 1:real_list_size + list_size_inc) = -1

!     //   Swap the new neighbor list to the olds position
!     //   Note that the old data will be deallocated
      call move_alloc(from=new_list, to=neighbor_list_jy)

!     //   Allocate new list
      allocate(new_list(real_list_size + list_size_inc))

!     //   Copy the data from the old list and add -1 to new elements
      new_list(1:real_list_size) = neighbor_list_jz
      new_list(real_list_size + 1:real_list_size + list_size_inc) = -1

!     //   Swap the new neighbor list to the olds position
!     //   Note that the old data will be deallocated
      call move_alloc(from=new_list, to=neighbor_list_jz)

      real_list_size = real_list_size + list_size_inc

!***********************************************************************
    end subroutine extend_neighbor_list_XMPI
!***********************************************************************





!***********************************************************************
      subroutine initialize_neighbor_list_XMPI(cutoff)
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

       use common_variables, only : iounit

       use neighbor_list_variables, only : real_list_size,   &
     &   list_size_inc, neighbor_list, neighbor_list_skin,   &
     &   dmax_list_sqr, local_beads, local_atoms, rcut_list, &
     &   rcut_list_sqr, atom_navigation, neighbor_list_jx,   &
     &   neighbor_list_jy, neighbor_list_jz, list_size_inc

       use XMPI_variables, only : &
     &   jstart_atom, jend_atom, jstart_bead, jend_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

      integer :: i, j, k

      real(8) :: cutoff

!-----------------------------------------------------------------------
!     /*   Initialize array and sizes                                 */
!-----------------------------------------------------------------------

       if ( .not. allocated(neighbor_list)) &
     &    allocate(neighbor_list(list_size_inc))
       if ( .not. allocated(neighbor_list_jx)) &
     &    allocate(neighbor_list_jx(list_size_inc))
       if ( .not. allocated(neighbor_list_jy)) &
     &    allocate(neighbor_list_jy(list_size_inc))
       if ( .not. allocated(neighbor_list_jz)) &
     &    allocate(neighbor_list_jz(list_size_inc))

       real_list_size = list_size_inc

!-----------------------------------------------------------------------
!     /*   Read neighbor list skin                                    */
!-----------------------------------------------------------------------

      call read_real1_MPI &
     &   ( neighbor_list_skin, '<neighbor_list_skin>', 20, iounit )

!     /*   Max squared displacement before recalculation   */
      dmax_list_sqr = 0.25 * neighbor_list_skin * neighbor_list_skin

!-----------------------------------------------------------------------
!     /*   Initialize neighbor list cutoffs                           */
!-----------------------------------------------------------------------

      rcut_list     = cutoff + neighbor_list_skin
      rcut_list_sqr = rcut_list*rcut_list

!-----------------------------------------------------------------------
!     /*   Generate Local Bead and Atom Lists                         */
!-----------------------------------------------------------------------

      if ( .not. allocated(local_beads) ) &
     &   allocate( local_beads(jend_bead - jstart_bead + 1) )

      j = 0

      do k = jstart_bead, jend_bead
         j = j + 1
         local_beads(j) = k
      end do

      if ( .not. allocated(local_atoms) ) &
     &   allocate( local_atoms(jend_atom - jstart_atom + 1) )

      j = 0

      do i = jstart_atom, jend_atom
         j = j + 1
         local_atoms(j) = i
      end do

      if ( .not. allocated(atom_navigation) ) &
     &   allocate( atom_navigation( size(local_beads) * &
     &                              size(local_atoms) + 1 ) )

!***********************************************************************
      end subroutine initialize_neighbor_list_XMPI
!***********************************************************************





!***********************************************************************
      subroutine initialize_neighbor_list_MPI(cutoff)
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

       use common_variables, only : iounit, nbead, natom, &
     &   nprocs_main, nprocs_sub, myrank_main, myrank_sub

       use neighbor_list_variables, only : real_list_size,   &
     &   list_size_inc, neighbor_list, neighbor_list_skin,   &
     &   dmax_list_sqr, local_beads, local_atoms, rcut_list, &
     &   rcut_list_sqr, atom_navigation, neighbor_list_jx,   &
     &   neighbor_list_jy, neighbor_list_jz, list_size_inc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

      integer :: i, j, k

      real(8) :: cutoff

!-----------------------------------------------------------------------
!     /*   Initialize array and sizes                                 */
!-----------------------------------------------------------------------

       if ( .not. allocated(neighbor_list)) &
     &    allocate(neighbor_list(list_size_inc))
       if ( .not. allocated(neighbor_list_jx)) &
     &    allocate(neighbor_list_jx(list_size_inc))
       if ( .not. allocated(neighbor_list_jy)) &
     &    allocate(neighbor_list_jy(list_size_inc))
       if ( .not. allocated(neighbor_list_jz)) &
     &    allocate(neighbor_list_jz(list_size_inc))

       real_list_size = list_size_inc

!-----------------------------------------------------------------------
!     /*   Read neighbor list skin                                    */
!-----------------------------------------------------------------------

      call read_real1_MPI &
     &   ( neighbor_list_skin , '<neighbor_list_skin>', 20, iounit )

!     /*   Max squared displacement before recalculation   */
      dmax_list_sqr = 0.25 * neighbor_list_skin * neighbor_list_skin

!-----------------------------------------------------------------------
!     /*   Initialize neighbor list cutoffs                           */
!-----------------------------------------------------------------------

      rcut_list     = cutoff + neighbor_list_skin
      rcut_list_sqr = rcut_list*rcut_list

!-----------------------------------------------------------------------
!     /*   Generate Local Bead and Atom Lists                         */
!-----------------------------------------------------------------------

     j = 0

     do k = 1, nbead
        if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle
        j = j + 1
     end do

     if ( .not. allocated(local_beads) ) allocate( local_beads(j) )

     j = 0
     do k = 1, nbead
        if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle
        j = j + 1
        local_beads(j) = k
     end do

     j = 0
     do i = 1, natom
        if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle
        j = j + 1
     end do

     if ( .not. allocated(local_atoms) ) allocate( local_atoms(j) )

     j = 0

     do i = 1, natom
        if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle
        j = j + 1
        local_atoms(j) = i
     end do

      if ( .not. allocated(atom_navigation) ) &
     &   allocate( atom_navigation( size(local_beads) * &
     &                              size(local_atoms) + 1 ) )

!***********************************************************************
      end subroutine initialize_neighbor_list_MPI
!***********************************************************************





!***********************************************************************
      subroutine make_neighbor_list_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, box, volume, natom, iounit, iboundary

      use neighbor_list_variables, only : x_list, y_list, z_list,    &
     &   bigbox, bigboxinv, local_beads, local_atoms, dmax_list_sqr, &
     &   nbox_list, rcut_list, rcut_list_sqr, neighbor_list,         &
     &   neighbor_list_jx, neighbor_list_jy, neighbor_list_jz,       &
     &   atom_navigation, max_neighbors

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, jx, jy, jz, j2, idx_1, idx_2

!     //   real variables
      real(8) :: dx, dy, dz, d2, d2max, rx, ry, rz, r2, aij, bij, cij, &
     &           ax, ay, az, bx, by, bz, cx, cy, cz, absa, absb, absc

!      //   real variables
!      real(8) :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

!        //   last position
         if ( .not. allocated(x_list) ) &
     &      allocate( x_list(size(local_atoms),size(local_beads)) )
         if ( .not. allocated(y_list) ) &
     &      allocate( y_list(size(local_atoms),size(local_beads)) )
         if ( .not. allocated(z_list) ) &
     &      allocate( z_list(size(local_atoms),size(local_beads)) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads
         do idx_1 = 1, size(local_beads)

!           //   bead index
            k = local_beads(idx_1)

!           //   loop of atoms
            do idx_2 = 1, size(local_atoms)

!              //   atom index
               i = local_atoms(idx_2)

!              //   deviation from last position
               dx = x(i,k) - x_list(idx_2,idx_1)
               dy = y(i,k) - y_list(idx_2,idx_1)
               dz = z(i,k) - z_list(idx_2,idx_1)

!              //   deviation squared
               d2 = dx*dx + dy*dy + dz*dz

!              //   maximum deviation squared
               d2max = max( d2, d2max )

!           //   loop of atoms
            end do

!        //   loop of beads
         end do

!        //   communicate all processes and get maximum
         call my_mpi_allreduce_max_0( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( d2max .lt. dmax_list_sqr ) return
        
!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_list(1) = 1
         nbox_list(2) = 1
         nbox_list(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_list(1) = int(2.d0*rcut_list/absa) + 1
         nbox_list(2) = int(2.d0*rcut_list/absb) + 1
         nbox_list(3) = int(2.d0*rcut_list/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   counter
      l = 0

!     //   mark
      atom_navigation(1) = 1

!     /*   periodic boundary   */
      if ( nbox_list(1)*nbox_list(2)*nbox_list(3) .eq. 1 ) then

!        //   loop of beads
         do idx_1 = 1, size(local_beads)

!           //   bead index
            k = local_beads(idx_1)

!           //   loop of atom i
            do idx_2 = 1, size(local_atoms)

!              //   atom index
               i = local_atoms(idx_2)

!              //  loop of atom j
               do j = 1, natom

!                 /*   skip same atom   */
                  if ( j .eq. i ) cycle

!                 //   separation of atoms i and j
                  rx = x(j,k) - x(i,k)
                  ry = y(j,k) - y(i,k)
                  rz = z(j,k) - z(i,k)

!                 //   apply boundary condition
                  call pbc_atom_MPI( rx, ry, rz )

!                 //   distance of atoms i and j squared
                  r2 = rx*rx + ry*ry + rz*rz

!                 //   if distance is smaller than list cutoff radius
                  if ( r2 .lt. rcut_list_sqr ) then

!                    //   update counter
                     l = l + 1

!                    //   extend memory size if needed
                     if (l .gt. size(neighbor_list)) &
     &                   call extend_neighbor_list_XMPI

!                    //   record atom in neighbor list
                     neighbor_list(l) = j

!                 //   end of if statement
                  end if

!              //  loop of atom j
               end do

!              //   mark
               atom_navigation( (idx_1 - 1) * &
     &            size(local_atoms) + idx_2 + 1 ) = l + 1

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_list(1))*box(:,1)
         bigbox(:,2) = dble(nbox_list(2))*box(:,2)
         bigbox(:,3) = dble(nbox_list(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do idx_1 = 1, size(local_beads)

!           //   bead index
            k = local_beads(idx_1)

!           /*   loop of atom pairs   */
            do idx_2 = 1, size(local_atoms)

!              //   atom index
               i = local_atoms(idx_2)

!              /*   loop of atom pairs   */
               do j = 1, natom

!                 /*   loop of replicated boxes   */
                  do jx = 0, nbox_list(1)-1
                  do jy = 0, nbox_list(2)-1
                  do jz = 0, nbox_list(3)-1

!                    /*   square of box index   */
                     j2 = jx*jx + jy*jy + jz*jz

!                    /*   skip same atom   */
                     if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!                    /*   distance of i and j in same box   */
                     rx = x(i,k) - x(j,k)
                     ry = y(i,k) - y(j,k)
                     rz = z(i,k) - z(j,k)

!                    /*   distance of i and j in different box  */
                     rx = rx - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                     ry = ry - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                     rz = rz - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!                    /*   vector in big box   */
                     aij = bigboxinv(1,1)*rx + bigboxinv(1,2)*ry &
     &                   + bigboxinv(1,3)*rz
                     bij = bigboxinv(2,1)*rx + bigboxinv(2,2)*ry &
     &                   + bigboxinv(2,3)*rz
                     cij = bigboxinv(3,1)*rx + bigboxinv(3,2)*ry &
     &                   + bigboxinv(3,3)*rz

!                    /*   apply periodic boundary in big box   */
                     aij = aij - nint(aij)
                     bij = bij - nint(bij)
                     cij = cij - nint(cij)

!                    /*   distance of nearest i and j   */
                     rx = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &                  + bigbox(1,3)*cij
                     ry = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &                  + bigbox(2,3)*cij
                     rz = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &                  + bigbox(3,3)*cij

!                    //   distance of atoms i and j squared
                     r2 = rx*rx + ry*ry + rz*rz

!                    //   if distance is smaller than list cutoff radius
                     if ( r2 .lt. rcut_list_sqr ) then

!                       //   update counter
                        l = l + 1

!                       //   extend memory size if needed
                        if ( l .gt. size(neighbor_list) ) &
     &                     call extend_neighbor_list_XMPI

!                       //   list of j atoms for atom i, bead k
                        neighbor_list(l) = j

!                       //   list of box for atom i, bead k
                        neighbor_list_jx(l) = jx
                        neighbor_list_jy(l) = jy
                        neighbor_list_jz(l) = jz

!                    //   end of if statement
                     end if

!                 /*   loop of replicated boxes   */
                  end do
                  end do
                  end do

!              //   loop of atom j
               end do

!              //   mark
               atom_navigation( ( idx_1 - 1 ) * &
     &            size(local_atoms) + idx_2 + 1 ) = l + 1

!           //   loop of atom i
            end do

!        //   loop of beads
         end do

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   update the largest neighbor list size                      */
!-----------------------------------------------------------------------

      do i = 2, size(atom_navigation)
         max_neighbors = max( max_neighbors, atom_navigation(i) &
     &                      - atom_navigation(i-1) )
      end do

!-----------------------------------------------------------------------
!     /*   save last position                                         */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do idx_1 = 1, size(local_beads)

!        //   bead index
         j = local_beads(idx_1)

!        /*   loop of atoms   */
         do idx_2 = 1, size(local_atoms)

!           //   atom index
            i = local_atoms(idx_2)

!           /*   distance of i and j in same box   */
            rx = x(i,j)
            ry = y(i,j)
            rz = z(i,j)

            x_list(idx_2,idx_1) = rx
            y_list(idx_2,idx_1) = ry
            z_list(idx_2,idx_1) = rz

!        /*   loop of atoms   */
         end do

!     /*   loop of beads   */
      end do

      return
      end

