!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 8, 2024 by M. Shiga
!      Description:     metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine metacycle_dual_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_hmc, istep_start, istep_end, nstep, iexit, iref

      use rehmc_variables, only : &
     &   jstep

      use meta_variables, only : nref_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   ux --> x   */
      call nm_trans_cart( 2 )

!     /*   get interatomic forces   */
      call getforce_dual_lo_MPI

!     /*   get interatomic forces   */
      call getforce_dual_hi_MPI

!     /*   fx --> fux   */
      call nm_trans_force_cart

!     /*   calculate gaussian potential   */
      call getforce_hills_meta_MPI

!     /*   calculate harmonic potential   */
      call getforce_ref_meta_MPI

!     /*   energy   */
      call getenergy_meta_hmc

!     /*   save data   */
      call save_meta_hmc

!     /*   adjust HMC step   */
      call adjust_step_rehmc_MPI

!     /*   hamiltonian and temperature   */
      call standard_meta_hmc_MPI

!     /*   do some analysis   */
      call analysis_meta_MPI ( 0 )
      call analysis_meta_MPI ( 1 )
!cc      call analysis_slhmc_MPI

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

!     //   mc loop
      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   add new gaussian   */
         call addhills_meta_MPI

!        /*   calculate gaussian potential   */
         call getforce_hills_meta_MPI

!        /*   calculate harmonic potential   */
         call getforce_ref_meta_MPI

!        /*   get interatomic forces   */
         call getforce_dual_lo_MPI

!        /*   restore hamiltonian if just trained   */
         call justtrained_slhmc

!        //   md loop
         do jstep = 1, istep_hmc

!           /*   update system velocity: real potential    */
            call update_vel_sys_meta

!           //   inner loop
            do iref = 1, nref_meta

!              /*   update cv velocity: gaussian potential   */
               call update_vel_cv_meta

!              /*   update cv velocity: harmonic potential   */
               call update_vel_cv_ref_meta

!              /*   update system velocity: harmonic potential   */
               call update_vel_sys_ref_meta

!              /*   update system position   */
               call update_pos_sys_meta

!              /*   ux --> x   */
               call nm_trans_cart( 2 )

!              /*   update cv position   */
               call update_pos_cv_meta

!              /*   apply periodic boundary condition   */
               call pbc_cv_meta

!              /*   calculate harmonic potential   */
               call getforce_ref_meta_MPI

!              /*   update system velocity: harmonic potential   */
               call update_vel_sys_ref_meta

!              /*   update cv velocity: harmonic potential   */
               call update_vel_cv_ref_meta

!              /*   calculate gaussian potential   */
               call getforce_hills_meta_MPI

!              /*   update cv velocity: gaussian potential   */
               call update_vel_cv_meta

!           //   inner loop
            end do

!           /*   get interatomic forces   */
            call getforce_dual_lo_MPI

!           /*   fx --> fux   */
            call nm_trans_force_cart

!           /*   update system velocity: real potential    */
            call update_vel_sys_meta

!        //   md loop
         end do

!        /*   get interatomic forces   */
         call getforce_dual_hi_MPI

!        /*   energy   */
         call getenergy_meta_hmc

!        /*   judge accept or reject   */
         call judge_meta_hmc_MPI

!        /*   adjust HMC step   */
         call adjust_step_rehmc_MPI

!        /*   hamiltonian and temperature   */
         call standard_meta_hmc_MPI

!        /*   output restart   */
         call backup_meta_hmc_MPI

!        /*   do some analysis   */
         call analysis_meta_MPI ( 2 )
!cc         call analysis_slhmc

!        /*   exit if `exit.dat' exists   */
         call softexit_MPI
         if ( iexit .eq. 1 ) exit

!     //   mc loop
      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine getenergy_meta_hmc
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, fictmass, pot, natom, nbead

      use meta_variables, only: &
     &   pot_meta, pot_ref_meta, vmeta, fictmass_meta, nmeta

      use rehmc_variables, only : &
     &   hamiltonian_bead

      use meta_variables, only : &
     &   ekin_bead, epot_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      do i = 1, nbead

         ekin_bead(i) = 0.d0

         do j = 1, natom
             ekin_bead(i) = ekin_bead(i) &
     &          + fictmass(j,i) * vux(j,i) * vux(j,i) &
     &          + fictmass(j,i) * vuy(j,i) * vuy(j,i) &
     &          + fictmass(j,i) * vuz(j,i) * vuz(j,i)
         end do

         do j = 1, nmeta
             ekin_bead(i) = ekin_bead(i) &
     &          + fictmass_meta(j) * vmeta(j,i) * vmeta(j,i)
         end do

         ekin_bead(i) = 0.5d0 * ekin_bead(i)

         epot_bead(i) = pot(i) + pot_ref_meta(i) + pot_meta(i)

         hamiltonian_bead(i) = ekin_bead(i) + epot_bead(i)

      end do

      return
      end






!***********************************************************************
      subroutine judge_meta_hmc_MPI
!***********************************************************************
!=======================================================================
!
!     metropolis step for hybrid monte carlo.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, beta, nbead, myrank

      use rehmc_variables, only : &
     &   hamiltonian_bead, hamiltonian_bead_save, pot_rehmc_last, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, fx_rehmc_last, &
     &   fy_rehmc_last, fz_rehmc_last, ratio_hmc, naccept_hmc, &
     &   nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh, bfactor

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of each bead                                    */
!-----------------------------------------------------------------------

      do i = 1, nbead

!        /*   energy difference   */
         dh = hamiltonian_bead(i) - hamiltonian_bead_save(i)

!        /*   beta times energy difference   */
         bfactor = beta * dh

!        /*   start judge acceptance   */
         if ( bfactor .lt. 75.d0 ) then

            if ( bfactor .le. 0.d0 ) then

!              /*   accepted   */
               naccept_hmc = naccept_hmc + 1

            else

!              /*   uniform random number   */
               if ( myrank .eq. 0 ) randomno = ranf1()

!              /*   use the same random number   */
               call my_mpi_bcast_real_0 ( randomno )

               if ( exp(-bfactor) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_hmc = naccept_hmc + 1

               else

!                 /*   rejected   */
                  nreject_hmc = nreject_hmc + 1

!                 /*   recover saved data   */
                  call recover_meta_hmc( i )

               end if

            end if

         else

!           /*   rejected   */
            nreject_hmc = nreject_hmc + 1

!           /*   recover saved data   */
            call recover_meta_hmc( i )

!        /*   end judge acceptance   */
         end if

      end do

!     /*   acceptance ratio   */
      if ( naccept_hmc .eq. 0 ) then
         ratio_hmc = 0.d0
      else
         ratio_hmc = dble(naccept_hmc)/dble(naccept_hmc+nreject_hmc)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of each bead                                      */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_meta_MPI

!     /*   energy   */
      call getenergy_meta_hmc

!     /*   save data   */
      call save_meta_hmc

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine init_velocity_meta_MPI
!***********************************************************************
!=======================================================================
!
!     initialize atomic and cv velocities for metadynamics
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, hamiltonian_cor, beta, nbead, natom, &
     &   myrank

      use meta_variables, only : &
     &   vmeta, fictmass_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: gasdev, vsigma, ekin_cor

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      ekin_cor = 0.d0

      do i = 1, nbead
      do j = 1, natom

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      - 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

         vsigma = sqrt( 1.d0/beta/fictmass(j,i) )

         vux(j,i) = vsigma*gasdev()
         vuy(j,i) = vsigma*gasdev()
         vuz(j,i) = vsigma*gasdev()

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass(j,i)*vux(j,i)*vux(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuy(j,i)*vuy(j,i) &
     &      + 0.5d0*fictmass(j,i)*vuz(j,i)*vuz(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   initial velocity of cv                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nmeta

         ekin_cor = ekin_cor &
     &      - 0.5d0*fictmass_meta(i)*vmeta(i,j)*vmeta(i,j)

         vsigma = sqrt(1.d0/beta/fictmass_meta(i))

         vmeta(i,j) = vsigma*gasdev()

         ekin_cor = ekin_cor &
     &      + 0.5d0*fictmass_meta(i)*vmeta(i,j)*vmeta(i,j)

      end do
      end do

      end if

      call my_mpi_bcast_real_2( vux, natom, nbead)
      call my_mpi_bcast_real_2( vuy, natom, nbead)
      call my_mpi_bcast_real_2( vuz, natom, nbead)

      call my_mpi_bcast_real_2( vmeta, nmeta, nbead)

      call my_mpi_bcast_real_0( ekin_cor )

!-----------------------------------------------------------------------
!     /*   energy correction                                          */
!-----------------------------------------------------------------------

      hamiltonian_cor = hamiltonian_cor - ekin_cor

      return
      end





!***********************************************************************
      subroutine setup_meta_dual_MPI
!***********************************************************************

      implicit none

      call setup_meta_hmc_MPI

      return
      end





!***********************************************************************
      subroutine backup_meta_dual_MPI
!***********************************************************************

      implicit none

      call backup_meta_hmc_MPI

      return
      end





!***********************************************************************
      subroutine setup_meta_hmc_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start

      use meta_variables, only : &
     &   icv_start, ihills_start

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   read input file and memory allocation                      */
!-----------------------------------------------------------------------

      call setmdparams_MPI

      call setup_meta_hmc_param_MPI

!-----------------------------------------------------------------------
!     /*   atomic mass                                                */
!-----------------------------------------------------------------------

      call init_mass_md_MPI

!-----------------------------------------------------------------------
!     /*   atom position                                              */
!-----------------------------------------------------------------------

      if      ( ipos_start .eq. 0 ) then

!        /*   initialize position   */
         call init_position_cart_MPI

!        /*   ux --> x   */
         call nm_trans_cart( 2 )

      else if ( ipos_start .eq. 1 ) then

!        /*   restart position   */
         call restart_position_MPI( 2 )

      else

!        /*   error handling   */
         call error_handling_MPI( 1, 'subroutine setup_meta_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   atom velocity                                              */
!-----------------------------------------------------------------------

      if      ( ivel_start .eq. 0 ) then

!        /*   initialize velocity   */
         call init_velocity_MPI

      else if ( ivel_start .eq. 1 ) then

!        /*   restart velocity   */
         call restart_velocity_MPI( 2 )

      else

!        /*   error handling   */
         call error_handling_MPI( 1, 'subroutine setup_meta_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   actual position of cv                                      */
!-----------------------------------------------------------------------

      call get_cv_meta_MPI

!-----------------------------------------------------------------------
!     /*   initialize/restart cv                                      */
!-----------------------------------------------------------------------

      if      ( icv_start .le. 0 ) then

!        /*   initialize cv   */
         call init_cv_meta_MPI

      else if ( icv_start .eq. 1 ) then

!        /*   restart cv   */
         call restart_cv_meta_MPI( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   initialize/restart hills                                   */
!-----------------------------------------------------------------------

      if      ( ihills_start .le. 0 ) then

!        /*   initialize hills   */
         call init_hills_meta_MPI

      else if ( ihills_start .eq. 1 ) then

!        /*   initialize hills   */
         call restart_hills_meta_MPI( 1 )

      end if

      return
      end





!***********************************************************************
      subroutine setup_meta_hmc_param_MPI
!***********************************************************************
!=======================================================================
!
!     read parameters for metadynamics
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_time, dt, beta, pi, boltz, hamiltonian_cor, iounit, natom, &
     &   nbead, istep_start, istep_hmc, myrank

      use meta_variables, only : &
     &   rmeta, vmeta, smeta, fictmass_meta, fref_meta, fc_meta, &
     &   gh_meta, gw_meta, gw_meta_type, gw_meta_save, gc_meta_save, &
     &   gh_meta_save, time_limit_meta, pot_ref_meta, time_meta_save, &
     &   dmeta, req_meta, params_rec_meta, dtemp_meta, &
     &   fx_meta, fy_meta, fz_meta, fux_meta, fuy_meta, fuz_meta, &
     &   time_cv_meta, time_fc_meta,  pot_meta, fmeta, gc_meta, &
     &   dt_ref_meta, time_cv_meta, dmeta_max, params_meta, &
     &   itype_meta, i_meta, j_meta, k_meta, l_meta, mg_meta, &
     &   mu_meta, nu_meta, idim_rec_meta, nref_meta, itype_meta, &
     &   iprint_rec_meta, ihills_start, icv_start, ntype_meta, &
     &   ioption_meta, joption_meta, ipbc_meta, nmeta

      use rehmc_variables, only : &
     &   hamiltonian_bead, pot_rehmc_last, hamiltonian_bead_save, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, mbox_save, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   pot_save, x_save, y_save, z_save, ux_save, uy_save, uz_save, &
     &   vx_save, vy_save, vz_save, fx_save, fy_save, fz_save, &
     &   ratio_min_hmc, ratio_max_hmc, istep_adjust_hmc, &
     &   istep_min_hmc, istep_max_hmc, istep_mul_hmc

      use meta_variables, only : &
     &   pot_meta_save, pot_ref_meta_save, rmeta_save, smeta_save, &
     &   vmeta_save, fmeta_save, fref_meta_save, ekin_bead, epot_bead, &
     &   ekin_bead_save, epot_bead_save

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, ierr, itest

      real(8) :: f1, f2

      real(8) :: huge = 1.d38

      character(len=8) :: params_char, char_meta

      character(len=120) :: char_line

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   read number of collective variables                        */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nmeta, '<nmeta>', 7, iounit )

!-----------------------------------------------------------------------
!     /*   read maximum number of gaussians                           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( mg_meta, '<mg_meta>', 9, iounit )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   cv position   */
      if ( .not. allocated( rmeta ) ) &
     &   allocate( rmeta(nmeta,nbead) )

!     /*   cv velocity   */
      if ( .not. allocated( vmeta ) ) &
     &   allocate( vmeta(nmeta,nbead) )

!     /*   actual cv position   */
      if ( .not. allocated( smeta ) ) &
     &   allocate( smeta(nmeta,nbead) )

!     /*   cv mass   */
      if ( .not. allocated( fictmass_meta ) ) &
     &   allocate( fictmass_meta(nmeta) )

!     /*   type of cv   */
      if ( .not. allocated( itype_meta ) ) &
     &   allocate( itype_meta(nmeta) )

!     /*   boundary condition of cv   */
      if ( .not. allocated( ipbc_meta ) ) &
     &   allocate( ipbc_meta(nmeta) )

!     /*   harmonic potential, each bead    */
      if ( .not. allocated( pot_ref_meta ) ) &
     &   allocate( pot_ref_meta(nbead) )

!     /*   cv force: harmonic   */
      if ( .not. allocated( fref_meta ) ) &
     &   allocate( fref_meta(nmeta,nbead) )

!     /*   force constant of harmonic potential   */
      if ( .not. allocated( fc_meta ) ) &
     &   allocate( fc_meta(nmeta) )

!     /*   gaussian potential, each bead   */
      if ( .not. allocated( pot_meta ) ) &
     &   allocate( pot_meta(nbead) )

!     /*   cv force: gaussian   */
      if ( .not. allocated( fmeta ) ) &
     &   allocate( fmeta(nmeta,nbead) )

!     /*   gaussian center   */
      if ( .not. allocated( gc_meta ) ) &
     &   allocate( gc_meta(nmeta,mg_meta) )

!     /*   gaussian height   */
      if ( .not. allocated( gh_meta ) ) &
     &   allocate( gh_meta(mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta ) ) &
     &   allocate( gw_meta(nmeta,mg_meta) )

!     /*   gaussian width   */
      if ( .not. allocated( gw_meta_type ) ) &
     &   allocate( gw_meta_type(ntype_meta) )

!     /*   current value of gaussian width   */
      if ( .not. allocated( gw_meta_save ) ) &
     &   allocate( gw_meta_save(nmeta) )

!     /*   latest gaussian center   */
      if ( .not. allocated( gc_meta_save ) ) &
     &   allocate( gc_meta_save(nmeta,nbead) )

!     /*   time since last gaussian has been added   */
      if ( .not. allocated( time_meta_save ) ) &
     &   allocate( time_meta_save(nbead) )

!     /*   maximum displacement of cv   */
      if ( .not. allocated( dmeta ) ) &
     &   allocate( dmeta(nbead) )

!     /*   atoms i, j, k, l of cv   */
      if ( .not. allocated( i_meta ) ) &
     &   allocate( i_meta(nmeta) )
      if ( .not. allocated( j_meta ) ) &
     &   allocate( j_meta(nmeta) )
      if ( .not. allocated( k_meta ) ) &
     &   allocate( k_meta(nmeta) )
      if ( .not. allocated( l_meta ) ) &
     &   allocate( l_meta(nmeta) )

!     /*   rational function parameters for coordination number   */
      if ( .not. allocated( nu_meta ) ) &
     &   allocate( nu_meta(nmeta,2) )
      if ( .not. allocated( mu_meta ) ) &
     &   allocate( mu_meta(nmeta,2) )
      if ( .not. allocated( req_meta ) ) &
     &   allocate( req_meta(nmeta,2) )

!     /*   parameters for reconstruction of pes   */
      if ( .not. allocated( params_rec_meta ) ) &
     &   allocate( params_rec_meta(3,ntype_meta) )

!     /*   axes of hills reconstruction   */
      if ( .not. allocated( idim_rec_meta ) ) &
     &   allocate( idim_rec_meta(nmeta) )

!     /*   forces    */
      if ( .not. allocated( fx_meta ) ) &
     &   allocate( fx_meta(natom,nbead) )
      if ( .not. allocated( fy_meta ) ) &
     &   allocate( fy_meta(natom,nbead) )
      if ( .not. allocated( fz_meta ) ) &
     &   allocate( fz_meta(natom,nbead) )

!     /*   forces    */
      if ( .not. allocated( fux_meta ) ) &
     &   allocate( fux_meta(natom,nbead) )
      if ( .not. allocated( fuy_meta ) ) &
     &   allocate( fuy_meta(natom,nbead) )
      if ( .not. allocated( fuz_meta ) ) &
     &   allocate( fuz_meta(natom,nbead) )

!     /*   minimum and maximum   */
      if ( .not. allocated( params_meta ) ) &
     &   allocate( params_meta(2,nmeta) )

!-----------------------------------------------------------------------
!     //   for hmc
!-----------------------------------------------------------------------

!     /*   hamiltonian   */
      if ( .not. allocated( hamiltonian_bead) ) &
     &   allocate( hamiltonian_bead(nbead) )

!     /*   kinetic energy   */
      if ( .not. allocated( ekin_bead) ) &
     &   allocate( ekin_bead(nbead) )

!     /*   potential energy   */
      if ( .not. allocated( epot_bead) ) &
     &   allocate( epot_bead(nbead) )

!     /*   potential   */
      if ( .not. allocated( pot_rehmc_last) ) &
     &   allocate( pot_rehmc_last(nbead) )

!     /*   position   */
      if ( .not. allocated( x_rehmc_last) ) &
     &   allocate( x_rehmc_last(natom,nbead) )
      if ( .not. allocated( y_rehmc_last) ) &
     &   allocate( y_rehmc_last(natom,nbead) )
      if ( .not. allocated( z_rehmc_last) ) &
     &   allocate( z_rehmc_last(natom,nbead) )

!     /*   force   */
      if ( .not. allocated( fx_rehmc_last) ) &
     &   allocate( fx_rehmc_last(natom,nbead) )
      if ( .not. allocated( fy_rehmc_last) ) &
     &   allocate( fy_rehmc_last(natom,nbead) )
      if ( .not. allocated( fz_rehmc_last) ) &
     &   allocate( fz_rehmc_last(natom,nbead) )

!     /*   saved coordinates   */
      if ( .not. allocated( x_save ) ) &
     &   allocate( x_save(natom,nbead) )
      if ( .not. allocated( y_save ) ) &
     &   allocate( y_save(natom,nbead) )
      if ( .not. allocated( z_save ) ) &
     &   allocate( z_save(natom,nbead) )

!     /*   saved coordinates   */
      if ( .not. allocated( ux_save ) ) &
     &   allocate( ux_save(natom,nbead) )
      if ( .not. allocated( uy_save ) ) &
     &   allocate( uy_save(natom,nbead) )
      if ( .not. allocated( uz_save ) ) &
     &   allocate( uz_save(natom,nbead) )

!     /*   saved velocities   */
      if ( .not. allocated( vx_save ) ) &
     &   allocate( vx_save(natom,nbead) )
      if ( .not. allocated( vy_save ) ) &
     &   allocate( vy_save(natom,nbead) )
      if ( .not. allocated( vz_save ) ) &
     &   allocate( vz_save(natom,nbead) )

!     /*   saved forces   */
      if ( .not. allocated( fx_save ) ) &
     &   allocate( fx_save(natom,nbead) )
      if ( .not. allocated( fy_save ) ) &
     &   allocate( fy_save(natom,nbead) )
      if ( .not. allocated( fz_save ) ) &
     &   allocate( fz_save(natom,nbead) )

!     /*   saved potential   */
      if ( .not. allocated( pot_save ) ) &
     &   allocate( pot_save(nbead) )

!     /*   saved hamiltonian   */
      if ( .not. allocated( hamiltonian_bead_save ) ) &
     &   allocate( hamiltonian_bead_save(nbead) )

!     /*   saved kinetic energy   */
      if ( .not. allocated( ekin_bead_save ) ) &
     &   allocate( ekin_bead_save(nbead) )

!     /*   saved potential energy   */
      if ( .not. allocated( epot_bead_save ) ) &
     &   allocate( epot_bead_save(nbead) )

      if ( .not. allocated( mbox_save ) ) &
     &   allocate( mbox_save(3,natom,nbead) )

!     /*   cv position   */
      if ( .not. allocated( rmeta_save ) ) &
     &   allocate( rmeta_save(nmeta,nbead) )

!     /*   cv velocity   */
      if ( .not. allocated( vmeta_save ) ) &
     &   allocate( vmeta_save(nmeta,nbead) )

!     /*   actual cv position   */
      if ( .not. allocated( smeta_save ) ) &
     &   allocate( smeta_save(nmeta,nbead) )

!     /*   harmonic potential, each bead    */
      if ( .not. allocated( pot_ref_meta_save ) ) &
     &   allocate( pot_ref_meta_save(nbead) )

!     /*   cv force: harmonic   */
      if ( .not. allocated( fref_meta_save ) ) &
     &   allocate( fref_meta_save(nmeta,nbead) )

!     /*   gaussian potential, each bead   */
      if ( .not. allocated( pot_meta_save ) ) &
     &   allocate( pot_meta_save(nbead) )

!     /*   cv force: gaussian   */
      if ( .not. allocated( fmeta_save ) ) &
     &   allocate( fmeta_save(nmeta,nbead) )

!-----------------------------------------------------------------------
!     /*   hmc step control                                           */
!-----------------------------------------------------------------------

!     /*   istep_hmc:   molecular dynamics steps   */
      call read_int1_MPI ( istep_hmc, '<istep_hmc>', 11, iounit )

!     //   interval of adjusting trial steps
      call read_int1_MPI &
     &   ( istep_adjust_hmc, '<istep_adjust_hmc>', 18, iounit )

!     //   minimum trial steps
      istep_min_hmc = istep_hmc

!     //   maximum trial steps
      call read_int1_MPI &
     &   ( istep_max_hmc, '<istep_max_hmc>', 15, iounit )

!     //   magnifying factor
      call read_int1_MPI &
     &   ( istep_mul_hmc, '<istep_mul_hmc>', 15, iounit )

!     //   lower bound of target acceptance ratio
      call read_real1_MPI &
     &   ( ratio_min_hmc, '<ratio_min_hmc>', 15, iounit )

!     //   upper bound of target acceptance ratio
      call read_real1_MPI &
     &   ( ratio_max_hmc, '<ratio_max_hmc>', 15, iounit )

!-----------------------------------------------------------------------
!     /*   initialize energy correction                               */
!-----------------------------------------------------------------------

      call testfile ( 'step.ini', 8, itest )

      if ( itest .eq. 0 ) then

         open ( iounit, file = 'step.ini' )

         read ( iounit, *, iostat=ierr )
         read ( iounit, *, iostat=ierr ) hamiltonian_cor

         close( iounit )

         if ( ierr .ne. 0 ) hamiltonian_cor = 0.d0

      else

         hamiltonian_cor = 0.d0

      end if

!-----------------------------------------------------------------------
!     /*   number of reference steps for harmonic potential           */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI ( nref_meta, '<nref_meta>', 11, iounit )

!-----------------------------------------------------------------------
!     /*   step size for harmonic potential                           */
!-----------------------------------------------------------------------

      dt_ref_meta = dt/dble(nref_meta)

!-----------------------------------------------------------------------
!     /*   read type of collective variables                          */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<nmeta>', 7, iounit, ierr )

!     /*   read integer   */
      read ( iounit, *, iostat=ierr )

      do k = 1, nmeta

         read ( iounit, *, iostat=ierr ) char_meta

         backspace( iounit )

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            itype_meta(k) = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            itype_meta(k) = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            itype_meta(k) = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            itype_meta(k) = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            itype_meta(k) = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            itype_meta(k) = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            itype_meta(k) = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            itype_meta(k) = 8
         else
            ierr = 1
            exit
         end if

         read ( iounit, '(a)', iostat=ierr ) char_line

         params_meta(1,k) = - huge
         params_meta(2,k) = + huge

         if      ( itype_meta(k) .eq. 1 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 2 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 3 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), l_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), l_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 4 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 5 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 6 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &                    k_meta(k), l_meta(k), &
     &         nu_meta(k,2), mu_meta(k,2), req_meta(k,2), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         nu_meta(k,1), mu_meta(k,1), req_meta(k,1), &
     &                    k_meta(k), l_meta(k), &
     &         nu_meta(k,2), mu_meta(k,2), req_meta(k,2)
            end if
         else if ( itype_meta(k) .eq. 7 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
         else if ( itype_meta(k) .eq. 8 ) then
            read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k), &
     &         params_meta(1,k), params_meta(2,k)
            if ( ierr .ne. 0 ) then
               read ( char_line, *, iostat=ierr ) &
     &         char_meta, i_meta(k), j_meta(k), k_meta(k)
            end if
            if ( ierr .ne. 0 ) exit
            if ( ierr .ne. 0 ) exit
         else
            ierr = 1
            exit
         end if

      end do

!     /*   file close   */
      close( iounit )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <nmeta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_int_1 ( i_meta, nmeta )
      call my_mpi_bcast_int_1 ( j_meta, nmeta )
      call my_mpi_bcast_int_1 ( k_meta, nmeta )
      call my_mpi_bcast_int_1 ( l_meta, nmeta )
      call my_mpi_bcast_int_2 ( nu_meta, nmeta, 2 )
      call my_mpi_bcast_int_2 ( mu_meta, nmeta, 2 )
      call my_mpi_bcast_real_2( req_meta, nmeta, 2 )
      call my_mpi_bcast_int_1 ( itype_meta, nmeta )
      call my_mpi_bcast_real_2( params_meta, 2, nmeta )

!-----------------------------------------------------------------------
!     /*   read gaussian height                                       */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( gh_meta_save, '<gh_meta>', 9, iounit )

!     /*   kelvin ->  au   */
      gh_meta_save = gh_meta_save * boltz

!-----------------------------------------------------------------------
!     /*   read gaussian width                                        */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<gw_meta>', 9, iounit, ierr )

!     /*   width of each cv type   */

      do i = 1, ntype_meta

         read ( iounit, *, iostat=ierr ) char_meta

         backspace( iounit )

         if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &             ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
            exit
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) char_meta, gw_meta_type(k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<gw_meta>', 9, iounit, ierr )

!     /*   width of each cv type   */

         do i = 1, ntype_meta

            read ( iounit, *, iostat=ierr ) char_meta

            backspace( iounit )

            if      ( ( char_meta(1:6) .eq. '1     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( char_meta(1:6) .eq. '2     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( char_meta(1:6) .eq. '3     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( char_meta(1:6) .eq. '4     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( char_meta(1:6) .eq. '5     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( char_meta(1:6) .eq. '6     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( char_meta(1:6) .eq. '7     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( char_meta(1:6) .eq. '8     ' ) .or. &
     &                ( char_meta(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
               exit
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) char_meta, gw_meta_type(k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <gw_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_real_1 ( gw_meta_type, ntype_meta )

!-----------------------------------------------------------------------
!     /*   gaussian width                                             */
!-----------------------------------------------------------------------

      do k = 1, nmeta
         gw_meta_save(k) = gw_meta_type(itype_meta(k))
      end do

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for cvs                     */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_cv_meta, '<time_cv_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_cv_meta = time_cv_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   fictitious mass of cv                                      */
!-----------------------------------------------------------------------

      do k = 1, nmeta

!        /*   distance   */
         f1 = dmeta_max*gw_meta_save(k)

!        /*   velocity   */
         f2 = f1 / time_cv_meta

!        /*   mass   */
         fictmass_meta(k) = 1.d0 / beta / (f2*f2)

      end do

!-----------------------------------------------------------------------
!     /*   read characteristic time scale for force constant          */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI ( time_fc_meta, '<time_fc_meta>', 14, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_fc_meta = time_fc_meta *(1.d-15/au_time)

!-----------------------------------------------------------------------
!     /*   force constant of harmonic term                            */
!-----------------------------------------------------------------------

      do k = 1, nmeta

!        /*   mass   */
         f1 = fictmass_meta(k)

!        /*   angular frequency   */
         f2 = 2.d0*pi / time_fc_meta

         fc_meta(k) = f1 * f2*f2

      end do

!-----------------------------------------------------------------------
!     /*   option for metadynamics                                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_int1_MPI &
     &   ( ioption_meta, '<ioption_meta>', 14, iounit )

!     /*   read value   */
      call read_int1_MPI &
     &   ( joption_meta, '<joption_meta>', 14, iounit )

!-----------------------------------------------------------------------
!     /*   read print inverval for reconstruction of pes              */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_int1_MPI &
     &   ( iprint_rec_meta, '<iprint_rec_meta>', 17, iounit )

!-----------------------------------------------------------------------
!     /*   read parameters for reconstruction of pes                  */
!-----------------------------------------------------------------------

!     /*   master process only   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!     /*   min, max, mesh of each cv type   */

      do i = 1, ntype_meta

         read ( iounit, *, iostat=ierr ) params_char

         backspace( iounit )

         if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIST  ' ) ) then
            k = 1
         else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &             ( params_char(1:6) .eq. 'ANGL  ' ) ) then
            k = 2
         else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIH   ' ) ) then
            k = 3
         else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DIFF  ' ) ) then
            k = 4
         else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &             ( params_char(1:6) .eq. 'CN    ' ) ) then
            k = 5
         else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DCN   ' ) ) then
            k = 6
         else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &             ( params_char(1:6) .eq. 'XYZ   ' ) ) then
            k = 7
         else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &             ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
            k = 8
         else
            ierr = 1
         end if

         if ( ierr .ne. 0 ) exit

         read ( iounit, *, iostat=ierr ) &
     &      params_char, &
     &      params_rec_meta(1,k), &
     &      params_rec_meta(2,k), &
     &      params_rec_meta(3,k)

         if ( ierr .ne. 0 ) exit

      end do

!     /*   file close   */
      close( iounit )

!     /*   if error is found, read default values   */
      if ( ierr .ne. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input_default.dat' )

!        /*   search for tag    */
         call search_tag ( '<params_rec_meta>', 17, iounit, ierr )

!        /*   min, max, mesh of each cv type   */

         do i = 1, ntype_meta

            read ( iounit, *, iostat=ierr ) params_char

            backspace( iounit )

            if      ( ( params_char(1:6) .eq. '1     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIST  ' ) ) then
               k = 1
            else if ( ( params_char(1:6) .eq. '2     ' ) .or. &
     &                ( params_char(1:6) .eq. 'ANGL  ' ) ) then
               k = 2
            else if ( ( params_char(1:6) .eq. '3     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIH   ' ) ) then
               k = 3
            else if ( ( params_char(1:6) .eq. '4     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DIFF  ' ) ) then
               k = 4
            else if ( ( params_char(1:6) .eq. '5     ' ) .or. &
     &                ( params_char(1:6) .eq. 'CN    ' ) ) then
               k = 5
            else if ( ( params_char(1:6) .eq. '6     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DCN   ' ) ) then
               k = 6
            else if ( ( params_char(1:6) .eq. '7     ' ) .or. &
     &                ( params_char(1:6) .eq. 'XYZ   ' ) ) then
               k = 7
            else if ( ( params_char(1:6) .eq. '8     ' ) .or. &
     &                ( params_char(1:6) .eq. 'DXYZ  ' ) ) then
               k = 8
            else
               ierr = 1
            end if

            if ( ierr .ne. 0 ) exit

            read ( iounit, *, iostat=ierr ) &
     &         params_char, &
     &         params_rec_meta(1,k), &
     &         params_rec_meta(2,k), &
     &         params_rec_meta(3,k)

            if ( ierr .ne. 0 ) exit

         end do

!        /*   file close   */
         close( iounit )

!     /*   end   */
      end if

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - keyword <params_rec_meta> is incorrect.'
      end if

!     /*   master process only   */
      end if

!     /*   broadcast   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   check error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine setup_meta_param_MPI', 31 )

!     /*   broadcast   */
      call my_mpi_bcast_real_2 ( params_rec_meta, 3, ntype_meta )

!-----------------------------------------------------------------------
!     /*   time since last gaussian has been added                    */
!-----------------------------------------------------------------------

!     /*   read value   */
      call read_real1_MPI &
     &   ( time_limit_meta, '<time_limit_meta>', 17, iounit )

!     /*   change unit  [fs]  -->  [au]   */
      time_limit_meta = time_limit_meta *(1.d-15/au_time)

!     /*   initialize   */
      time_meta_save(:) = istep_start*dt

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  hills                         */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      ihills_start = 0

!     /*   check if file called `hills.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'hills.ini', 9, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  ihills_start = 1

!-----------------------------------------------------------------------
!     /*   initial/restart conditions:  metadynamics                  */
!-----------------------------------------------------------------------

!     /*   default:  new collective variables with random numbers     */
      icv_start = 0

!     /*   check if file called `cv.ini' exists   */
      if ( myrank .eq. 0 ) then
         call testfile ( 'cv.ini', 6, itest )
      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( itest )

!     /*   if the file exists, restart    */
      if ( itest .eq. 0 )  icv_start = 1

!-----------------------------------------------------------------------
!     /*   scaling factor                                             */
!-----------------------------------------------------------------------

!     /*   read integer   */
      call read_real1_MPI ( dtemp_meta, '<dtemp_meta>', 12, iounit )

      return
      end





!***********************************************************************
      subroutine save_meta_hmc
!***********************************************************************
!=======================================================================
!
!     save data before trial move
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, vx, vy, vz, pot, fx, fy, fz, natom, nbead, &
     &   mbox

      use rehmc_variables, only : &
     &   pot_save, x_save, y_save, z_save, ux_save, uy_save, uz_save, &
     &   vx_save, vy_save, vz_save, fx_save, fy_save, fz_save, &
     &   hamiltonian_bead_save, hamiltonian_bead, mbox_save

      use meta_variables, only : &
     &   pot_meta, pot_ref_meta, rmeta, vmeta, smeta, fmeta, fref_meta, &
     &   nmeta

      use meta_variables, only : &
     &   pot_meta_save, pot_ref_meta_save, rmeta_save, vmeta_save, &
     &   smeta_save, fmeta_save, fref_meta_save, &
     &   ekin_bead, ekin_bead_save, epot_bead, epot_bead_save

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   save data                                                  */
!-----------------------------------------------------------------------

      do i = 1, nbead

         pot_save(i)                = pot(i)

         hamiltonian_bead_save(i)   = hamiltonian_bead(i)

         ekin_bead_save(i)          = ekin_bead(i)
         epot_bead_save(i)          = epot_bead(i)

         do j = 1, natom

            x_save(j,i)                = x(j,i)
            y_save(j,i)                = y(j,i)
            z_save(j,i)                = z(j,i)

            ux_save(j,i)               = ux(j,i)
            uy_save(j,i)               = uy(j,i)
            uz_save(j,i)               = uz(j,i)

            vx_save(j,i)               = vx(j,i)
            vy_save(j,i)               = vy(j,i)
            vz_save(j,i)               = vz(j,i)

            fx_save(j,i)               = fx(j,i)
            fy_save(j,i)               = fy(j,i)
            fz_save(j,i)               = fz(j,i)

         end do

         mbox_save(:,:,i)              = mbox(:,:,i)

      end do

!-----------------------------------------------------------------------
!     /*   save data                                                  */
!-----------------------------------------------------------------------

      do i = 1, nbead

         pot_meta_save(i) = pot_meta(i)

         pot_ref_meta_save(i) = pot_ref_meta(i)

         do j = 1, nmeta

            rmeta_save(j,i) = rmeta(j,i)
            vmeta_save(j,i) = vmeta(j,i)
            smeta_save(j,i) = smeta(j,i)
            fmeta_save(j,i) = fmeta(j,i)

            fref_meta_save(j,i) = fref_meta(j,i)

         end do

      end do

      return
      end





!***********************************************************************
      subroutine recover_meta_hmc( i )
!***********************************************************************
!=======================================================================
!
!     save data before trial move
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, vx, vy, vz, pot, fx, fy, fz, natom, mbox

      use rehmc_variables, only : &
     &   pot_save, x_save, y_save, z_save, ux_save, uy_save, uz_save, &
     &   vx_save, vy_save, vz_save, fx_save, fy_save, fz_save, &
     &   hamiltonian_bead_save, hamiltonian_bead, mbox_save

      use meta_variables, only : &
     &   pot_meta, pot_ref_meta, rmeta, vmeta, smeta, fmeta, fref_meta, &
     &   nmeta

      use meta_variables, only : &
     &   pot_meta_save, pot_ref_meta_save, rmeta_save, vmeta_save, &
     &   smeta_save, fmeta_save, fref_meta_save, &
     &   ekin_bead, ekin_bead_save, epot_bead, epot_bead_save

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   save data                                                  */
!-----------------------------------------------------------------------

      pot(i)                = pot_save(i)

      hamiltonian_bead(i)   = hamiltonian_bead_save(i)

      ekin_bead(i)          = ekin_bead_save(i)
      epot_bead(i)          = epot_bead_save(i)

      do j = 1, natom

         x(j,i)                = x_save(j,i)
         y(j,i)                = y_save(j,i)
         z(j,i)                = z_save(j,i)

         ux(j,i)               = ux_save(j,i)
         uy(j,i)               = uy_save(j,i)
         uz(j,i)               = uz_save(j,i)

         vx(j,i)               = vx_save(j,i)
         vy(j,i)               = vy_save(j,i)
         vz(j,i)               = vz_save(j,i)

         fx(j,i)               = fx_save(j,i)
         fy(j,i)               = fy_save(j,i)
         fz(j,i)               = fz_save(j,i)

      end do

      mbox(:,:,i)              = mbox_save(:,:,i)

!-----------------------------------------------------------------------
!     /*   save data                                                  */
!-----------------------------------------------------------------------

      pot_meta(i) = pot_meta_save(i)

      pot_ref_meta(i) = pot_ref_meta_save(i)

      do j = 1, nmeta

         rmeta(j,i) = rmeta_save(j,i)
         vmeta(j,i) = vmeta_save(j,i)
         smeta(j,i) = smeta_save(j,i)
         fmeta(j,i) = fmeta_save(j,i)

         fref_meta(j,i) = fref_meta_save(j,i)

      end do

      return
      end





!***********************************************************************
      subroutine standard_meta_hmc_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature'' in metadynamics.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, vux, vuy, vuz, temp, boltz, hamiltonian, potential, &
     &   hamiltonian_cor, dt, au_time, iprint_std, natom, nbead, &
     &   iounit, iounit_std, istep, istep_start, char_date, myrank

      use meta_variables, only : &
     &   fictmass_meta, vmeta, ekin_meta, pot_ref_meta, ebath_sys, &
     &   potential_ref_meta, ekin_sys, potential_meta, pot_meta, &
     &   ebath_meta, pot_lifted, nmeta

      use rehmc_variables, only : &
     &   ratio_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, itest

      real(8) :: dt_fs

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         if ( myrank .ne. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open( iounit_std, file = 'standard.out', access = 'append' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step dt [fs]   ratio  potential [au]  temp [K]  ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

      end if

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   kinetic energy                                             */
!-----------------------------------------------------------------------

      ekin_sys = 0.d0

      do j = 1, nbead
      do i = 1, natom
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vux(i,j)*vux(i,j)
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vuy(i,j)*vuy(i,j)
         ekin_sys = ekin_sys + 0.5d0*fictmass(i,j)*vuz(i,j)*vuz(i,j)
      end do
      end do

      temp = 2.d0*ekin_sys/dble(3*natom*nbead)/boltz

!-----------------------------------------------------------------------
!     /*   kinetic energy of collective variables                     */
!-----------------------------------------------------------------------

      ekin_meta = 0.d0

      do j = 1, nbead
      do i = 1, nmeta
         ekin_meta = ekin_meta &
     &             + 0.5d0*fictmass_meta(i)*vmeta(i,j)*vmeta(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ebath_sys  =  thermostats attached to system               */
!-----------------------------------------------------------------------

      ebath_sys = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_meta  =  thermostats to collective variables         */
!-----------------------------------------------------------------------

      ebath_meta = 0.d0

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      potential_meta = 0.d0

      do i = 1, nbead
         potential_meta = potential_meta + pot_meta(i)
      end do

      potential_ref_meta = 0.d0

      do i = 1, nbead
         potential_ref_meta = potential_ref_meta + pot_ref_meta(i)
      end do

!-----------------------------------------------------------------------
!     /*   total energy                                               */
!-----------------------------------------------------------------------

      hamiltonian = ekin_sys + potential + ekin_meta + potential_meta &
     &            + potential_ref_meta + ebath_sys + ebath_meta &
     &            - pot_lifted + hamiltonian_cor

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

!     /*   print interval   */
      if ( ( myrank .eq. 0) .and. (mod(istep,iprint_std) .eq. 0) ) then

!        /*   wall clock time   */
         call getdate

!        /*   dt in fs   */
         dt_fs = dt / (1.d-15/au_time)

!        /*   if initial step   */
         if ( istep .eq. istep_start ) then

!           /*   open   */
            open( iounit_std, file = 'standard.out', access = 'append' )

!           /*   output   */
            write( iounit_std, '(i8,f8.3,a8,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ' -------', potential, temp, char_date

!           /*   close   */
            close( iounit_std )

!           /*   output   */
            write( 6, '(i8,f8.3,a8,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ' -------', potential, temp, char_date

!           /*   make sure output   */
            flush( 6 )

!        /*   if not initial step   */
         else

!           /*   open   */
            open( iounit_std, file = 'standard.out', access = 'append' )

            write( iounit_std, '(i8,f8.3,f8.5,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ratio_hmc, potential, temp, char_date

!           /*   close   */
            close( iounit_std )

!           /*   output   */
            write( 6, '(i8,f8.3,f8.5,f16.8,f10.2,2x,a26)' ) &
     &         istep, dt_fs, ratio_hmc, potential, temp, char_date

!           /*   make sure output   */
            flush( 6 )

!        /*   end of if statement   */
         end if

!     /*   print interval   */
      end if

      return
      end





!***********************************************************************
      subroutine backup_meta_hmc_MPI
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep_end, iprint_rest, iounit, nstep, myrank

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         if ( iprint_rest .le. 0 ) then
            return
         else
            continue
         end if
      else if ( istep_end .eq. nstep ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

!     /*   system position   */
      call restart_position_MPI( 4 )

!     /*   system velocity   */
      call restart_velocity_MPI( 4 )

!     /*   cv position and velocity   */
      call restart_cv_meta_MPI( 2 )

!     /*   hills   */
      call restart_hills_meta_MPI( 2 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         open ( iounit, file = 'step.ini' )
            write ( iounit, '(i8)' ) istep_end
         close( iounit )
      end if

!-----------------------------------------------------------------------
!     /*   print final geometry                                       */
!-----------------------------------------------------------------------

      call print_final_xyz_MPI

      return
      end
