!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, Y. Kataoka
!      Last updated:    Dec 24, 2022 by M. Shiga
!      Description:     Matsubara dynamics
!
!///////////////////////////////////////////////////////////////////////
!=======================================================================
!
!     notation of modes:
!
!        Althorpe      PIMD
!        T_ni      =   uinv(j,i)   = tnminv(j,i)*sqrt(N)
!        T_in      =   u(i,j)      = tnm(i,j)/sqrt(N)
!
!     mode index:
!
!        Althorpe:  n = 0, -1, +1, -2, +2, ..., -(M-1)/2, -(M+1)/2
!        PIMD:      j = 1,  2,  3,  4,  5, ...,      M-1,        M
!
!=======================================================================
!***********************************************************************
      subroutine setup_matsubara
!***********************************************************************
!=======================================================================
!
!     set up matsubara dynamics
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ivel_start, ipos_start, iounit, nstep, mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   parameters for Suzuki-Yoshida propagator ( order = nys )   */
!-----------------------------------------------------------------------

      call suzuki_yoshida

!-----------------------------------------------------------------------
!     /*   bead parameter for matsubara dynamics                      */
!-----------------------------------------------------------------------

!     /*   read from file   */
      call read_int1( mbead_matsubara, '<mbead_matsubara>', 17, iounit )

!-----------------------------------------------------------------------
!     /*   path integral parameters                                   */
!-----------------------------------------------------------------------

      call setpiparams

!-----------------------------------------------------------------------
!     /*   get normal mode transformation matrix                      */
!-----------------------------------------------------------------------

      call nm_matrix

!-----------------------------------------------------------------------
!     /*   real and fictitous masses of normal modes                  */
!-----------------------------------------------------------------------

      call init_mass_rpmd

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file, and               */
!     /*   non-centroid modes are in Gaussian distribution            */
      if     ( ipos_start .eq. 0 ) then
         call init_centroid
         call init_mode
!     /*   read normal mode position                                  */
      else if ( ipos_start .eq. 1 ) then
         call restart_position( 1 )
!     /*   read Cartesian position                                    */
      else if ( ipos_start .eq. 2 ) then
         call restart_position( 2 )
      else
         call error_handling( 1, 'subroutine setup_matsubara', 26)
      end if

!     /*   Maxwell distribution of velocities                         */
      if     ( ivel_start .eq. 0 ) then
         call init_velocity
      else if ( ivel_start .eq. 1 ) then
!        /*   read normal mode momentum (scaled)                      */
         call restart_velocity( 1 )
      else if ( ivel_start .eq. 2 ) then
!        /*   read Cartesian momentum                                 */
         call restart_velocity( 2 )
      else
         call error_handling( 1, 'subroutine setup_matsubara', 26)
      end if

!-----------------------------------------------------------------------
!     /*   subtract translational and rotational part of velocities   */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         call backup_pimd_nve

         stop

      end if

      return
      end





!***********************************************************************
      subroutine setup_matsubara_1d
!***********************************************************************

!     //   initialize local variables
      implicit none

!     /*   setup matsubara dynamics   */
      call setup_matsubara

!     /*   set y and z position to zero   */
      call resetposition_1d
      call nm_trans_matsubara( 0 )

!     /*   set y and z velocity to zero   */
      call resetvelocity_1d
      call nm_trans_velocity_matsubara( 0 )

      return
      end





!***********************************************************************
      subroutine matsubaracycle_1d
!***********************************************************************
!=======================================================================
!
!     The matsubara dynamics cycle.
!
!=======================================================================

!     /*   shared variables   */
      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!     /*   local variables   */
      implicit none

!     /*   initialize step   */
      istep = istep_start
      istep_end = istep

!     /*   normal mode position -> Cartesian position   */
      call nm_trans_matsubara( 0 )

!     /*   get interatomic forces   */
      call getforce

!     /*   Cartesian force -> normal mode force   */
      call nm_trans_force_matsubara( 1 )

!     /*   calculate the hamiltonian and temperature   */
      call standard_matsubara

!     /*   do some analysis   */
      call analysis_matsubara( 1 )
      call analysis_matsubara( 2 )

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   update all the normal mode coordinates   */
         call update_pos_matsubara

!        /*   set y and z position to zero   */
         call resetposition_1d

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode position -> Cartesian position   */
         call nm_trans_matsubara(0)

!        /*   get interatomic forces   */
         call getforce

!        /*   Cartesian force -> normal mode force   */
         call nm_trans_force_matsubara(1)

!        /*   update the velocities by interatomic forces   */
         call update_vel

!        /*   set y and z velocity to zero   */
         call resetvelocity_1d

!        /*   normal mode velocity -> Cartesian velocity   */
         call nm_trans_velocity_matsubara(0)

!        /*   calculate the hamiltonian and temperature   */
         call standard_matsubara

!        /*   do some analysis   */
         call analysis_matsubara( 2 )

!        /*   output restart   */
         call backup_pimd_nve

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine analysis_matsubara( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit_avg, iounit

      use analysis_variables, only : &
     &   iprint_trj, iprint_dip, iprint_xyz, iprint_xsf, iprint_dcd, &
     &   iprint_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest, ioption

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!-----------------------------------------------------------------------
!        /*   step intervals of analysis                              */
!-----------------------------------------------------------------------

         call read_int1 ( iprint_trj,  '<iprint_trj>',  12, iounit )
         call read_int1 ( iprint_dip,  '<iprint_dip>',  12, iounit )
         call read_int1 ( iprint_xyz,  '<iprint_xyz>',  12, iounit )
         call read_int1 ( iprint_xsf,  '<iprint_xsf>',  12, iounit )
         call read_int1 ( iprint_dcd,  '<iprint_dcd>',  12, iounit )
         call read_int1 &
     &        ( iprint_matsubara, '<iprint_matsubara>', 18, iounit )

!-----------------------------------------------------------------------
!        /*   check if file called `averages.ini' exists              */
!-----------------------------------------------------------------------

         call testfile ( 'averages.ini', 12, itest, iounit )

!-----------------------------------------------------------------------
!        /*   if the file does not exist, initial start.              */
!-----------------------------------------------------------------------

         if ( itest .eq. 1 ) then

            call analysis_trj       ( 0 )
            call analysis_dip       ( 0 )
            call analysis_xyz       ( 0 )
            call analysis_xsf       ( 0 )
            call analysis_dcd       ( 0 )
            call analysis_phase     ( 0 )

!-----------------------------------------------------------------------
!        /*   if the file exists, restart.                            */
!-----------------------------------------------------------------------

         else

            open ( iounit_avg, file = 'averages.ini' )

            call analysis_trj       ( 1 )
            call analysis_dip       ( 1 )
            call analysis_xyz       ( 1 )
            call analysis_xsf       ( 1 )
            call analysis_dcd       ( 1 )
            call analysis_phase     ( 1 )

            close( iounit_avg )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         call analysis_trj       ( 2 )
         call analysis_dip       ( 2 )
         call analysis_xyz       ( 2 )
         call analysis_xsf       ( 2 )
         call analysis_dcd       ( 2 )
         call analysis_phase     ( 2 )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         open ( iounit_avg, file = 'averages.ini' )

         call analysis_trj       ( 3 )
         call analysis_dip       ( 3 )
         call analysis_xyz       ( 3 )
         call analysis_xsf       ( 3 )
         call analysis_dcd       ( 3 )
         call analysis_phase     ( 3 )

         close( iounit_avg )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_phase( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, fux, fuy, fuz, pi, hbar, beta, &
     &   hamiltonian, fictmass, natom, mbead_matsubara, dnmmass, &
     &   physmass, omega_p, iounit, istep, nbead

      use analysis_variables, only : &
     &   iprint_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: joption = 0

      integer :: i, j, ioption

      real(8) :: omega, theta, phase, cos_phase, sin_phase, dtheta

      real(8) :: weight

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_matsubara .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 0:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         open ( iounit, file = 'matsubara.out', access='append' )

         write( iounit, '(a)' ) &
     &      '------------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------'

         write( iounit, '(a)' ) &
     &      '    step     hamiltonian       theta' // &
     &      '   dtheta/dt cos_phase sin_phase' // &
     &      '        weight'
         write( iounit, '(a)' ) &
     &      '------------------------------------' // &
     &      '--------------------------------' // &
     &      '--------------'

         close( iounit )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 1:  restart                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  matsubara phase and print data               */
!-----------------------------------------------------------------------

!     /*   mode 1 is the centroid variable   */
!     /*   modes (2, 3), (4, 5), ... are degenerate modes   */

      if ( ioption .eq. 2 ) then

!        /*   initialize   */
         theta = 0.d0
         dtheta = 0.d0

!        /*   even-number non-centroid modes (omega is positive)   */

         do i = 2, min(mbead_matsubara-1,nbead-1), 2

            if ( joption .eq. 0 ) then
               omega = - 2.d0 * pi * dble(i/2) / ( beta * hbar )
            else
               omega = - sqrt(dnmmass(1,i)/physmass(1)) * omega_p
            end if

            do j = 1, natom

               theta = theta &
     &               + fictmass(j,i)*vux(j,i)*omega*ux(j,i+1)
               theta = theta &
     &               + fictmass(j,i)*vuy(j,i)*omega*uy(j,i+1)
               theta = theta &
     &               + fictmass(j,i)*vuz(j,i)*omega*uz(j,i+1)

               dtheta = dtheta &
     &                + fictmass(j,i)*vux(j,i)*omega*vux(j,i+1) &
     &                + fux(j,i)*omega*ux(j,i+1)
               dtheta = dtheta &
     &                + fictmass(j,i)*vuy(j,i)*omega*vuy(j,i+1) &
     &                + fuy(j,i)*omega*uy(j,i+1)
               dtheta = dtheta &
     &                + fictmass(j,i)*vuz(j,i)*omega*vuz(j,i+1) &
     &                + fuz(j,i)*omega*uz(j,i+1)

            end do

         end do

!        /*   odd-number non-centroid modes (omega is positive)  */

         do i = 3, min(mbead_matsubara,nbead), 2

            if ( joption .eq. 0 ) then
               omega = + 2.d0 * pi * dble((i-1)/2) / ( beta * hbar )
            else
               omega = sqrt(dnmmass(1,i)/physmass(1)) * omega_p
            end if

            do j = 1, natom

               theta = theta &
     &               + fictmass(j,i)*vux(j,i)*omega*ux(j,i-1)
               theta = theta &
     &               + fictmass(j,i)*vuy(j,i)*omega*uy(j,i-1)
               theta = theta &
     &               + fictmass(j,i)*vuz(j,i)*omega*uz(j,i-1)

               dtheta = dtheta &
     &                + fictmass(j,i)*vux(j,i)*omega*vux(j,i-1) &
     &                + fux(j,i)*omega*ux(j,i-1)
               dtheta = dtheta &
     &                + fictmass(j,i)*vuy(j,i)*omega*vuy(j,i-1) &
     &                + fuy(j,i)*omega*uy(j,i-1)
               dtheta = dtheta &
     &                + fictmass(j,i)*vuz(j,i)*omega*vuz(j,i-1) &
     &                + fuz(j,i)*omega*uz(j,i-1)

            end do

         end do

!        /*   phase factor   */

         phase = beta * theta

         cos_phase = cos(phase)
         sin_phase = sin(phase)

         weight = cos_phase * exp( - beta*hamiltonian )

!        /*   print output   */

         if ( mod(istep,iprint_matsubara) .eq. 0 ) then

            open ( iounit, file = 'matsubara.out', access='append' )

            write( iounit, '(i8,f16.8,2f12.8,2f10.6,e14.6)' ) &
     &         istep, hamiltonian, theta, dtheta, &
     &         cos_phase, sin_phase, weight

            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end





!***********************************************************************
      subroutine standard_matsubara
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian'' and ``temperature''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   temp, ekin, boltz, qkin, ekin, potential, hamiltonian, &
     &   hamiltonian_sys, ebath_cent, ebath_mode, ndof, iprint_std

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      call standard_init( iset )

      if ( iprint_std .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      call kinetic_energy_matsubara

      temp = 2.d0*ekin/dble(ndof)/boltz

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebath_cent  =  thermostats attached to centroid            */
!-----------------------------------------------------------------------

      ebath_cent = 0.d0

!-----------------------------------------------------------------------
!     /*   ebath_mode  =  thermostats attached to non-centroid        */
!-----------------------------------------------------------------------

      ebath_mode = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebath_mode + ebath_cent

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      call standard_output

      return
      end





!***********************************************************************
      subroutine kinetic_energy_matsubara
!***********************************************************************
!=======================================================================
!
!     calculate ekin = fictitious kinetic energy of the system
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ekin, fictmass, vux, vuy, vuz, ekin, natom, mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do imode = 1, mbead_matsubara
      do iatom = 1, natom
         ekin = ekin &
     &        + fictmass(iatom,imode)*vux(iatom,imode)*vux(iatom,imode) &
     &        + fictmass(iatom,imode)*vuy(iatom,imode)*vuy(iatom,imode) &
     &        + fictmass(iatom,imode)*vuz(iatom,imode)*vuz(iatom,imode)
      end do
      end do

      ekin = 0.5d0*ekin

      return
      end





!***********************************************************************
      subroutine update_pos_matsubara
!***********************************************************************
!-----------------------------------------------------------------------
!
!     update normal mode coordinates.
!
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, vux, vuy, vuz, dt, natom, mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer i, j

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do j = 1, mbead_matsubara
      do i = 1, natom
         ux(i,j) = ux(i,j) + dt*vux(i,j)
         uy(i,j) = uy(i,j) + dt*vuy(i,j)
         uz(i,j) = uz(i,j) + dt*vuz(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      call pbc_cent

      return
      end





!***********************************************************************
      subroutine update_vel_matsubara
!***********************************************************************
!=======================================================================
!
!     update normal mode velocities by interatomic forces
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vux, vuy, vuz, fux, fuy, fuz, dt, fictmass, natom, &
     &   mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer i, j

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do j = 1, mbead_matsubara
      do i = 1, natom
         vux(i,j) = vux(i,j) + 0.5d0*dt*fux(i,j)/fictmass(i,j)
         vuy(i,j) = vuy(i,j) + 0.5d0*dt*fuy(i,j)/fictmass(i,j)
         vuz(i,j) = vuz(i,j) + 0.5d0*dt*fuz(i,j)/fictmass(i,j)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine nm_trans_matsubara(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!
!     itrans = 1 :   backward transformation
!                    u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, tnm, tnminv, natom, nbead, mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            x(k,i) = 0.d0
            y(k,i) = 0.d0
            z(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, mbead_matsubara
            x(k,i) = x(k,i) + tnm(i,j)*ux(k,j)
            y(k,i) = y(k,i) + tnm(i,j)*uy(k,j)
            z(k,i) = z(k,i) + tnm(i,j)*uz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, mbead_matsubara
         do k = 1, natom
            ux(k,i) = 0.d0
            uy(k,i) = 0.d0
            uz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, mbead_matsubara
         do j = 1, nbead
            ux(k,i) = ux(k,i) + tnminv(i,j)*x(k,j)
            uy(k,i) = uy(k,i) + tnminv(i,j)*y(k,j)
            uz(k,i) = uz(k,i) + tnminv(i,j)*z(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_velocity_matsubara(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic coordinates.
!
!     itrans = 0 :   forward transformation
!                    vr(i) = vr(i) + sum_j tnm(i,j)*vu(j)
!
!     itrans = 1 :   backward transformation
!                    vu(i) = vu(i) + sum_j tnminv(i,j)*vr(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, vux, vuy, vuz, tnm, tnminv, natom, nbead, &
     &   mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            vx(k,i) = 0.d0
            vy(k,i) = 0.d0
            vz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, mbead_matsubara
            vx(k,i) = vx(k,i) + tnm(i,j)*vux(k,j)
            vy(k,i) = vy(k,i) + tnm(i,j)*vuy(k,j)
            vz(k,i) = vz(k,i) + tnm(i,j)*vuz(k,j)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, mbead_matsubara
         do k = 1, natom
            vux(k,i) = 0.d0
            vuy(k,i) = 0.d0
            vuz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, mbead_matsubara
         do j = 1, nbead
            vux(k,i) = vux(k,i) + tnminv(i,j)*vx(k,j)
            vuy(k,i) = vuy(k,i) + tnminv(i,j)*vy(k,j)
            vuz(k,i) = vuz(k,i) + tnminv(i,j)*vz(k,j)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_trans_force_matsubara(itrans)
!***********************************************************************
!=======================================================================
!
!     Cartesian <--> normal mode transformation of atomic forces.
!
!     itrans = 0 :   forward transformation
!                    r(i) = r(i) + sum_j tnm(i,j)*u(j)
!                    fr(i) = fr(i) + sum_j fu(j)*tnminv(j,i)
!
!     itrans = 1 :   backward transformation
!                    fu(i) = fu(i) + sum_j fr(j)*tnm(j,i)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, fux, fuy, fuz, tnm, tnminv, natom, nbead, &
     &   mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, itrans, j, k

!-----------------------------------------------------------------------
!     /*   normal mode  -->   Cartesian                               */
!-----------------------------------------------------------------------

      if ( itrans .eq. 0 ) then

         do i = 1, nbead
         do k = 1, natom
            fx(k,i) = 0.d0
            fy(k,i) = 0.d0
            fz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, nbead
         do j = 1, mbead_matsubara
            fx(k,i) = fx(k,i) + fux(k,j)*tnminv(j,i)
            fy(k,i) = fy(k,i) + fuy(k,j)*tnminv(j,i)
            fz(k,i) = fz(k,i) + fuz(k,j)*tnminv(j,i)
         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   Cartesian   -->   normal mode                              */
!-----------------------------------------------------------------------

      else if ( itrans .eq. 1 ) then

         do i = 1, mbead_matsubara
         do k = 1, natom
            fux(k,i) = 0.d0
            fuy(k,i) = 0.d0
            fuz(k,i) = 0.d0
         end do
         end do

         do k = 1, natom
         do i = 1, mbead_matsubara
         do j = 1, nbead
            fux(k,i) = fux(k,i) + fx(k,j)*tnm(j,i)
            fuy(k,i) = fuy(k,i) + fy(k,j)*tnm(j,i)
            fuz(k,i) = fuz(k,i) + fz(k,j)*tnm(j,i)
         end do
         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine nm_matrix_matsubara_test
!***********************************************************************
!=======================================================================
!
!     the matrix of the normal mode transformation
!
!     tnm    :     u -> r      r(i) = r(i) + sum_j tnm(i,j)*u(j)
!     tnminv :     r -> u      u(i) = u(i) + sum_j tnminv(i,j)*r(j)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   u, uinv, tnm, tnminv, dnmmass, physmass, pi, nbead, natom, &
     &   mbead_matsubara

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, n

      real(8) :: const, di, dj, dp, dum, sqp, sqpinv, dnorm

!-----------------------------------------------------------------------
!     /*   make trm and trminv                                        */
!-----------------------------------------------------------------------

      tnminv(:,:) = 0.d0
      tnm(:,:)    = 0.d0

      do l = 1, nbead

         do i = 1, (mbead_matsubara-1)/2

            n = i - (mbead_matsubara+1)/2

            tnminv(i,l) &
     &         = sqrt(2.d0)/dble(nbead) &
     &         * cos( 2.d0*pi*dble(l*n)/dble(nbead) )

         end do

         tnminv((mbead_matsubara+1)/2,l) = 1.d0 / dble(nbead)

         do i = (mbead_matsubara+1)/2+1, mbead_matsubara

            n = i - (mbead_matsubara+1)/2

            tnminv(i,l) = sqrt(2.d0)/dble(nbead) &
     &                  * sin( 2.d0*pi*dble(l*n)/dble(nbead) )

         end do

      end do

      do n = 1, mbead_matsubara
      do l = 1, nbead
         tnm(l,n) = dble(nbead)*tnminv(n,l)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dp     = dble(nbead)
      sqp    = sqrt(dp)
      sqpinv = 1.d0/sqrt(dp)
      dnorm  = sqrt(2.d0/dp)

!-----------------------------------------------------------------------
!     /*   u  =  unitary matrix that diagnalizes the spring matrix    */
!-----------------------------------------------------------------------

      dum = -1.d0
      do i = 1, nbead
         u(i,1)     = sqpinv
         u(i,nbead) = dum*sqpinv
         dum = dum*(-1.d0)
      end do

      do i = 1, (nbead-2)/2
         di = dble(i)
         do j = 1, nbead
            dj = dble(j)
            u(j,2*i)   = dnorm*cos(2.d0*pi*di*dj/dp)
            u(j,2*i+1) = dnorm*sin(2.d0*pi*di*dj/dp)
         end do
      end do

      do i = 1, nbead
      do j = 1, nbead
         uinv(j,i) = u(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   make trm and trminv                                        */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, mbead_matsubara

         n = (2*mod(j,2)-1)*j/2 + (mbead_matsubara+1)/2
         if ( j .eq. 1 ) n = (mbead_matsubara+1)/2

         write( 6, '(3i4,4f16.8)' ) &
     &      i, j, n, tnm(i,n)/sqp, u(i,j), tnminv(n,i)*sqp, uinv(j,i)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   real masses of normal modes:                               */
!     /*      dnmmass = eigenvalues of A matrix times nbead.          */
!-----------------------------------------------------------------------

!     /*   centroid   */

      do j = 1, natom
         dnmmass(j,1) = 0.d0
      end do

!     /*   non-centroid   */

      dp = dble(nbead)

      do j = 1, natom
         do i = 1, (nbead-1)/2
            di = dble(i)
            const = 2.d0*(1.d0-cos(2.d0*pi*di/dp))*dp*physmass(j)
            dnmmass(j,2*i)   = const
            dnmmass(j,2*i+1) = const
         end do
      end do

      stop
      end

