!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine judge_remc_t
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, nbead

      use rehmc_variables, only : &
     &   pot_save, ratio_mc, beta_bead, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   naccept_mc, nreject_mc

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ranf1, bfactor, randomno

      integer :: m

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     //   judge acceptance
!-----------------------------------------------------------------------

!     //   loop of replicas
      do m = 1, nbead

!        //   exponent
         bfactor = beta_bead(m) * ( pot(m) - pot_save(m) )

!        //   if exponent is not too large
         if ( bfactor .lt. 75.d0 ) then

!           //   exponent is smaller than 0
            if ( bfactor .le. 0.d0 ) then

!              /*   accepted   */
               naccept_mc = naccept_mc + 1

!           //   exponent is larger than 0
            else

!              //   random number
               randomno = ranf1()

!              //   if probability is larger
               if ( exp(-bfactor) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_mc = naccept_mc + 1

!              //   if probability is smaller
               else

!                 /*   rejected   */
                  nreject_mc = nreject_mc + 1

!                 /*   recover saved data   */
                  call recover_remc( m )

!              //   end of if statement
               end if

!           //   end of if statement
            end if

!        //   if exponent is too large
         else

!           /*   rejected   */
            nreject_mc = nreject_mc + 1

!           /*   recover saved data   */
            call recover_remc( m )

!        //   end of if statement
         end if


!     //   loop of replicas
      end do

!     /*   acceptance ratio   */
      ratio_mc = dble(naccept_mc) / dble(naccept_mc+nreject_mc)

      return
      end





!***********************************************************************
      subroutine judge_remc_t_dual
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, nbead

      use rehmc_variables, only : &
     &   pot_save, pot_low_save, ratio_mc, beta_bead, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, naccept_mc, &
     &   nreject_mc

      use dual_variables, only : &
     &   pot_low

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ranf1, bfactor, randomno

      integer :: m

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     //   judge acceptance
!-----------------------------------------------------------------------

!     //   loop of replicas
      do m = 1, nbead

!        //   exponent
         bfactor = beta_bead(m) &
     &      * ( pot(m) - pot_save(m) - pot_low(m) + pot_low_save(m) )

!        //   if exponent is not too large
         if ( bfactor .lt. 75.d0 ) then

!           //   exponent is smaller than 0
            if ( bfactor .le. 0.d0 ) then

!              /*   accepted   */
               naccept_mc = naccept_mc + 1

!           //   exponent is larger than 0
            else

!              //   random number
               randomno = ranf1()

!              //   if probability is larger
               if ( exp(-bfactor) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_mc = naccept_mc + 1

!              //   if probability is smaller
               else

!                 /*   rejected   */
                  nreject_mc = nreject_mc + 1

!                 /*   recover saved data   */
                  call recover_remc( m )

!              //   end of if statement
               end if

!           //   end of if statement
            end if

!        //   if exponent is too large
         else

!           /*   rejected   */
            nreject_mc = nreject_mc + 1

!           /*   recover saved data   */
            call recover_remc( m )

!        //   end of if statement
         end if


!     //   loop of replicas
      end do

!     /*   acceptance ratio   */
      ratio_mc = dble(naccept_mc) / dble(naccept_mc+nreject_mc)

      return
      end





!***********************************************************************
      subroutine judge_remc_sub
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, nbead

      use rehmc_variables, only : &
     &   x_sub_save, y_sub_save, z_sub_save, pot_sub_save, &
     &   ratio_mc_sub, beta_bead, naccept_mc_sub, nreject_mc_sub

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ranf1, bfactor, randomno

      integer :: m

!-----------------------------------------------------------------------
!     //   judge acceptance
!-----------------------------------------------------------------------

!     //   loop of replicas
      do m = 1, nbead

!        //   exponent
         bfactor = beta_bead(m) * ( pot(m) - pot_sub_save(m) )

!        //   if exponent is not too large
         if ( bfactor .lt. 75.d0 ) then

!           //   exponent is smaller than 0
            if ( bfactor .le. 0.d0 ) then

!              /*   accepted   */
               naccept_mc_sub = naccept_mc_sub + 1

!           //   exponent is larger than 0
            else

!              //   random number
               randomno = ranf1()

!              //   if probability is larger
               if ( exp(-bfactor) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_mc_sub = naccept_mc_sub + 1

!              //   if probability is smaller
               else

!                 /*   rejected   */
                  nreject_mc_sub = nreject_mc_sub + 1

!                 /*   recover saved data   */
                  x(:,m) = x_sub_save(:,m)
                  y(:,m) = y_sub_save(:,m)
                  z(:,m) = z_sub_save(:,m)
                  pot(m) = pot_sub_save(m)

!              //   end of if statement
               end if

!           //   end of if statement
            end if

!        //   if exponent is too large
         else

!           /*   rejected   */
            nreject_mc_sub = nreject_mc_sub + 1

!           /*   recover saved data   */
            x(:,m) = x_sub_save(:,m)
            y(:,m) = y_sub_save(:,m)
            z(:,m) = z_sub_save(:,m)
            pot(m) = pot_sub_save(m)

!        //   end of if statement
         end if


!     //   loop of replicas
      end do

!     /*   acceptance ratio   */
      ratio_mc_sub = dble(naccept_mc_sub) &
     &            / dble(naccept_mc_sub+nreject_mc_sub)

      return
      end
