!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     replica exchange hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine judge_hmc_rehmc
!***********************************************************************
!=======================================================================
!
!     metropolis step for hybrid monte carlo.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, nbead

      use rehmc_variables, only : &
     &   hamiltonian_bead, hamiltonian_bead_save, bfactor_bead, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   beta_bead, ratio_hmc, naccept_hmc, nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of each bead                                    */
!-----------------------------------------------------------------------

      do i = 1, nbead

!        /*   energy difference   */
         dh = hamiltonian_bead(i) - hamiltonian_bead_save(i)

!        /*   beta times energy difference   */
         bfactor_bead(i) = beta_bead(i) * dh

!        /*   start judge acceptance   */
         if ( bfactor_bead(i) .lt. 75.d0 ) then

            if ( bfactor_bead(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_hmc = naccept_hmc + 1

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_bead(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_hmc = naccept_hmc + 1

               else

!                 /*   rejected   */
                  nreject_hmc = nreject_hmc + 1

!                 /*   recover saved data   */
                  call recover_rehmc( i )

               end if

            end if

         else

!           /*   rejected   */
            nreject_hmc = nreject_hmc + 1

!           /*   recover saved data   */
            call recover_rehmc( i )

!        /*   end judge acceptance   */
         end if

      end do

!     /*   acceptance ratio   */
      if ( naccept_hmc .eq. 0 ) then
         ratio_hmc = 0.d0
      else
         ratio_hmc = dble(naccept_hmc) / dble(naccept_hmc+nreject_hmc)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of each bead                                      */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   energy   */
      call getenergy_rehmc

!     /*   save data   */
      call save_rehmc

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_rem_rehmc
!***********************************************************************
!=======================================================================
!
!     metropolis step for replica echange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, nbead

      use rehmc_variables, only : &
     &   hamiltonian_bead, beta_bead, bfactor_rem, ratio_rem, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   naccept_rem, nreject_rem

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh, db

!     /*   integers   */
      integer :: niter_rem

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start exchange                                             */
!-----------------------------------------------------------------------

!     /*   number of iterations   */
      niter_rem = nbead/2

!     /*   iterations   */
      do j = 1, niter_rem

!        /*   random integer from 1 to (nbead-1)   */
         i = int( ranf1() * dble(nbead-1) ) + 1

!        /*   energy difference   */
         dh = hamiltonian_bead(i+1) - hamiltonian_bead(i)

!        /*   beta difference   */
         db = beta_bead(i+1) - beta_bead(i)

!        /*   beta times energy difference   */
         bfactor_rem(i) = - db * dh

!        /*   judge acceptance   */
         if ( bfactor_rem(i) .lt. 75.d0 ) then

            if ( bfactor_rem(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_rem = naccept_rem + 1

!              /*   exchange data   */
               call exchange_rem_rehmc( i, i+1 )

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_rem(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_rem = naccept_rem + 1

!                 /*   exchange data   */
                  call exchange_rem_rehmc( i, i+1 )

               else

!                 /*   rejected   */
                  nreject_rem = nreject_rem + 1

               end if

            end if

         else

!           /*   rejected   */
            nreject_rem = nreject_rem + 1

         end if

!     /*   iterations   */
      end do

!     /*   acceptance ratio   */
      if ( naccept_rem .eq. 0 ) then
         ratio_rem = 0.d0
      else
         ratio_rem = dble(naccept_rem) / dble(naccept_rem+nreject_rem)
      end if

!-----------------------------------------------------------------------
!     /*   end exchange                                               */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   energy   */
      call getenergy_rehmc

!     /*   save data   */
      call save_rehmc

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_rex_rehmc
!***********************************************************************
!=======================================================================
!
!     metropolis step for replica echange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, nbead

      use rehmc_variables, only : &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   beta_bead, bfactor_rex, ratio_rex, naccept_rex, nreject_rex

      use alchem_variables, only : &
     &   pot_alchem

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   real numbers   */
      real(8) :: ranf1, randomno, bh1, bh2, bh3, bh4

!     /*   integers   */
      integer :: niter_rex

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of each bead                                    */
!-----------------------------------------------------------------------

!     /*   number of iterations   */
      niter_rex = nbead/2

      do j = 1, niter_rex

!        /*   uniform random number   */
         i = int(ranf1()*dble(nbead-1)) + 1

!        /*   energy difference   */
         bh1 = beta_bead(i)   * pot(i)
         bh2 = beta_bead(i+1) * pot(i+1)
         bh3 = beta_bead(i)   * pot_alchem(i,i+1)
         bh4 = beta_bead(i+1) * pot_alchem(i+1,i)

!        /*   beta times energy difference   */
         bfactor_rex(i) = bh3 + bh4 - bh1 - bh2

!        /*   judge acceptance   */

         if ( bfactor_rex(i) .lt. 75.d0 ) then

            if ( bfactor_rex(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_rex = naccept_rex + 1

!              /*   exchange data   */
               call exchange_rex_rehmc( i, i+1 )

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_rex(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_rex = naccept_rex + 1

!                 /*   exchange data   */
                  call exchange_rex_rehmc( i, i+1 )

               else

!                 /*   rejected   */
                  nreject_rex = nreject_rex + 1

               end if

            end if

         else

!           /*   rejected   */
            nreject_rex = nreject_rex + 1

         end if

      end do

!     /*   acceptance ratio   */

      if ( naccept_rex+nreject_rex .eq. 0 ) then
         ratio_rex = 0.d0
      else
         ratio_rex = dble(naccept_rex) / dble(naccept_rex+nreject_rex)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of each bead                                      */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   energy   */
      call getenergy_rehmc

!     /*   save data   */
      call save_rex_rehmc

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_hmc_rex_rehmc
!***********************************************************************
!=======================================================================
!
!     metropolis step for hybrid monte carlo.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, nbead

      use rehmc_variables, only : &
     &   beta_bead, bfactor_bead, ratio_hmc, hamiltonian_bead, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   hamiltonian_bead_save, naccept_hmc, nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of each bead                                    */
!-----------------------------------------------------------------------

      do i = 1, nbead

!        /*   energy difference   */
         dh = hamiltonian_bead(i) - hamiltonian_bead_save(i)

!        /*   beta times energy difference   */
         bfactor_bead(i) = beta_bead(i) * dh

!        /*   judge acceptance   */

         if ( bfactor_bead(i) .lt. 75.d0 ) then

            if ( bfactor_bead(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_hmc = naccept_hmc + 1

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_bead(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_hmc = naccept_hmc + 1

               else

!                 /*   rejected   */
                  nreject_hmc = nreject_hmc + 1

!                 /*   recover saved data   */
                  call recover_rex_rehmc( i )

               end if

            end if

         else

!           /*   rejected   */
            nreject_hmc = nreject_hmc + 1

!           /*   recover saved data   */
            call recover_rex_rehmc( i )

!        /*   judge acceptance   */
         end if

      end do

!     /*   acceptance ratio   */
      if ( naccept_hmc .eq. 0 ) then
         ratio_hmc = 0.d0
      else
         ratio_hmc = dble(naccept_hmc) / dble(naccept_hmc+nreject_hmc)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of each bead                                      */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   energy   */
      call getenergy_rehmc

!     /*   save data   */
      call save_rex_rehmc

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_hmc_rehmc_npt
!***********************************************************************
!=======================================================================
!
!     metropolis step for hybrid monte carlo.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, box_bead, nbead

      use rehmc_variables, only : &
     &   hamiltonian_bead, hamiltonian_bead_save, bfactor_bead, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, box_rehmc_last, &
     &   beta_bead, ratio_hmc, naccept_hmc, nreject_hmc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

      box_rehmc_last(:,:,:) = box_bead(:,:,:)

!-----------------------------------------------------------------------
!     /*   start loop of each bead                                    */
!-----------------------------------------------------------------------

      do i = 1, nbead

!        /*   energy difference   */
         dh = hamiltonian_bead(i) - hamiltonian_bead_save(i)

!        /*   beta times energy difference   */
         bfactor_bead(i) = beta_bead(i) * dh

!        /*   start judge acceptance   */
         if ( bfactor_bead(i) .lt. 75.d0 ) then

            if ( bfactor_bead(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_hmc = naccept_hmc + 1

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_bead(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_hmc = naccept_hmc + 1

               else

!                 /*   rejected   */
                  nreject_hmc = nreject_hmc + 1

!                 /*   recover saved data   */
                  call recover_rehmc_npt( i )

               end if

            end if

         else

!           /*   rejected   */
            nreject_hmc = nreject_hmc + 1

!           /*   recover saved data   */
            call recover_rehmc_npt( i )

!        /*   end judge acceptance   */
         end if

      end do

!     /*   acceptance ratio   */
      if ( naccept_hmc .eq. 0 ) then
         ratio_hmc = 0.d0
      else
         ratio_hmc = dble(naccept_hmc) / dble(naccept_hmc+nreject_hmc)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of each bead                                      */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   initialize velocity   */
      call init_box_velocity_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_rem_rehmc_npt
!***********************************************************************
!=======================================================================
!
!     metropolis step for replica echange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, box_bead, nbead

      use rehmc_variables, only : &
     &   hamiltonian_bead, beta_bead, bfactor_rem, ratio_rem, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, box_rehmc_last, &
     &   naccept_rem, nreject_rem

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   integers   */
      integer :: niter_rem

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh, db

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

      box_rehmc_last(:,:,:) = box_bead(:,:,:)

!-----------------------------------------------------------------------
!     /*   start exchange                                             */
!-----------------------------------------------------------------------

!     /*   number of iterations   */
      niter_rem = nbead/2

!     /*   iterations   */
      do j = 1, niter_rem

!        /*   random integer from 1 to (nbead-1)   */
         i = int( ranf1() * dble(nbead-1) ) + 1

!        /*   energy difference   */
         dh = hamiltonian_bead(i+1) - hamiltonian_bead(i)

!        /*   beta difference   */
         db = beta_bead(i+1) - beta_bead(i)

!        /*   beta times energy difference   */
         bfactor_rem(i) = - db * dh

!        /*   judge acceptance   */
         if ( bfactor_rem(i) .lt. 75.d0 ) then

            if ( bfactor_rem(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_rem = naccept_rem + 1

!              /*   exchange data   */
               call exchange_rem_rehmc_npt( i, i+1 )

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_rem(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_rem = naccept_rem + 1

!                 /*   exchange data   */
                  call exchange_rem_rehmc_npt( i, i+1 )

               else

!                 /*   rejected   */
                  nreject_rem = nreject_rem + 1

               end if

            end if

         else

!           /*   rejected   */
            nreject_rem = nreject_rem + 1

         end if

!     /*   iterations   */
      end do

!     /*   acceptance ratio   */
      if ( naccept_rem .eq. 0 ) then
         ratio_rem = 0.d0
      else
         ratio_rem = dble(naccept_rem) / dble(naccept_rem+nreject_rem)
      end if

!-----------------------------------------------------------------------
!     /*   end exchange                                               */
!-----------------------------------------------------------------------

!     /*   initialize velocity   */
      call init_velocity_real

!     /*   initialize velocity   */
      call init_box_velocity_rehmc

!     /*   energy   */
      call getenergy_rehmc_npt

!     /*   save data   */
      call save_rehmc_npt

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine judge_rem_remc
!***********************************************************************
!=======================================================================
!
!     metropolis step for replica echange.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, nbead

      use rehmc_variables, only : &
     &   beta_bead, bfactor_rem, ratio_rem, &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, &
     &   naccept_rem, nreject_rem

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!     /*   real numbers   */
      real(8) :: ranf1, randomno, dh, db

!     /*   integers   */
      integer :: niter_rem

!-----------------------------------------------------------------------
!     /*   preserve last values before rejection                      */
!-----------------------------------------------------------------------

      x_rehmc_last(:,:) = x(:,:)
      y_rehmc_last(:,:) = y(:,:)
      z_rehmc_last(:,:) = z(:,:)

      pot_rehmc_last(:) = pot(:)

      fx_rehmc_last(:,:) = fx(:,:)
      fy_rehmc_last(:,:) = fy(:,:)
      fz_rehmc_last(:,:) = fz(:,:)

!-----------------------------------------------------------------------
!     /*   start exchange                                             */
!-----------------------------------------------------------------------

!     /*   number of iterations   */
      niter_rem = nbead/2

!     /*   iterations   */
      do j = 1, niter_rem

!        /*   random integer from 1 to (nbead-1)   */
         i = int( ranf1() * dble(nbead-1) ) + 1

!        /*   energy difference   */
         dh = pot(i+1) - pot(i)

!        /*   beta difference   */
         db = beta_bead(i+1) - beta_bead(i)

!        /*   beta times energy difference   */
         bfactor_rem(i) = - db * dh

!        /*   judge acceptance   */
         if ( bfactor_rem(i) .lt. 75.d0 ) then

            if ( bfactor_rem(i) .le. 0.d0 ) then

!              /*   accepted   */
               naccept_rem = naccept_rem + 1

!              /*   exchange data   */
               call exchange_rem_remc( i, i+1 )

            else

!              /*   uniform random number   */
               randomno = ranf1()

               if ( exp(-bfactor_rem(i)) .gt. randomno ) then

!                 /*   accepted   */
                  naccept_rem = naccept_rem + 1

!                 /*   exchange data   */
                  call exchange_rem_remc( i, i+1 )

               else

!                 /*   rejected   */
                  nreject_rem = nreject_rem + 1

               end if

            end if

         else

!           /*   rejected   */
            nreject_rem = nreject_rem + 1

         end if

!     /*   iterations   */
      end do

!     /*   acceptance ratio   */
      if ( naccept_rem .eq. 0 ) then
         ratio_rem = 0.d0
      else
         ratio_rem = dble(naccept_rem) / dble(naccept_rem+nreject_rem)
      end if

!-----------------------------------------------------------------------
!     /*   end exchange                                               */
!-----------------------------------------------------------------------

      return
      end
