!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     set up Tully surface hopping MD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_velocity_cart_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, beta, myrank, natom, nbead, iboundary

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         do i = 1, nbead
         do j = 1, natom
            vsigma = sqrt(1.d0/beta/physmass(j))
            vx(j,i) = vsigma*gasdev()
            vy(j,i) = vsigma*gasdev()
            vz(j,i) = vsigma*gasdev()
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2 ( vx, natom, nbead )
      call my_mpi_bcast_real_2 ( vy, natom, nbead )
      call my_mpi_bcast_real_2 ( vz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   subtract translation                                       */
!-----------------------------------------------------------------------

      if ( natom .ne. 1 ) then
         call subtract_velocity_real
      end if

!-----------------------------------------------------------------------
!     /*   subtract rotation                                          */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
      if ( natom .ne. 1 ) then
         call subtract_rotation_real_MPI
      end if
      end if

!-----------------------------------------------------------------------
!     /*   scale according to temperature                             */
!-----------------------------------------------------------------------

      if ( natom .ne. 1 ) then
         call scale_velocity_real
      end if

      return
      end





!***********************************************************************
      subroutine subtract_velocity_real
!***********************************************************************
!=======================================================================
!
!     remove net momentum
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumvx = 0.d0
         sumvy = 0.d0
         sumvz = 0.d0
         sump  = 0.d0

         do j = 1, natom
            sumvx = sumvx + physmass(j)*vx(j,k)
            sumvy = sumvy + physmass(j)*vy(j,k)
            sumvz = sumvz + physmass(j)*vz(j,k)
            sump  = sump  + physmass(j)
         end do

         sumvx = sumvx/sump
         sumvy = sumvy/sump
         sumvz = sumvz/sump

         do j = 1, natom
            vx(j,k) = vx(j,k) - sumvx
            vy(j,k) = vy(j,k) - sumvy
            vz(j,k) = vz(j,k) - sumvz
         end do

      end do

      return
      end





!***********************************************************************
      subroutine scale_velocity_real
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, temperature, boltz, natom, nbead, ndof

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: ekin_xyz, temperature_xyz, factor_xyz

!-----------------------------------------------------------------------
!     /*   scale velocity:  all                                       */
!-----------------------------------------------------------------------

      do k = 1, nbead

         ekin_xyz = 0.d0

         do j = 1, natom
            ekin_xyz = ekin_xyz + physmass(j)*vx(j,k)*vx(j,k) &
     &                          + physmass(j)*vy(j,k)*vy(j,k) &
     &                          + physmass(j)*vz(j,k)*vz(j,k)
         end do

         ekin_xyz = 0.5d0*ekin_xyz

         temperature_xyz = ekin_xyz &
     &      /(0.5d0*dble(ndof)/dble(nbead)*boltz)

         if ( temperature_xyz .gt. 0.d0 ) then

            factor_xyz = sqrt( temperature / temperature_xyz )

            do j = 1, natom
               vx(j,k) = vx(j,k)*factor_xyz
               vy(j,k) = vy(j,k)*factor_xyz
               vz(j,k) = vz(j,k)*factor_xyz
            end do

         end if

      end do

      return
      end





!***********************************************************************
      subroutine subtract_rotation_real_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, xg, yg, zg, x, y, z, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i, k

      real(8)                 :: fm
      real(8), dimension(3,3) :: a, c
      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass_real

!-----------------------------------------------------------------------
!     /*   centroids and non-centroids                                */
!-----------------------------------------------------------------------

      do k = 1, nbead

!        /*   angular momentum                                        */

         b(:) = 0.d0

         do i = 1, natom
            fm   = physmass(i)
            b(1) = b(1) + fm*(y(i,k)-yg(k))*vz(i,k) &
     &                  - fm*(z(i,k)-zg(k))*vy(i,k)
            b(2) = b(2) + fm*(z(i,k)-zg(k))*vx(i,k) &
     &                  - fm*(x(i,k)-xg(k))*vz(i,k)
            b(3) = b(3) + fm*(x(i,k)-xg(k))*vy(i,k) &
     &                  - fm*(y(i,k)-yg(k))*vx(i,k)
         end do

!        /*   moment of inertia                                       */

         a(:,:) = 0.d0

         do i = 1, natom
            fm   = physmass(i)
            a(1,1) = a(1,1) + fm*(y(i,k)-yg(k))*(y(i,k)-yg(k)) &
     &                      + fm*(z(i,k)-zg(k))*(z(i,k)-zg(k))
            a(1,2) = a(1,2) - fm*(x(i,k)-xg(k))*(y(i,k)-yg(k))
            a(1,3) = a(1,3) - fm*(x(i,k)-xg(k))*(z(i,k)-zg(k))
            a(2,1) = a(2,1) - fm*(y(i,k)-yg(k))*(x(i,k)-xg(k))
            a(2,2) = a(2,2) + fm*(z(i,k)-zg(k))*(z(i,k)-zg(k)) &
     &                      + fm*(x(i,k)-xg(k))*(x(i,k)-xg(k))
            a(2,3) = a(2,3) - fm*(y(i,k)-yg(k))*(z(i,k)-zg(k))
            a(3,1) = a(3,1) - fm*(z(i,k)-zg(k))*(x(i,k)-xg(k))
            a(3,2) = a(3,2) - fm*(z(i,k)-zg(k))*(y(i,k)-yg(k))
            a(3,3) = a(3,3) + fm*(x(i,k)-xg(k))*(x(i,k)-xg(k)) &
     &                      + fm*(y(i,k)-yg(k))*(y(i,k)-yg(k))
         end do

!        /*   principal axis:  diagonalize moment of inertia          */

         call ddiag_MPI ( a, e, c, 3 )

!        /*   in principal axis:  angular momentum                    */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = angular momentum divided by moment of inertia       */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame                                   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            vx(i,k) = vx(i,k) - f(2)*(z(i,k)-zg(k)) &
     &                        + f(3)*(y(i,k)-yg(k))
            vy(i,k) = vy(i,k) - f(3)*(x(i,k)-xg(k)) &
     &                        + f(1)*(z(i,k)-zg(k))
            vz(i,k) = vz(i,k) - f(1)*(y(i,k)-yg(k)) &
     &                        + f(2)*(x(i,k)-xg(k))
         end do

      end do

      return
      end





!***********************************************************************
      subroutine center_of_mass_real
!***********************************************************************
!=======================================================================
!
!     calculate center of mass
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, xg, yg, zg, x, y, z, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumx, sumy, sumz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0
         sump = 0.d0

         do j = 1, natom
            sumx = sumx + physmass(j)*x(j,k)
            sumy = sumy + physmass(j)*y(j,k)
            sumz = sumz + physmass(j)*z(j,k)
            sump = sump + physmass(j)
         enddo

         xg(k) = sumx/sump
         yg(k) = sumy/sump
         zg(k) = sumz/sump

      end do

      return
      end





!***********************************************************************
      subroutine init_velocity_real_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, natom, nbead, myrank

      use rehmc_variables, only : &
     &   beta_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j
      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         do i = 1, nbead
         do j = 1, natom
            vsigma = sqrt(1.d0/beta_bead(i)/physmass(j))
            vx(j,i) = vsigma*gasdev()
            vy(j,i) = vsigma*gasdev()
            vz(j,i) = vsigma*gasdev()
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2 ( vx, natom, nbead )
      call my_mpi_bcast_real_2 ( vy, natom, nbead )
      call my_mpi_bcast_real_2 ( vz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   subtract translation                                       */
!-----------------------------------------------------------------------

!      if ( natom .ne. 1 ) then
!         call subtract_velocity_real
!      end if

!-----------------------------------------------------------------------
!     /*   subtract rotation                                          */
!-----------------------------------------------------------------------

!      if ( iboundary .eq. 0 ) then
!      if ( natom .ne. 1 ) then
!         call subtract_rotation_real_MPI
!      end if
!      end if

!-----------------------------------------------------------------------
!     /*   scale according to temperature                             */
!-----------------------------------------------------------------------

!      if ( natom .ne. 1 ) then
!         call scale_velocity_real
!      end if

      return
      end

