!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     Gaussian random distribution of velocity
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_velocity
!***********************************************************************
!=======================================================================
!
!    Translational velocities from Maxwell-Boltzmann distribution.
!    The distribution is determined by temperature and mass.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   vsigma: standard deviation of Maxwell distribution         */
!-----------------------------------------------------------------------

      call nm_velocity

!-----------------------------------------------------------------------
!     /*   remove net linear and angular momentum                     */
!-----------------------------------------------------------------------

      call correct_velocity

!-----------------------------------------------------------------------
!     /*   scale velocity:  centroid and non-centroid                 */
!-----------------------------------------------------------------------

!     /*   centroid and non-centroid   */
      call scale_velocity

!      /*   centroid   */
!      call scale_velocity_cent

!      /*   non-centroid   */
!      call scale_velocity_noncent

!-----------------------------------------------------------------------
!     /*   normal modes to Cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

!-----------------------------------------------------------------------
!     /*   check linear and angular momentum                          */
!-----------------------------------------------------------------------

!      call analysis_momentum ( 0 )
!      call analysis_momentum ( 2 )
!      stop

      return
      end





!***********************************************************************
      subroutine correct_velocity
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   itrans_start, irot_start, iboundary, method

      implicit none

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if      ( method(1:5) .eq. 'PIMD ' ) then
         continue
      else if ( method(1:5) .eq. 'PIHMC' ) then
         continue
      else if ( method(1:5) .eq. 'CMD  ' ) then
         continue
      else if ( method(1:5) .eq. 'RPMD ' ) then
         continue
      else if ( method(1:5) .eq. 'TRPMD' ) then
         continue
      else if ( method(1:5) .eq. 'BCMD ' ) then
         continue
      else
         return
      end if

      if ( ( itrans_start .eq. 0 ) .and. ( irot_start .eq. 0 ) ) return

!-----------------------------------------------------------------------
!     /*   subtract centroid translation                              */
!-----------------------------------------------------------------------

      if ( itrans_start .eq. 1 ) call subtract_velocity_cent
      if ( itrans_start .eq. 2 ) call subtract_velocity_cent
      if ( itrans_start .eq. 3 ) call subtract_velocity_modes
      if ( itrans_start .eq. 4 ) call subtract_velocity_modes
      if ( itrans_start .eq. 5 ) call subtract_velocity_all
      if ( itrans_start .eq. 6 ) call subtract_velocity_all

!-----------------------------------------------------------------------
!     /*   subtract rotation:  only free boundary condition           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then
         if ( irot_start .eq. 1 ) call subtract_rotation_cent
         if ( irot_start .eq. 2 ) call subtract_rotation_cent
         if ( irot_start .eq. 3 ) call subtract_rotation_modes
         if ( irot_start .eq. 4 ) call subtract_rotation_modes
         if ( irot_start .eq. 5 ) call subtract_rotation_mix
         if ( irot_start .eq. 6 ) call subtract_rotation_mix
      end if

      return
      end





!***********************************************************************
      subroutine subtract_velocity_cent
!***********************************************************************
!=======================================================================
!     remove net momentum of the centroid mode
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j
      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      sumvx = 0.d0
      sumvy = 0.d0
      sumvz = 0.d0
      sump  = 0.d0

      do j = 1, natom
         sumvx = sumvx + fictmass(j,1)*vux(j,1)
         sumvy = sumvy + fictmass(j,1)*vuy(j,1)
         sumvz = sumvz + fictmass(j,1)*vuz(j,1)
         sump  = sump  + fictmass(j,1)
      end do

      sumvx = sumvx/sump
      sumvy = sumvy/sump
      sumvz = sumvz/sump

      do j = 1, natom
         vux(j,1) = vux(j,1) - sumvx
         vuy(j,1) = vuy(j,1) - sumvy
         vuz(j,1) = vuz(j,1) - sumvz
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_velocity_noncent
!***********************************************************************
!=======================================================================
!
!     remove net momentum of non centroid modes
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, nbead, natom

      integer :: j, k

      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 2, nbead

         sumvx = 0.d0
         sumvy = 0.d0
         sumvz = 0.d0
         sump  = 0.d0

         do j = 1, natom
            sumvx = sumvx + fictmass(j,k)*vux(j,k)
            sumvy = sumvy + fictmass(j,k)*vuy(j,k)
            sumvz = sumvz + fictmass(j,k)*vuz(j,k)
            sump  = sump  + fictmass(j,k)
         end do

         sumvx = sumvx/sump
         sumvy = sumvy/sump
         sumvz = sumvz/sump

         do j = 1, natom
            vux(j,k) = vux(j,k) - sumvx
            vuy(j,k) = vuy(j,k) - sumvy
            vuz(j,k) = vuz(j,k) - sumvz
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine scale_velocity
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, temperature, boltz, &
     &    nbead, natom, ndof

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: ekin_xyz, temperature_xyz, factor_xyz

!-----------------------------------------------------------------------
!     /*   scale velocity:  all                                       */
!-----------------------------------------------------------------------

      ekin_xyz = 0.d0

      do k = 1, nbead
      do j = 1, natom
         ekin_xyz = ekin_xyz + fictmass(j,k)*vux(j,k)*vux(j,k) &
     &                       + fictmass(j,k)*vuy(j,k)*vuy(j,k) &
     &                       + fictmass(j,k)*vuz(j,k)*vuz(j,k)
      end do
      end do

      ekin_xyz = 0.5d0*ekin_xyz

      temperature_xyz = ekin_xyz /(0.5d0*ndof*boltz)

      if ( temperature_xyz .gt. 0.d0 ) then

         factor_xyz = sqrt( temperature / temperature_xyz )

         do k = 1, nbead
         do j = 1, natom
            vux(j,k) = vux(j,k)*factor_xyz
            vuy(j,k) = vuy(j,k)*factor_xyz
            vuz(j,k) = vuz(j,k)*factor_xyz
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine scale_velocity_cent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, temperature, boltz, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j

      real(8) :: ekin_x, ekin_y, ekin_z, factor_x, factor_y, factor_z, &
     &           temperature_x, temperature_y, temperature_z

!-----------------------------------------------------------------------
!     /*   scale velocity:  centroid                                  */
!-----------------------------------------------------------------------

      ekin_x = 0.d0
      ekin_y = 0.d0
      ekin_z = 0.d0

      do j = 1, natom
         ekin_x = ekin_x + fictmass(j,1)*vux(j,1)*vux(j,1)
         ekin_y = ekin_y + fictmass(j,1)*vuy(j,1)*vuy(j,1)
         ekin_z = ekin_z + fictmass(j,1)*vuz(j,1)*vuz(j,1)
      end do

      ekin_x = 0.5d0*ekin_x
      ekin_y = 0.5d0*ekin_y
      ekin_z = 0.5d0*ekin_z

      temperature_x = ekin_x /(0.5d0*natom*boltz)
      temperature_y = ekin_y /(0.5d0*natom*boltz)
      temperature_z = ekin_z /(0.5d0*natom*boltz)

      if ( temperature_x .gt. 0.d0 ) then
         factor_x = sqrt( temperature / temperature_x )
         do j = 1, natom
            vux(j,1) = vux(j,1)*factor_x
         end do
      end if

      if ( temperature_y .gt. 0.d0 ) then
         factor_y = sqrt( temperature / temperature_y )
         do j = 1, natom
            vuy(j,1) = vuy(j,1)*factor_y
         end do
      end if

      if ( temperature_z .gt. 0.d0 ) then
         factor_z = sqrt( temperature / temperature_z )
         do j = 1, natom
            vuz(j,1) = vuz(j,1)*factor_z
         end do
      end if

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine scale_velocity_noncent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, temperature, boltz, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: ekin_x, ekin_y, ekin_z, factor_x, factor_y, factor_z, &
     &           temperature_x, temperature_y, temperature_z

!-----------------------------------------------------------------------
!     /*   scale velocity:  non-centroids                             */
!-----------------------------------------------------------------------

      ekin_x = 0.d0
      ekin_y = 0.d0
      ekin_z = 0.d0

      do i = 2, nbead
      do j = 1, natom
         ekin_x = ekin_x + fictmass(j,i)*vux(j,i)*vux(j,i)
         ekin_y = ekin_y + fictmass(j,i)*vuy(j,i)*vuy(j,i)
         ekin_z = ekin_z + fictmass(j,i)*vuz(j,i)*vuz(j,i)
      end do
      end do

      ekin_x = 0.5d0*ekin_x
      ekin_y = 0.5d0*ekin_y
      ekin_z = 0.5d0*ekin_z

      temperature_x = ekin_x /(0.5d0*natom*(nbead-1)*boltz)
      temperature_y = ekin_y /(0.5d0*natom*(nbead-1)*boltz)
      temperature_z = ekin_z /(0.5d0*natom*(nbead-1)*boltz)

      if ( temperature_x .gt. 0.d0 ) then

         factor_x = sqrt( temperature / temperature_x )

         do i = 2, nbead
         do j = 1, natom
            vux(j,i) = vux(j,i)*factor_x
         end do
         end do

      end if

      if ( temperature_y .gt. 0.d0 ) then

         factor_y = sqrt( temperature / temperature_y )

         do i = 2, nbead
         do j = 1, natom
            vuy(j,i) = vuy(j,i)*factor_y
         end do
         end do

      end if

      if ( temperature_z .gt. 0.d0 ) then

         factor_z = sqrt( temperature / temperature_z )

         do i = 2, nbead
         do j = 1, natom
            vuz(j,i) = vuz(j,i)*factor_z
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )


      return
      end





!***********************************************************************
      subroutine center_of_mass
!***********************************************************************
!=======================================================================
!
!     calculate center of mass
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xg, yg, zg, ux, uy, uz, fictmass, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumx, sumy, sumz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0
         sump = 0.d0

         do j = 1, natom
            sumx = sumx + fictmass(j,k)*ux(j,k)
            sumy = sumy + fictmass(j,k)*uy(j,k)
            sumz = sumz + fictmass(j,k)*uz(j,k)
            sump = sump + fictmass(j,k)
         enddo

         xg(k) = sumx/sump
         yg(k) = sumy/sump
         zg(k) = sumz/sump

      end do

      return
      end





!***********************************************************************
      subroutine subtract_rotation_cent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, xg, yg, zg, ux, uy, uz, vux, vuy, vuz, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i

      real(8)                 :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids only                                             */
!-----------------------------------------------------------------------

!     /*   angular momentum                                        */

      b(:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         b(1) = b(1) + fm*(uy(i,1)-yg(1))*vuz(i,1) &
     &               - fm*(uz(i,1)-zg(1))*vuy(i,1)
         b(2) = b(2) + fm*(uz(i,1)-zg(1))*vux(i,1) &
     &               - fm*(ux(i,1)-xg(1))*vuz(i,1)
         b(3) = b(3) + fm*(ux(i,1)-xg(1))*vuy(i,1) &
     &               - fm*(uy(i,1)-yg(1))*vux(i,1)
      end do

!     /*   moment of inertia                                       */

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                   + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
         a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
         a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
         a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
         a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                   + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
         a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
         a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
         a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
         a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                   + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
      end do

!     /*   principal axis:  diagonalize moment of inertia          */

      call ddiag ( a, e, c, 3 )

!     /*   in principal axis:  angular momentum                    */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = angular momentum divided by moment of inertia       */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                   */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do i = 1, natom
         vux(i,1) = vux(i,1) - f(2)*(uz(i,1)-zg(1)) &
     &                       + f(3)*(uy(i,1)-yg(1))
         vuy(i,1) = vuy(i,1) - f(3)*(ux(i,1)-xg(1)) &
     &                       + f(1)*(uz(i,1)-zg(1))
         vuz(i,1) = vuz(i,1) - f(1)*(uy(i,1)-yg(1)) &
     &                       + f(2)*(ux(i,1)-xg(1))
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_rotation_noncent
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, xg, yg, zg, ux, uy, uz, vux, vuy, vuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i, k

      real(8)                 :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   non-centroids only                                         */
!-----------------------------------------------------------------------

      do k = 2, nbead

!        /*   angular momentum                                        */

         b(:) = 0.d0

         do i = 1, natom
            fm   = fictmass(i,k)
            b(1) = b(1) + fm*(uy(i,k)-yg(k))*vuz(i,k) &
     &                  - fm*(uz(i,k)-zg(k))*vuy(i,k)
            b(2) = b(2) + fm*(uz(i,k)-zg(k))*vux(i,k) &
     &                  - fm*(ux(i,k)-xg(k))*vuz(i,k)
            b(3) = b(3) + fm*(ux(i,k)-xg(k))*vuy(i,k) &
     &                  - fm*(uy(i,k)-yg(k))*vux(i,k)
         end do

!        /*   moment of inertia                                       */

         a(:,:) = 0.d0

         do i = 1, natom
            fm   = fictmass(i,k)
            a(1,1) = a(1,1) + fm*(uy(i,k)-yg(k))*(uy(i,k)-yg(k)) &
     &                      + fm*(uz(i,k)-zg(k))*(uz(i,k)-zg(k))
            a(1,2) = a(1,2) - fm*(ux(i,k)-xg(k))*(uy(i,k)-yg(k))
            a(1,3) = a(1,3) - fm*(ux(i,k)-xg(k))*(uz(i,k)-zg(k))
            a(2,1) = a(2,1) - fm*(uy(i,k)-yg(k))*(ux(i,k)-xg(k))
            a(2,2) = a(2,2) + fm*(uz(i,k)-zg(k))*(uz(i,k)-zg(k)) &
     &                      + fm*(ux(i,k)-xg(k))*(ux(i,k)-xg(k))
            a(2,3) = a(2,3) - fm*(uy(i,k)-yg(k))*(uz(i,k)-zg(k))
            a(3,1) = a(3,1) - fm*(uz(i,k)-zg(k))*(ux(i,k)-xg(k))
            a(3,2) = a(3,2) - fm*(uz(i,k)-zg(k))*(uy(i,k)-yg(k))
            a(3,3) = a(3,3) + fm*(ux(i,k)-xg(k))*(ux(i,k)-xg(k)) &
     &                      + fm*(uy(i,k)-yg(k))*(uy(i,k)-yg(k))
         end do

!        /*   principal axis:  diagonalize moment of inertia          */

         call ddiag ( a, e, c, 3 )

!        /*   in principal axis:  angular momentum                    */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = angular momentum divided by moment of inertia       */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame                                   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            vux(i,k) = vux(i,k) - f(2)*(uz(i,k)-zg(k)) &
     &                          + f(3)*(uy(i,k)-yg(k))
            vuy(i,k) = vuy(i,k) - f(3)*(ux(i,k)-xg(k)) &
     &                          + f(1)*(uz(i,k)-zg(k))
            vuz(i,k) = vuz(i,k) - f(1)*(uy(i,k)-yg(k)) &
     &                          + f(2)*(ux(i,k)-xg(k))
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine nm_velocity
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, beta, vux, vuy, vuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do i = 1, nbead
      do j = 1, natom
         vsigma = sqrt(1.d0/beta/fictmass(j,i))
         vux(j,i) = vsigma*gasdev()
         vuy(j,i) = vsigma*gasdev()
         vuz(j,i) = vsigma*gasdev()
      end do
      end do

      return
      end





!***********************************************************************
      subroutine subtract_velocity_cart
!***********************************************************************
!=======================================================================
!
!     remove net momentum
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumvx = 0.d0
         sumvy = 0.d0
         sumvz = 0.d0
         sump  = 0.d0

         do j = 1, natom
            sumvx = sumvx + physmass(j)*vx(j,k)
            sumvy = sumvy + physmass(j)*vy(j,k)
            sumvz = sumvz + physmass(j)*vz(j,k)
            sump  = sump  + physmass(j)
         end do

         sumvx = sumvx/sump
         sumvy = sumvy/sump
         sumvz = sumvz/sump

         do j = 1, natom
            vx(j,k) = vx(j,k) - sumvx
            vy(j,k) = vy(j,k) - sumvy
            vz(j,k) = vz(j,k) - sumvz
         end do

      end do

!-----------------------------------------------------------------------
!     /*   cartesian to normal modes                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 1 )

      return
      end





!***********************************************************************
      subroutine subtract_rotation_cart
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, vx, vy, vz, xg, yg, zg, x, y, z, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i, k

      real(8)                 :: fm
      real(8), dimension(3,3) :: a, c
      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans ( 0 )

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass_cart

!-----------------------------------------------------------------------
!     /*   centroids and non-centroids                                */
!-----------------------------------------------------------------------

      do k = 1, nbead

!        /*   angular momentum                                        */

         b(:) = 0.d0

         do i = 1, natom
            fm   = physmass(i)
            b(1) = b(1) + fm*(y(i,k)-yg(k))*vz(i,k) &
     &                  - fm*(z(i,k)-zg(k))*vy(i,k)
            b(2) = b(2) + fm*(z(i,k)-zg(k))*vx(i,k) &
     &                  - fm*(x(i,k)-xg(k))*vz(i,k)
            b(3) = b(3) + fm*(x(i,k)-xg(k))*vy(i,k) &
     &                  - fm*(y(i,k)-yg(k))*vx(i,k)
         end do

!        /*   moment of inertia                                       */

         a(:,:) = 0.d0

         do i = 1, natom
            fm   = physmass(i)
            a(1,1) = a(1,1) + fm*(y(i,k)-yg(k))*(y(i,k)-yg(k)) &
     &                      + fm*(z(i,k)-zg(k))*(z(i,k)-zg(k))
            a(1,2) = a(1,2) - fm*(x(i,k)-xg(k))*(y(i,k)-yg(k))
            a(1,3) = a(1,3) - fm*(x(i,k)-xg(k))*(z(i,k)-zg(k))
            a(2,1) = a(2,1) - fm*(y(i,k)-yg(k))*(x(i,k)-xg(k))
            a(2,2) = a(2,2) + fm*(z(i,k)-zg(k))*(z(i,k)-zg(k)) &
     &                      + fm*(x(i,k)-xg(k))*(x(i,k)-xg(k))
            a(2,3) = a(2,3) - fm*(y(i,k)-yg(k))*(z(i,k)-zg(k))
            a(3,1) = a(3,1) - fm*(z(i,k)-zg(k))*(x(i,k)-xg(k))
            a(3,2) = a(3,2) - fm*(z(i,k)-zg(k))*(y(i,k)-yg(k))
            a(3,3) = a(3,3) + fm*(x(i,k)-xg(k))*(x(i,k)-xg(k)) &
     &                      + fm*(y(i,k)-yg(k))*(y(i,k)-yg(k))
         end do

!        /*   principal axis:  diagonalize moment of inertia          */

         call ddiag ( a, e, c, 3 )

!        /*   in principal axis:  angular momentum                    */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = angular momentum divided by moment of inertia       */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame                                   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            vx(i,k) = vx(i,k) - f(2)*(z(i,k)-zg(k)) &
     &                        + f(3)*(y(i,k)-yg(k))
            vy(i,k) = vy(i,k) - f(3)*(x(i,k)-xg(k)) &
     &                        + f(1)*(z(i,k)-zg(k))
            vz(i,k) = vz(i,k) - f(1)*(y(i,k)-yg(k)) &
     &                        + f(2)*(x(i,k)-xg(k))
         end do

      end do

!-----------------------------------------------------------------------
!     /*   cartesian to normal modes                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 1 )

      return
      end





!***********************************************************************
      subroutine center_of_mass_cart
!***********************************************************************
!=======================================================================
!
!     calculate center of mass
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, xg, yg, zg, x, y, z, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumx, sumy, sumz, sump

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans ( 0 )

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumx = 0.d0
         sumy = 0.d0
         sumz = 0.d0
         sump = 0.d0

         do j = 1, natom
            sumx = sumx + physmass(j)*x(j,k)
            sumy = sumy + physmass(j)*y(j,k)
            sumz = sumz + physmass(j)*z(j,k)
            sump = sump + physmass(j)
         enddo

         xg(k) = sumx/sump
         yg(k) = sumy/sump
         zg(k) = sumz/sump

      end do

      return
      end





!***********************************************************************
      subroutine scale_velocity_cart
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, physmass, boltz, temperature, natom, nbead, ndof

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: ekin_xyz, temperature_xyz, factor_xyz

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

!-----------------------------------------------------------------------
!     /*   scale velocity:  all                                       */
!-----------------------------------------------------------------------

      ekin_xyz = 0.d0

      do k = 1, nbead
      do j = 1, natom
         ekin_xyz = ekin_xyz + physmass(j)*vx(j,k)*vx(j,k) &
     &                       + physmass(j)*vy(j,k)*vy(j,k) &
     &                       + physmass(j)*vz(j,k)*vz(j,k)
      end do
      end do

      ekin_xyz = 0.5d0*ekin_xyz

      temperature_xyz = ekin_xyz /(0.5d0*ndof*boltz)

      if ( temperature_xyz .gt. 0.d0 ) then

         factor_xyz = sqrt( temperature / temperature_xyz )

         do k = 1, nbead
         do j = 1, natom
            vx(j,k) = vx(j,k)*factor_xyz
            vy(j,k) = vy(j,k)*factor_xyz
            vz(j,k) = vz(j,k)*factor_xyz
         end do
         end do

      end if

!-----------------------------------------------------------------------
!     /*   cartesian to normal modes                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 1 )

      return
      end





!***********************************************************************
      subroutine subtract_position_cent
!***********************************************************************
!=======================================================================
!     remove net position of the centroid mode
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : ux, uy, uz, fictmass, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j
      real(8) :: sumx, sumy, sumz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0
      sump = 0.d0

      do j = 1, natom
         sumx = sumx + fictmass(j,1)*ux(j,1)
         sumy = sumy + fictmass(j,1)*uy(j,1)
         sumz = sumz + fictmass(j,1)*uz(j,1)
         sump = sump + fictmass(j,1)
      end do

      sumx = sumx/sump
      sumy = sumy/sump
      sumz = sumz/sump

      do j = 1, natom
         ux(j,1) = ux(j,1) - sumx
         uy(j,1) = uy(j,1) - sumy
         uz(j,1) = uz(j,1) - sumz
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_velocity_modes
!***********************************************************************
!=======================================================================
!
!     remove net momentum
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   centroids                                                  */
!-----------------------------------------------------------------------

      sumvx = 0.d0
      sumvy = 0.d0
      sumvz = 0.d0
      sump  = 0.d0

      do j = 1, natom
         sumvx = sumvx + fictmass(j,1)*vux(j,1)
         sumvy = sumvy + fictmass(j,1)*vuy(j,1)
         sumvz = sumvz + fictmass(j,1)*vuz(j,1)
         sump  = sump  + fictmass(j,1)
      end do

      sumvx = sumvx/sump
      sumvy = sumvy/sump
      sumvz = sumvz/sump

      do j = 1, natom
         vux(j,1) = vux(j,1) - sumvx
         vuy(j,1) = vuy(j,1) - sumvy
         vuz(j,1) = vuz(j,1) - sumvz
      end do

!-----------------------------------------------------------------------
!     /*   noncentroids                                               */
!-----------------------------------------------------------------------

      sumvx = 0.d0
      sumvy = 0.d0
      sumvz = 0.d0
      sump  = 0.d0

      do k = 2, nbead
      do j = 1, natom
         sumvx = sumvx + fictmass(j,k)*vux(j,k)
         sumvy = sumvy + fictmass(j,k)*vuy(j,k)
         sumvz = sumvz + fictmass(j,k)*vuz(j,k)
         sump  = sump  + fictmass(j,k)
      end do
      end do

      sumvx = sumvx/sump
      sumvy = sumvy/sump
      sumvz = sumvz/sump

      do k = 2, nbead
      do j = 1, natom
         vux(j,k) = vux(j,k) - sumvx
         vuy(j,k) = vuy(j,k) - sumvy
         vuz(j,k) = vuz(j,k) - sumvz
      end do
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_rotation_modes
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, xg, yg, zg, ux, uy, uz, vux, vuy, vuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i, k

      real(8)                 :: fm
      real(8), dimension(3,3) :: a, c
      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   centroids                                                  */
!-----------------------------------------------------------------------

!     /*   angular momentum                                        */

      b(:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         b(1) = b(1) + fm*(uy(i,1)-yg(1))*vuz(i,1) &
     &               - fm*(uz(i,1)-zg(1))*vuy(i,1)
         b(2) = b(2) + fm*(uz(i,1)-zg(1))*vux(i,1) &
     &               - fm*(ux(i,1)-xg(1))*vuz(i,1)
         b(3) = b(3) + fm*(ux(i,1)-xg(1))*vuy(i,1) &
     &               - fm*(uy(i,1)-yg(1))*vux(i,1)
      end do

!     /*   moment of inertia                                       */

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = fictmass(i,1)
         a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                   + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
         a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
         a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
         a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
         a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                   + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
         a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
         a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
         a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
         a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                   + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
      end do

!     /*   principal axis:  diagonalize moment of inertia          */

      call ddiag ( a, e, c, 3 )

!     /*   in principal axis:  angular momentum                    */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = angular momentum divided by moment of inertia       */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                   */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do i = 1, natom
         vux(i,1) = vux(i,1) - f(2)*(uz(i,1)-zg(1)) &
     &                       + f(3)*(uy(i,1)-yg(1))
         vuy(i,1) = vuy(i,1) - f(3)*(ux(i,1)-xg(1)) &
     &                       + f(1)*(uz(i,1)-zg(1))
         vuz(i,1) = vuz(i,1) - f(1)*(uy(i,1)-yg(1)) &
     &                       + f(2)*(ux(i,1)-xg(1))
      end do

!-----------------------------------------------------------------------
!     /*   non-centroids                                              */
!-----------------------------------------------------------------------

!     /*   angular momentum                                        */

      b(:) = 0.d0

      do k = 2, nbead
      do i = 1, natom
         fm   = fictmass(i,k)
         b(1) = b(1) + fm*(uy(i,k)-yg(k))*vuz(i,k) &
     &               - fm*(uz(i,k)-zg(k))*vuy(i,k)
         b(2) = b(2) + fm*(uz(i,k)-zg(k))*vux(i,k) &
     &               - fm*(ux(i,k)-xg(k))*vuz(i,k)
         b(3) = b(3) + fm*(ux(i,k)-xg(k))*vuy(i,k) &
     &               - fm*(uy(i,k)-yg(k))*vux(i,k)
      end do
      end do

!     /*   moment of inertia                                       */

      a(:,:) = 0.d0

      do k = 2, nbead
      do i = 1, natom
         fm   = fictmass(i,k)
         a(1,1) = a(1,1) + fm*(uy(i,k)-yg(k))*(uy(i,k)-yg(k)) &
     &                   + fm*(uz(i,k)-zg(k))*(uz(i,k)-zg(k))
         a(1,2) = a(1,2) - fm*(ux(i,k)-xg(k))*(uy(i,k)-yg(k))
         a(1,3) = a(1,3) - fm*(ux(i,k)-xg(k))*(uz(i,k)-zg(k))
         a(2,1) = a(2,1) - fm*(uy(i,k)-yg(k))*(ux(i,k)-xg(k))
         a(2,2) = a(2,2) + fm*(uz(i,k)-zg(k))*(uz(i,k)-zg(k)) &
     &                   + fm*(ux(i,k)-xg(k))*(ux(i,k)-xg(k))
         a(2,3) = a(2,3) - fm*(uy(i,k)-yg(k))*(uz(i,k)-zg(k))
         a(3,1) = a(3,1) - fm*(uz(i,k)-zg(k))*(ux(i,k)-xg(k))
         a(3,2) = a(3,2) - fm*(uz(i,k)-zg(k))*(uy(i,k)-yg(k))
         a(3,3) = a(3,3) + fm*(ux(i,k)-xg(k))*(ux(i,k)-xg(k)) &
     &                   + fm*(uy(i,k)-yg(k))*(uy(i,k)-yg(k))
      end do
      end do

!     /*   principal axis:  diagonalize moment of inertia          */

      call ddiag ( a, e, c, 3 )

!     /*   in principal axis:  angular momentum                    */

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!     /*   d = angular momentum divided by moment of inertia       */

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!     /*   d in laboratory frame                                   */

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

      do k = 2, nbead
      do i = 1, natom
         vux(i,k) = vux(i,k) - f(2)*(uz(i,k)-zg(k)) &
     &                       + f(3)*(uy(i,k)-yg(k))
         vuy(i,k) = vuy(i,k) - f(3)*(ux(i,k)-xg(k)) &
     &                       + f(1)*(uz(i,k)-zg(k))
         vuz(i,k) = vuz(i,k) - f(1)*(uy(i,k)-yg(k)) &
     &                       + f(2)*(ux(i,k)-xg(k))
      end do
      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_velocity_all
!***********************************************************************
!=======================================================================
!
!     remove net momentum of non centroid modes
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    vux, vuy, vuz, fictmass, nbead, natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k
      real(8) :: sumvx, sumvy, sumvz, sump

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do k = 1, nbead

         sumvx = 0.d0
         sumvy = 0.d0
         sumvz = 0.d0
         sump  = 0.d0

         do j = 1, natom
            sumvx = sumvx + fictmass(j,k)*vux(j,k)
            sumvy = sumvy + fictmass(j,k)*vuy(j,k)
            sumvz = sumvz + fictmass(j,k)*vuz(j,k)
            sump  = sump  + fictmass(j,k)
         end do

         sumvx = sumvx/sump
         sumvy = sumvy/sump
         sumvz = sumvz/sump

         do j = 1, natom
            vux(j,k) = vux(j,k) - sumvx
            vuy(j,k) = vuy(j,k) - sumvy
            vuz(j,k) = vuz(j,k) - sumvz
         end do

      end do

!-----------------------------------------------------------------------
!     /*   normal modes to cartesian                                  */
!-----------------------------------------------------------------------

      call nm_trans_velocity ( 0 )

      return
      end





!***********************************************************************
      subroutine subtract_rotation_mix
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, xg, yg, zg, ux, uy, uz, vux, vuy, vuz, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer                 :: i, k

      real(8)                 :: fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      call center_of_mass

!-----------------------------------------------------------------------
!     /*   non-centroids only                                         */
!-----------------------------------------------------------------------

      do k = 1, nbead

!        /*   angular momentum                                        */

         b(:) = 0.d0

         do i = 1, natom
            fm   = fictmass(i,1)
            b(1) = b(1) + fm*(uy(i,1)-xg(1))*vuz(i,k) &
     &                  - fm*(uz(i,1)-zg(1))*vuy(i,k)
            b(2) = b(2) + fm*(uz(i,1)-zg(1))*vux(i,k) &
     &                  - fm*(ux(i,1)-xg(1))*vuz(i,k)
            b(3) = b(3) + fm*(ux(i,1)-xg(1))*vuy(i,k) &
     &                  - fm*(uy(i,1)-yg(1))*vux(i,k)
         end do

!        /*   moment of inertia   */

         a(:,:) = 0.d0

          do i = 1, natom
            fm   = fictmass(i,1)
            a(1,1) = a(1,1) + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1)) &
     &                      + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1))
            a(1,2) = a(1,2) - fm*(ux(i,1)-xg(1))*(uy(i,1)-yg(1))
            a(1,3) = a(1,3) - fm*(ux(i,1)-xg(1))*(uz(i,1)-zg(1))
            a(2,1) = a(2,1) - fm*(uy(i,1)-yg(1))*(ux(i,1)-xg(1))
            a(2,2) = a(2,2) + fm*(uz(i,1)-zg(1))*(uz(i,1)-zg(1)) &
     &                      + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1))
            a(2,3) = a(2,3) - fm*(uy(i,1)-yg(1))*(uz(i,1)-zg(1))
            a(3,1) = a(3,1) - fm*(uz(i,1)-zg(1))*(ux(i,1)-xg(1))
            a(3,2) = a(3,2) - fm*(uz(i,1)-zg(1))*(uy(i,1)-yg(1))
            a(3,3) = a(3,3) + fm*(ux(i,1)-xg(1))*(ux(i,1)-xg(1)) &
     &                      + fm*(uy(i,1)-yg(1))*(uy(i,1)-yg(1))
         end do

!        /*   principal axis:  diagonalize moment of inertia          */

         call ddiag ( a, e, c, 3 )

!        /*   in principal axis:  angular momentum                    */

         d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
         d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
         d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!        /*   d = angular momentum divided by moment of inertia       */

         if ( natom .eq. 1 ) then
            d(1) = 0.d0
            d(2) = 0.d0
            d(3) = 0.d0
         else if ( natom .eq. 2 ) then
            d(1) = 0.d0
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         else
            if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
            if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
            if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
         end if

!        /*   d in laboratory frame                                   */

         f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
         f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
         f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

         do i = 1, natom
            vux(i,k) = vux(i,k) - f(2)*(uz(i,1)-zg(1)) &
     &                          + f(3)*(uy(i,1)-yg(1))
            vuy(i,k) = vuy(i,k) - f(3)*(ux(i,1)-xg(1)) &
     &                          + f(1)*(uz(i,1)-zg(1))
            vuz(i,k) = vuz(i,k) - f(1)*(uy(i,1)-yg(1)) &
     &                          + f(2)*(ux(i,1)-xg(1))
         end do

      end do

      return
      end
