!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    July 1, 2020 by M. Shiga
!      Description:     set up initial geometry in string method
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use common_variables, only : &
     &   x, y, z, au_length, nbead, natom, iounit

!     /*   local variables   */

      implicit none

      integer :: i, j, ierr, ibead

      real(8) :: dx, dy, dz, dr

      real(8) :: factor = 1.d0

      real(8) :: tiny_value = 1.d-4

      character(len=10) :: char

      real(8), parameter :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   read reactant and product geometries                       */
!-----------------------------------------------------------------------

      ierr = 0

      open ( iounit, file = 'structure.dat' )

      ibead = 0

      do j = 1, nbead

         read( iounit, *, iostat=ierr )
         read( iounit, *, iostat=ierr ) char

         if ( char(1:5) .eq. 'BOHR ' )     factor = 1.d0
         if ( char(1:9) .eq. 'ANGSTROM ' ) factor = bohr2ang

         do i = 1, natom
            read( iounit, *, iostat=ierr ) &
     &         char, x(i,j), y(i,j), z(i,j)
         end do

         if ( ierr .ne. 0 ) exit

         x(:,j) = x(:,j) / factor
         y(:,j) = y(:,j) / factor
         z(:,j) = z(:,j) / factor

         if ( j .ge. 2 ) then

            do i = 1, natom

               dx = x(i,j) - x(i,j-1)
               dy = y(i,j) - y(i,j-1)
               dz = z(i,j) - z(i,j-1)

               call pbc_atom( dx, dy, dz )

               x(i,j) = x(i,j-1) + dx
               y(i,j) = y(i,j-1) + dy
               z(i,j) = z(i,j-1) + dz

            end do

         end if

         ibead = j

      end do

      close( iounit )

      if ( ibead .le. 1 ) then

         write( 6, '(a)' ) 'Error - structure.dat read incorrectly.'

         ierr = 1

      else

         x(:,nbead) = x(:,ibead)
         y(:,nbead) = y(:,ibead)
         z(:,nbead) = z(:,ibead)

         do i = 1, natom

            dx = x(i,nbead) - x(i,ibead-1)
            dy = y(i,nbead) - y(i,ibead-1)
            dz = z(i,nbead) - z(i,ibead-1)

            call pbc_atom( dx, dy, dz )

            do j = ibead, nbead-1

               factor = dble(j-ibead+1) / dble(nbead-ibead+1)

               x(i,j) = x(i,ibead-1) + dx * factor
               y(i,j) = y(i,ibead-1) + dy * factor
               z(i,j) = z(i,ibead-1) + dz * factor

            end do

         end do

         ierr = 0

      end if

      call error_handling( ierr, 'subroutine init_string', 22 )

!-----------------------------------------------------------------------
!     /*   check reactant and product                                 */
!-----------------------------------------------------------------------

      ierr = 0

      dr = 0.d0

      do i = 1, natom

         dx = x(i,nbead) - x(i,1)
         dy = y(i,nbead) - y(i,1)
         dz = z(i,nbead) - z(i,1)

         call pbc_atom(dx, dy, dz)

         dr = dr + dx*dx + dy*dy + dz*dz

      end do

      dr = sqrt(dr)

      if ( dr .le. tiny_value ) ierr = 1

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - reactant and product too close.'
      end if

      call error_handling( ierr, 'subroutine init_string', 22 )

!-----------------------------------------------------------------------
!     /*   align the string                                           */
!-----------------------------------------------------------------------

      call align_string

      return
      end
