!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     polymers initialization
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_polymers_atom_MPI
!***********************************************************************
!=======================================================================
!
!     initial configuration
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */

      use common_variables, only : &
     &   au_length, natom, iounit, myrank_world

      use polymers_variables, only : &
     &   xc_poly, yc_poly, zc_poly, guess_poly, npoly

!     /*   local variables   */

      implicit none

      integer :: i, j, ierr

      real(8) :: drc

      character(len=10) :: char
      character(len=3)  :: symb

      real(8), parameter :: bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   read reactant and product geometries, and interpolate      */
!-----------------------------------------------------------------------

      if ( guess_poly(1:5) .eq. 'LINE ' ) then

         if ( myrank_world .eq. 0 ) then

!           /*   reactant geometry   */

            open ( iounit, file = 'structure.dat' )

            read ( iounit, *, iostat=ierr )
            read ( iounit, *, iostat=ierr ) char
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            symb, xc_poly(i,1), yc_poly(i,1), zc_poly(i,1)
            end do

            if      ( char(1:5) .eq. 'BOHR '     ) then
               continue
            else if ( char(1:9) .eq. 'ANGSTROM ' ) then
               xc_poly(:,1) = xc_poly(:,1) / bohr2ang
               yc_poly(:,1) = yc_poly(:,1) / bohr2ang
               zc_poly(:,1) = zc_poly(:,1) / bohr2ang
            else
               ierr = 1
            end if

            read ( iounit, *, iostat=ierr )
            read ( iounit, *, iostat=ierr ) char
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) symb, &
     &            xc_poly(i,npoly), yc_poly(i,npoly), zc_poly(i,npoly)
            end do

            if      ( char(1:5) .eq. 'BOHR '     ) then
               continue
            else if ( char(1:9) .eq. 'ANGSTROM ' ) then
               xc_poly(:,npoly) = xc_poly(:,npoly) / bohr2ang
               yc_poly(:,npoly) = yc_poly(:,npoly) / bohr2ang
               zc_poly(:,npoly) = zc_poly(:,npoly) / bohr2ang
            else
               ierr = 1
            end if

            close( iounit )

            if ( ierr .ne. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - structure.dat read incorrectly.'
               write( 6, '(a)' )
            end if

!           /*   straight line interpolation   */

            do j = 1, npoly

               drc = dble(j-1) / dble(npoly-1)

               do i = 1, natom

                  xc_poly(i,j) = xc_poly(i,1) &
     &               + ( xc_poly(i,npoly) - xc_poly(i,1) ) * drc
                  yc_poly(i,j) = yc_poly(i,1) &
     &               + ( yc_poly(i,npoly) - yc_poly(i,1) ) * drc
                  zc_poly(i,j) = zc_poly(i,1) &
     &               + ( zc_poly(i,npoly) - zc_poly(i,1) ) * drc

               end do

            end do

         end if

!        /*   mpi communication   */
         call my_mpi_bcast_int_0_world ( ierr )

!        /*   error handling   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine init_polymers_atom_MPI', 33 )

         call my_mpi_bcast_real_2_world( xc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( yc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( zc_poly, natom, npoly )

!-----------------------------------------------------------------------
!     /*   read reactant and product geometries, and interpolate      */
!-----------------------------------------------------------------------

      else if ( guess_poly(1:5) .eq. 'NONE ' ) then

         if ( myrank_world .eq. 0 ) then

!           /*   reactant geometry   */

            open ( iounit, file = 'structure.dat' )

            read ( iounit, *, iostat=ierr )
            read ( iounit, *, iostat=ierr ) char
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            symb, xc_poly(i,1), yc_poly(i,1), zc_poly(i,1)
            end do

            if      ( char(1:5) .eq. 'BOHR '     ) then
               continue
            else if ( char(1:9) .eq. 'ANGSTROM ' ) then
               xc_poly(:,1) = xc_poly(:,1) / bohr2ang
               yc_poly(:,1) = yc_poly(:,1) / bohr2ang
               zc_poly(:,1) = zc_poly(:,1) / bohr2ang
            else
               ierr = 1
            end if

            close( iounit )

            if ( ierr .ne. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - structure.dat read incorrectly.'
               write( 6, '(a)' )
            end if

!           /*   copy   */

            do j = 2, npoly

               do i = 1, natom

                  xc_poly(i,j) = xc_poly(i,1)
                  yc_poly(i,j) = yc_poly(i,1)
                  zc_poly(i,j) = zc_poly(i,1)

               end do

            end do

         end if

!        /*   mpi communication   */
         call my_mpi_bcast_int_0_world ( ierr )

!        /*   error handling   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine init_polymers_atom_MPI', 33 )

         call my_mpi_bcast_real_2_world( xc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( yc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( zc_poly, natom, npoly )

!-----------------------------------------------------------------------
!     /*   read all centroid geometries                               */
!-----------------------------------------------------------------------

      else

         if ( myrank_world .eq. 0 ) then

            open ( iounit, file = 'structure.dat' )

            do j = 1, npoly

               read ( iounit, *, iostat=ierr )
               read ( iounit, *, iostat=ierr ) char
               do i = 1, natom
                  read( iounit, *, iostat=ierr ) &
     &               symb, xc_poly(i,j), yc_poly(i,j), zc_poly(i,j)
               end do

               if      ( char(1:5) .eq. 'BOHR '     ) then
                  continue
               else if ( char(1:9) .eq. 'ANGSTROM ' ) then
                  xc_poly(:,j) = xc_poly(:,j) / bohr2ang
                  yc_poly(:,j) = yc_poly(:,j) / bohr2ang
                  zc_poly(:,j) = zc_poly(:,j) / bohr2ang
               else
                  ierr = 1
               end if

            end do

            if ( ierr .ne. 0 ) then
               write( 6, '(a)' ) &
     &            'Error - structure.dat read incorrectly.'
               write( 6, '(a)' )
            end if

            close( iounit )

         end if

!        /*   mpi communication   */
         call my_mpi_bcast_int_0_world ( ierr )

!        /*   error handling   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine init_polymers_atom_MPI', 33 )

         call my_mpi_bcast_real_2_world( xc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( yc_poly, natom, npoly )
         call my_mpi_bcast_real_2_world( zc_poly, natom, npoly )

      end if

      return
      end
