!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     initialize molecular compositions
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_molcomp
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : iounit, au_length

      use rotor_variables, only : &
     &   x_comp, y_comp, z_comp, x_mol, y_mol, z_mol, physmass_mol, &
     &   natom_comp_max, list_atom_mol, nmol_6, nmol_5, nmol_3, &
     &   ncomp_6, ncomp_5, ncomp_3, nmol, nmol_comp, natom_per_mol, &
     &   comp_spec, natom_per_comp, ndim_comp, ncomp, physmass_mol, &
     &   ndim_mol

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     /*   flag for initial setups   */
      integer, save :: iset = 0

!     /*   integer   */
      integer :: ierr

!     /*   integer   */
      integer :: i, j, k

!     /*   conversion factor   */
      real(8), parameter :: bohr2ang = au_length * 1.d+10

!     /*   character   */
      character(len=8) :: comp_spec_read, char_comp

!-----------------------------------------------------------------------
!     //   this is carried out only once
!-----------------------------------------------------------------------

      if ( iset .eq. 1 ) return

!-----------------------------------------------------------------------
!     //   read number of components
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<components>', 12, iounit, ierr )

!     /*   number of molecular components   */
      read ( iounit, *, iostat=ierr ) ncomp

!     /*   file close   */
      close( iounit )

!     /*   error handling  */
      call error_handling( ierr, 'subroutine init_molcomp', 23 )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   species of component   */
      if ( .not. allocated( comp_spec ) ) &
     &   allocate( comp_spec(ncomp) )

!     /*   number of atoms of component   */
      if ( .not. allocated( natom_per_comp ) ) &
     &   allocate( natom_per_comp(ncomp) )

!     /*   dimension of component   */
      if ( .not. allocated( ndim_comp ) ) &
     &   allocate( ndim_comp(ncomp) )

!-----------------------------------------------------------------------
!     //   read components
!-----------------------------------------------------------------------

!     /*   maximum number of atoms per molecular component   */
      natom_comp_max = 0

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<components>', 12, iounit, ierr )

!     /*   number of molecular components   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of molecular components   */
      do i = 1, ncomp

!        /*   species, number of atoms, dimension   */

         read ( iounit, *, iostat=ierr ) &
     &      comp_spec(i), natom_per_comp(i), char_comp

!        //   error termination
         if ( ierr .ne. 0 ) then
            write( 6, '(a,i4)' ) &
     &         'Error - <components> read incorrectly: component', i
            exit
         end if

!        //   molecular degrees of freedom
         if      ( char_comp(1:8) .eq. 'GENERAL ' ) then
            ndim_comp(i) = 6
         else if ( char_comp(1:8) .eq. 'LINEAR  ' ) then
            ndim_comp(i) = 5
         else if ( char_comp(1:8) .eq. 'MONOATOM' ) then
            ndim_comp(i) = 3
         else
            write( 6, '(a)' ) 'Error - <components> read incorrectly.'
            exit
         end if

!        /*   maximum number of atoms per molecular component   */
         natom_comp_max = max( natom_per_comp(i), natom_comp_max )

         do j = 1, natom_per_comp(i)
            read ( iounit, *, iostat=ierr )
         end do

!     /*   loop of molecular components   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   error handling  */
      call error_handling( ierr, 'subroutine init_molcomp', 23 )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   atomic coordinates of component in molecular axis   */

      if ( .not. allocated( x_comp ) ) &
     &   allocate( x_comp(natom_comp_max,ncomp) )
      if ( .not. allocated( y_comp ) ) &
     &   allocate( y_comp(natom_comp_max,ncomp) )
      if ( .not. allocated( z_comp ) ) &
     &   allocate( z_comp(natom_comp_max,ncomp) )

      x_comp(:,:) = 0.d0
      y_comp(:,:) = 0.d0
      z_comp(:,:) = 0.d0

!-----------------------------------------------------------------------
!     //   number of non-linear, linear and monoatomic components
!-----------------------------------------------------------------------

!     //  number of non-linear components
      ncomp_6 = 0

!     //  number of linear components
      ncomp_5 = 0

!     //  number of monoatomic components
      ncomp_3 = 0

!     /*   loop of molecular components   */
      do i = 1, ncomp

!        /*   non-linear, linear and monoatomic components   */

         if ( ndim_comp(i) .eq. 6 ) ncomp_6 = ncomp_6 + 1
         if ( ndim_comp(i) .eq. 5 ) ncomp_5 = ncomp_5 + 1
         if ( ndim_comp(i) .eq. 3 ) ncomp_3 = ncomp_3 + 1

!     /*   loop of molecular components   */
      end do

!-----------------------------------------------------------------------
!     //   read atomic coordinates of component
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<components>', 12, iounit, ierr )

!     /*   number of molecular components   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of molecular components   */
      do i = 1, ncomp

!        /*   number of molecular species   */
         read ( iounit, *, iostat=ierr )

!        /*   atomic coordinates of component in molecular axis   */

         do j = 1, natom_per_comp(i)
            read ( iounit, *, iostat=ierr ) &
     &         x_comp(j,i), y_comp(j,i), z_comp(j,i)
         end do

!     /*   loop of molecular components   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   error handling  */
      call error_handling( ierr, 'subroutine init_molcomp', 23 )

!-----------------------------------------------------------------------
!     //   read number of molecules
!-----------------------------------------------------------------------

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<molecules>', 11, iounit, ierr )

!     /*   number of molecules   */
      read ( iounit, *, iostat=ierr ) nmol

!     /*   file close   */
      close( iounit )

!     /*   error handling  */
      call error_handling( ierr, 'subroutine init_molcomp', 23 )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   list of atoms in a molecule   */
      if ( .not. allocated( list_atom_mol ) ) &
     &   allocate( list_atom_mol(natom_comp_max,nmol) )

!     //   number of molecules of a component
      if ( .not. allocated( nmol_comp ) ) &
     &   allocate( nmol_comp(ncomp) )

!     /*   number of atoms in a molecule   */
      if ( .not. allocated( natom_per_mol ) ) &
     &   allocate( natom_per_mol(nmol) )

!     /*   atomic coordinates of molecule in molecular axis   */

      if ( .not. allocated( x_mol ) ) &
     &   allocate( x_mol(natom_comp_max,nmol) )
      if ( .not. allocated( y_mol ) ) &
     &   allocate( y_mol(natom_comp_max,nmol) )
      if ( .not. allocated( z_mol ) ) &
     &   allocate( z_mol(natom_comp_max,nmol) )

      x_mol(:,:) = 0.d0
      y_mol(:,:) = 0.d0
      z_mol(:,:) = 0.d0

!     /*   mass of molecule   */
      if ( .not. allocated( physmass_mol ) ) &
     &   allocate( physmass_mol(nmol) )

!     /*   dimension of a molecule   */
      if ( .not. allocated( ndim_mol ) ) &
     &   allocate( ndim_mol(nmol) )

!-----------------------------------------------------------------------
!     //   number of molecules
!-----------------------------------------------------------------------

!     //  number of non-linear molecules
      nmol_6 = 0

!     //  number of linear molecules
      nmol_5 = 0

!     //  number of monoatomic molecules
      nmol_3 = 0

!     /*   number of molecules of a component   */
      nmol_comp(:) = 0

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   search for tag    */
      call search_tag ( '<molecules>', 11, iounit, ierr )

!     /*   number of molecules   */
      read ( iounit, *, iostat=ierr )

!     /*   loop of molecules   */
      do i = 1, nmol

!        /*   read component   */
         read ( iounit, *, iostat=ierr ) comp_spec_read

!        /*   step back one line   */
         backspace( iounit )

         k = 1

!        /*   loop of components start   */
         do j = 1, ncomp

!           /*   find species   */
            if ( comp_spec_read .eq. comp_spec(j) ) then

!              /*   species number   */
               k = j

!              /*   exit loop   */
               exit

!           /*   find species   */
            end if

!        /*   loop of components end   */
         end do

!       /*   number of molecules of a component   */
        nmol_comp(k) = nmol_comp(k) + 1

!       /*   number of atoms in a molecule   */
        natom_per_mol(i) = natom_per_comp(k)

!       /*   atomic coordinates of molecule in molecular axis   */

        x_mol(:,i) = x_comp(:,k)
        y_mol(:,i) = y_comp(:,k)
        z_mol(:,i) = z_comp(:,k)

!       /*   number of non-linear, linear and monoatomic molecules   */

        if ( ndim_comp(k) .eq. 6 ) nmol_6 = nmol_6 + 1
        if ( ndim_comp(k) .eq. 5 ) nmol_5 = nmol_5 + 1
        if ( ndim_comp(k) .eq. 3 ) nmol_3 = nmol_3 + 1

!       /*   dimension of a molecule   */
        ndim_mol(i) = ndim_comp(k)

!       /*   list of atoms in a molecule   */

        read( iounit, *, iostat=ierr ) comp_spec_read, &
     &      ( list_atom_mol(j,i), j = 1, natom_per_comp(k) )

!     /*   loop of molecules   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   error handling  */
      call error_handling( ierr, 'subroutine init_molcomp', 23 )

!-----------------------------------------------------------------------
!     //   this is carried out only once
!-----------------------------------------------------------------------

      iset = 1

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end
