!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     initialize metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_bath_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, vxbath, vybath, vzbath, qmass, beta, &
     &   natom, nbead, nnhc, ncolor, myrank

      use meta_variables, only : &
     &   rbath_meta, qmass_meta, vbath_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      real(8) :: vsigma, gasdev

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!-----------------------------------------------------------------------
!     /*   thermostat attached to system                              */
!-----------------------------------------------------------------------

      do k = 1, nbead
      do j = 1, nnhc
      do i = 1, natom
         xbath(i,j,k) = 0.d0
         ybath(i,j,k) = 0.d0
         zbath(i,j,k) = 0.d0
      end do
      end do
      end do

      do k = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(k))
         do j = 1, nnhc
         do i = 1, natom
            vxbath(i,j,k) = vsigma*gasdev()
            vybath(i,j,k) = vsigma*gasdev()
            vzbath(i,j,k) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   thermostat attached to cv                                  */
!-----------------------------------------------------------------------

      rbath_meta(:,:,:,:) = 0.d0
      vbath_meta(:,:,:,:) = 0.d0

      do k = 1, nbead
      do m = 1, ncolor
      do j = 1, nnhc
      do i = 1, nmeta
         vsigma = sqrt(1.d0/beta/qmass_meta(m))
         vbath_meta(i,j,m,k) = vsigma*gasdev()
      end do
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     /*   broadcast                                                  */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3( xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3( ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3( zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3( vzbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_4( rbath_meta, nmeta, nnhc, ncolor, nbead )
      call my_mpi_bcast_real_4( vbath_meta, nmeta, nnhc, ncolor, nbead )

      return
      end





!***********************************************************************
      subroutine init_cv_meta_MPI
!***********************************************************************
!=======================================================================
!
!     initialize cv for metadynamics
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   beta, nbead, myrank

      use meta_variables, only : &
     &   rmeta, smeta, vmeta, fictmass_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, n

      real(8) :: vsigma, gasdev

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!-----------------------------------------------------------------------
!     /*   initial position of cv                                     */
!-----------------------------------------------------------------------

      do m = 1, nbead
      do n = 1, nmeta
         rmeta(n,m) = smeta(n,m)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   initial velocity of cv                                     */
!-----------------------------------------------------------------------

      do m = 1, nbead
      do n = 1, nmeta
         vsigma = sqrt(1.d0/beta/fictmass_meta(n))
         vmeta(n,m) = vsigma*gasdev()
      end do
      end do

!-----------------------------------------------------------------------
!     /*   master process only                                        */
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     /*   mpi communication                                          */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2( rmeta, nmeta, nbead )
      call my_mpi_bcast_real_2( vmeta, nmeta, nbead )

      return
      end





!***********************************************************************
      subroutine init_hills_meta_MPI
!***********************************************************************
!=======================================================================
!
!     initialize hills
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   nbead, iounit, myrank

      use meta_variables, only : &
     &   pot_lifted, gc_meta, gc_meta_save, rmeta, &
     &   icv_start, ng_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, n, ierr

!-----------------------------------------------------------------------
!     /*   default                                                    */
!-----------------------------------------------------------------------

      ierr = 0

!-----------------------------------------------------------------------
!     /*   initialize number of gaussians                             */
!-----------------------------------------------------------------------

      ng_meta = 0

!-----------------------------------------------------------------------
!     /*   initialize gaussian centers                                */
!-----------------------------------------------------------------------

      gc_meta(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   history                                                    */
!-----------------------------------------------------------------------

      if      ( icv_start .eq. 0 ) then

!        //   initialize data   //

!        /*   potential lifted by gaussian   */
         pot_lifted = 0.d0

!        /*   last gaussian center   */

         do m = 1, nbead
         do n = 1, nmeta
            gc_meta_save(n,m) = rmeta(n,m)
         end do
         end do

      else if ( icv_start .eq. 1 ) then

!        //   read data to restart previous run   //

!        /*   master process only   */
         if ( myrank .eq. 0 ) then

         open ( iounit, file = 'cv.ini', status = 'unknown' )

!           /*   read skip   */

            do m = 1, nbead
            do n = 1, nmeta
               read ( iounit, *, iostat=ierr )
            end do
            end do

!           /*   potential lifted by gaussian   */

            read ( iounit, *, iostat=ierr ) pot_lifted

!           /*   last gaussian center   */

            do m = 1, nbead
            do n = 1, nmeta
               read ( iounit, *, iostat=ierr ) gc_meta_save(n,m)
            end do
            end do

         close ( iounit )

!        /*   master process only   */
         end if

!        /*   mpi communication   */
         call my_mpi_bcast_real_0( pot_lifted )
         call my_mpi_bcast_real_2( gc_meta_save, nmeta, nbead )

!        /*   check error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine init_hills_meta_MPI', 30 )

      else if ( icv_start .eq. 2 ) then

         continue

      end if

      return
      end
