!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jun 29, 2025 by M. Shiga
!      Description:     masses of BCMD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_mass_pibcmd_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for path integral brownian molecular dynamics
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fictmass, physmass, amu_mass_earth, dnmmass, omega_p2, pi, &
     &   dt, au_time, beta, hbar, tau_bcmd, natom, nbead, &
     &   scale_bcmd, integrator_bcmd, iounit, kickstep_bcmd, &
     &   atom_change, au_mass, amu_mass

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr

      real(8) :: omega2_bcmd, dt_fs_bcmd, dt_kick_bcmd, huge, amu

!-----------------------------------------------------------------------
!     /*   bcmd integrator                                            */
!-----------------------------------------------------------------------

      call read_char_MPI &
     &   ( integrator_bcmd, 12, '<integrator_bcmd>', 17, iounit )

!-----------------------------------------------------------------------
!     /*   bcmd decoherence time ( relative to beta hbar )            */
!-----------------------------------------------------------------------

      call read_real1_MPI &
     &   ( scale_bcmd, '<scale_bcmd>', 12, iounit )

!-----------------------------------------------------------------------
!     /*   kick step interval                                         */
!-----------------------------------------------------------------------

      call read_int1_MPI &
     &   ( kickstep_bcmd, '<kickstep_bcmd>', 15, iounit )

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,:) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            read ( atom_change(j)(1:32), *, iostat=ierr ) amu

            if ( amu .gt. 0 ) then

               physmass(j) = amu * amu_mass / au_mass

               fictmass(j,:) = amu * amu_mass / au_mass

            else

               fictmass(j,1) = abs(amu) * amu_mass / au_mass

            end if

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

!     /*   relaxation time   */
      tau_bcmd = ( beta * hbar ) * scale_bcmd

!     //   kick step size
      dt_kick_bcmd = dt * dble(kickstep_bcmd)

!     /*   bcmd chain frequency   */
      omega2_bcmd = 1.d0 / ( (0.5d0*dt_kick_bcmd) * tau_bcmd )

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = dnmmass(j,i) * omega_p2 / omega2_bcmd
      end do
      end do

      dt_fs_bcmd = 0.5d0 / sqrt(omega2_bcmd) * (au_time/1.d-15)

!      if ( myrank .eq. 0 ) then
!         write( 6, '(a,f8.4)' )
!     &     'Predicted step size for BCMD [fs]:', dt_fs_bcmd
!         write( 6, * )
!      end if

      return
      end
