!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 19, 2022 by M. Shiga
!      Description:     masses of PIMD, RPMD, CMD, PIHMC, multiscale MD
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_mass_md_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, amu_mass_earth, amu_mass, au_mass, natom, &
     &   atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,:) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            physmass(j) = physmass(j) * huge

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            physmass(j) = physmass(j) * huge

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            physmass(j) = amu_mass / au_mass

            fictmass(j,:) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_md_MPI", 27 )

         end if

      end do

      return
      end





!***********************************************************************
      subroutine init_mass_pimd_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for path integral MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dnmmass, physmass, fictmass, amu_mass_earth, gamma_p2, &
     &   amu_mass, au_mass, natom, nbead, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,1) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_pimd_MPI", 29 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = dnmmass(j,i) * gamma_p2
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_rpmd_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for ring polymer MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, amu_mass_earth, &
     &   amu_mass, au_mass, natom, nbead, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,1) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_rpmd_MPI", 29 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = physmass(j)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_rpmd_nvt_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for ring polymer MD (NVT)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, amu_mass_earth, &
     &   amu_mass, au_mass, natom, nbead, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,1) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_rpmd_npt_MPI", 33 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = physmass(j)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_cmd_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for centroid MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dnmmass, physmass, fictmass, amu_mass_earth, gamma_p2, &
     &   amu_mass, au_mass, gamma2, gamma2_cent, natom, nbead, &
     &   atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,1) = physmass(j) * gamma2_cent
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_cmd_MPI", 28 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = dnmmass(j,i) * gamma_p2 * gamma2
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_pihmc_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for path integral hybrid Monte Carlo
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dnmmass, physmass, fictmass, amu_mass_earth, gamma_p2, &
     &   amu_mass, au_mass, natom, nbead, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,1) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,1) = amu_mass / au_mass

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_pihmc_MPI", 30 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         fictmass(j,i) = dnmmass(j,i) * gamma_p2
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_multi_me_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for QM/MM multiscale MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dnmmass, physmass, fictmass, amu_mass_earth, gamma_p2, &
     &   amu_mass, au_mass, natom, nbead, atom_change

      use qmmm_variables, only : &
     &   layer, nmulti

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: huge

!-----------------------------------------------------------------------
!     /*  huge mass                                                    */
!-----------------------------------------------------------------------

      huge = amu_mass_earth

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         if      ( layer(j)(1:1) .eq. 'A' ) then
            fictmass(j,1) = physmass(j)
         else if ( layer(j)(1:1) .eq. 'B' ) then
            fictmass(j,1) = physmass(j) / dble(nmulti*nmulti)
            if ( layer(j)(1:2) .eq. 'BL' ) then
               fictmass(j,1) = physmass(j)
            end if
         end if
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

            fictmass(j,1) = fictmass(j,1) * huge

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

            fictmass(j,:) = fictmass(j,:) * huge

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            if      ( layer(j)(1:1) .eq. 'A' ) then
               fictmass(j,1) = amu_mass / au_mass
            else if ( layer(j)(1:1) .eq. 'B' ) then
               fictmass(j,1) = amu_mass / au_mass / dble(nmulti*nmulti)
               if ( layer(j)(1:2) .eq. 'BL' ) then
                  fictmass(j,1) = amu_mass / au_mass
               end if
            end if

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_multi_me_MPI", 33 )

         end if

      end do

!-----------------------------------------------------------------------
!     /*  fictitious masses of non-centroids                          */
!-----------------------------------------------------------------------

      do j = 1, natom
      do i = 2, nbead
         if      ( layer(j)(1:1) .eq. 'A' ) then
            fictmass(j,i) = dnmmass(j,i) * gamma_p2
         else if ( layer(j)(1:2) .eq. 'BL' ) then
            fictmass(j,i) = dnmmass(j,i) * gamma_p2
         else if ( layer(j)(1:1) .eq. 'B' ) then
            fictmass(j,i) = dnmmass(j,i) * gamma_p2
         end if
      end do
      end do

      return
      end





!***********************************************************************
      subroutine init_mass_string_MPI
!***********************************************************************
!=======================================================================
!
!     set up masses for MD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, fictmass, amu_mass, au_mass, natom, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j

!      real(8) :: huge = 1.d+12

!-----------------------------------------------------------------------
!     /*  fictitious masses of centroids                              */
!-----------------------------------------------------------------------

      do j = 1, natom
         fictmass(j,:) = physmass(j)
      end do

!-----------------------------------------------------------------------
!     /*  when mass scaling is applied                                */
!-----------------------------------------------------------------------

      do j = 1, natom

         if ( atom_change(j)(1:5) .eq. 'NONE ' ) then

            continue

         else if ( atom_change(j)(1:7) .eq. 'FREEZE ' ) then

!            fictmass(j,:) = fictmass(j,:) * huge
            continue

         else if ( atom_change(j)(1:6) .eq. 'HEAVY ' ) then

!            fictmass(j,:) = fictmass(j,:) * huge
            continue

         else if ( atom_change(j)(1:9) .eq. 'HYDROGEN ' ) then

            fictmass(j,:) = amu_mass / au_mass

         else if ( atom_change(j)(1:3) .eq. 'FIX' ) then

            continue

         else

            call error_handling_MPI &
     &         ( 1, "subroutine init_mass_string_MPI", 31 )

         end if

      end do

      return
      end
