!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     initialize and finalize external programs
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_extprograms_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : ipotential

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initialize potential                                       */
!-----------------------------------------------------------------------

!     ===  ABINIT-MP  ===

      if ( ipotential(1:10) .eq. 'ABINIT-MP ' ) then

         call init_abinit_mp_MPI

!     ===  VASP  ===

      else if ( ipotential(1:5) .eq. 'VASP ' ) then

         call init_vasp_MPI

!     ===  QUANTUM ESPRESSO  ===

      else if ( ipotential(1:3) .eq. 'QE ' ) then

         call init_qe_MPI

!     ===  DUAL  ===

      else if ( ipotential(1:5) .eq. 'DUAL ' ) then

         call init_dual_MPI

      else if ( ipotential(1:7) .eq. 'PHASE0 ' ) then

         call init_phase0_MPI

!     ===  PFP  ===

      else if ( ipotential(1:4) .eq. 'PFP ' ) then

         call initialize_pfp_MPI

      end if

      return
      end





!***********************************************************************
      subroutine init_dual_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, myrank

      use dual_variables, only : idual_hi, idual_lo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   only master process   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dual_potential>', 16, iounit, ierr )

!        /*   dual:  high and low levels   */
         read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         if ( ierr .ne. 0 ) then

!           /*   file close   */
            close( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<dual_potential>', 16, iounit, ierr )

!           /*   dual:  high and low levels   */
            read ( iounit, *, iostat=ierr ) idual_hi, idual_lo

!        /*   on error read default   */
         end if

!        /*   file close   */
         close( iounit )

!     /*   only master process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   stop on error   */
      call error_handling_MPI( ierr, 'subroutine init_dual_MPI', 24)

      call my_mpi_bcast_char_0 ( idual_hi, 80 )
      call my_mpi_bcast_char_0 ( idual_lo, 80 )

!-----------------------------------------------------------------------
!     /*   initialize potential                                       */
!-----------------------------------------------------------------------

!     ===  ABINIT-MP  ===

      if ( ( idual_hi(1:10) .eq. 'ABINIT-MP ' ) .or. &
     &     ( idual_lo(1:10) .eq. 'ABINIT-MP ' ) ) then

         call init_abinit_mp_MPI

!     ===  VASP  ===

      else if ( ( idual_hi(1:5) .eq. 'VASP ' ) .or. &
     &          ( idual_lo(1:5) .eq. 'VASP ' ) ) then

         call init_vasp_MPI

!     ===  QUANTUM ESPRESSO  ===

      else if ( ( idual_hi(1:3) .eq. 'QE ' ) .or. &
     &          ( idual_lo(1:3) .eq. 'QE ' ) ) then

         call init_qe_MPI

      else if ( ( idual_hi(1:7) .eq. 'PHASE0 ' ) .or. &
     &          ( idual_lo(1:7) .eq. 'PHASE0 ' ) ) then

         call init_phase0_MPI

      else if ( ( idual_hi(1:4) .eq. 'PFP ' ) .or. &
     &          ( idual_lo(1:4) .eq. 'PFP ' ) ) then

         call initialize_pfp_MPI

      end if

      return
      end





!***********************************************************************
      subroutine finalize_extprograms_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : ipotential

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   finalize potential                                         */
!-----------------------------------------------------------------------

!     ===  ABINIT-MP  ===

      if ( ipotential(1:10) .eq. 'ABINIT-MP ' ) then

         call finalize_abinit_mp_MPI

!     ===  VASP  ===

      else if ( ipotential(1:5) .eq. 'VASP ' ) then

         call finalize_vasp_MPI

!     ===  VASP6  ===

      else if ( ipotential(1:6) .eq. 'VASP6 ' ) then

         call finalize_vasp6_MPI

!     ===  QUANTUM ESPRESSO  ===

      else if ( ipotential(1:3) .eq. 'QE ' ) then

         call finalize_qe_MPI

!     ===  QUANTUM ESPRESSO  (git version) ===

      else if ( ipotential(1:5) .eq. 'QEGIT ' ) then

         call finalize_qegit_MPI

!     ===  PHASE0  ===

      else if ( ipotential(1:7) .eq. 'PHASE0 ' ) then

         call finalize_phase0_MPI

!     ===  QUANTUM ESPRESSO  ===

      else if ( ipotential(1:7) .eq. 'NTCHEM ' ) then

         call finalize_ntchem_MPI

!     ===  DFTB+  ===

      else if (ipotential(1:8) .eq. 'DFTBLIB ') then

         call finalize_libdftb_MPI

!     ===  CP2K  ===

      else if (ipotential(1:8) .eq. 'CP2KLIB ') then

         call finalize_libcp2k_MPI

      else if ( ipotential(1:4) .eq. 'PFP ' ) then

         call finalize_pfp_MPI

      end if

      return
      end

