!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     read atomic position or centroid position
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_centroid_MPI
!***********************************************************************

!     //   shared variables
      use common_variables, only : input_style

!     //   local variables
      implicit none

!     /*   input style old   */
      if      ( input_style(1:4) .eq. 'OLD ' ) then

!        /*   read cartesian from centroid.dat   */
         call init_centroid_old_MPI

!     /*   input style new   */
      else if ( input_style(1:4) .eq. 'NEW ' ) then

!        /*   read cartesian from structure.dat   */
         call init_centroid_new_MPI

!     /*   input style   */
      end if

      return
      end





!***********************************************************************
      subroutine init_centroid_old_MPI
!***********************************************************************
!=======================================================================
!
!     initial configuration of centroid
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, myrank, natom, nbead, iounit, mbox

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr

!-----------------------------------------------------------------------
!     /*   centroid coordinates are read from file                    */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         open ( iounit, file = 'centroid.dat', status = 'unknown' )

         do i = 1, natom
            read ( iounit, *, iostat=ierr ) ux(i,1), uy(i,1), uz(i,1)
         end do

         close( iounit )

         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error - centroid.dat read incorrectly.'
            write( 6, '(a)' )
         end if

      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &    ( ierr, 'subroutine init_centroid_old_MPI', 32 )

      call my_mpi_bcast_real_2 ( ux, natom, nbead )
      call my_mpi_bcast_real_2 ( uy, natom, nbead )
      call my_mpi_bcast_real_2 ( uz, natom, nbead )

!     /*   initial primary cell   */
      mbox(:,:,:) = 0

      return
      end





!***********************************************************************
      subroutine init_centroid_new_MPI
!***********************************************************************
!=======================================================================
!
!     initial configuration
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, au_length, natom, nbead, iounit, mbox, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr

      real(8), parameter :: bohr2ang = au_length * 1.d+10

      character(len=10) :: char

!-----------------------------------------------------------------------
!     /*   centroid coordinates are read from file                    */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         open ( iounit, file = 'structure.dat', status = 'unknown' )

         read ( iounit, * )
         read ( iounit, *, iostat=ierr ) char
         if ( char(1:5) .eq. 'BOHR ' ) then
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            char, ux(i,1), uy(i,1), uz(i,1)
            end do
         else if ( char(1:9) .eq. 'ANGSTROM ' ) then
            do i = 1, natom
               read ( iounit, *, iostat=ierr ) &
     &            char, ux(i,1), uy(i,1), uz(i,1)
            end do
            ux(:,1) = ux(:,1) / bohr2ang
            uy(:,1) = uy(:,1) / bohr2ang
            uz(:,1) = uz(:,1) / bohr2ang
         else
            ierr = 1
         end if

         close( iounit )

         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error - structure.dat read incorrectly.'
            write( 6, '(a)' )
         end if

      end if

!     /*   mpi communication   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &    ( ierr, 'subroutine init_centroid_new_MPI', 32 )

      call my_mpi_bcast_real_2 ( ux, natom, nbead )
      call my_mpi_bcast_real_2 ( uy, natom, nbead )
      call my_mpi_bcast_real_2 ( uz, natom, nbead )

!     /*   initial primary cell   */
      mbox(:,:,:) = 0

      return
      end

