!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     initialize thermostats
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine init_bath_nhc_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   rbath_cent, vbath_cent, qmass_cent, qmass, xbath, ybath, zbath, &
     &   vxbath, vybath, vzbath, beta, ncolor, nnhc, nbead, natom, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iatom, imode, inhc, m

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*  centroid coordinates                                        */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         rbath_cent(i,m) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*  centroid velocities                                         */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         vsigma = sqrt(1.d0/beta/qmass_cent(i,m))
         vbath_cent(i,m) = vsigma*gasdev()
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid coordinates                                    */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc  = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid velocities                                     */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2 (  rbath_cent, nnhc, ncolor  )

      call my_mpi_bcast_real_2 (  vbath_cent, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      return
      end





!***********************************************************************
      subroutine init_bath_nhcs_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath_cent, ybath_cent, zbath_cent,xbath, ybath, zbath, &
     &   vxbath_cent, vybath_cent, vzbath_cent, qmass_cent, qmass, &
     &   vxbath, vybath, vzbath, beta, ncolor, nnhc, nbead, natom, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iatom, imode, inhc, m

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*  centroid coordinates                                        */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         xbath_cent(1,i,m) = 0.d0
         ybath_cent(1,i,m) = 0.d0
         zbath_cent(1,i,m) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*  centroid velocities                                         */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         vsigma = sqrt(1.d0/beta/qmass_cent(i,m))
         vxbath_cent(1,i,m) = vsigma*gasdev()
         vybath_cent(1,i,m) = vsigma*gasdev()
         vzbath_cent(1,i,m) = vsigma*gasdev()
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid coordinates                                    */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc  = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid velocities                                     */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3 (   xbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   ybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   zbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  vxbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vzbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      return
      end





!***********************************************************************
      subroutine init_bath_mnhc_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath_cent, ybath_cent, zbath_cent,xbath, ybath, zbath, &
     &   vxbath_cent, vybath_cent, vzbath_cent, qmass_cent, qmass, &
     &   vxbath, vybath, vzbath, beta, ncolor, nnhc, nbead, natom, &
     &   myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iatom, imode, inhc, j, m

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*  centroid coordinates                                        */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         xbath_cent(j,i,m) = 0.d0
         ybath_cent(j,i,m) = 0.d0
         zbath_cent(j,i,m) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  centroid velocities                                         */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         vsigma = sqrt(1.d0/beta/qmass_cent(i,m))
         vxbath_cent(j,i,m) = vsigma*gasdev()
         vybath_cent(j,i,m) = vsigma*gasdev()
         vzbath_cent(j,i,m) = vsigma*gasdev()
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid coordinates                                    */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid velocities                                     */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3 (   xbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   ybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   zbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  vxbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vzbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      return
      end





!***********************************************************************
      subroutine init_bath_c2_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath_cent, ybath_cent, zbath_cent, xbath, ybath, zbath, &
     &   vxbath_cent, vybath_cent, vzbath_cent, qmass_cent, qmass, &
     &   vxbath, vybath, vzbath, rbath_box, vbath_box, beta, qmass_box, &
     &   ncolor, nnhc, nbead, natom, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iatom, imode, inhc, j, m

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*  centroid coordinates                                        */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         xbath_cent(j,i,m) = 0.d0
         ybath_cent(j,i,m) = 0.d0
         zbath_cent(j,i,m) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  centroid velocities                                         */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         vsigma = sqrt(1.d0/beta/qmass_cent(i,m))
         vxbath_cent(j,i,m) = vsigma*gasdev()
         vybath_cent(j,i,m) = vsigma*gasdev()
         vzbath_cent(j,i,m) = vsigma*gasdev()
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid coordinates                                    */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid velocities                                     */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*  box coordinates                                             */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         rbath_box(i,m) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*  box velocities                                              */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
         vsigma = sqrt(1.d0/beta/qmass_box(i,m))
         vbath_box(i,m) = vsigma*gasdev()
      end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3 (   xbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   ybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   zbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  vxbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vzbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_2 (  rbath_box, nnhc, ncolor )
      call my_mpi_bcast_real_2 (  vbath_box, nnhc, ncolor )

      return
      end





!***********************************************************************
      subroutine init_bath_pp_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath_cent, ybath_cent, zbath_cent,xbath, ybath, zbath, &
     &   vxbath_cent, vybath_cent, vzbath_cent, qmass_cent, qmass, &
     &   vxbath, vybath, vzbath, qmass_box, &
     &   xbath_box, ybath_box, zbath_box, vxbath_box, vybath_box, &
     &   vzbath_box, beta, ncolor, nnhc, nbead, natom, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, iatom, imode, inhc, j, l, m

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*  centroid coordinates                                        */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         xbath_cent(j,i,m) = 0.d0
         ybath_cent(j,i,m) = 0.d0
         zbath_cent(j,i,m) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  centroid velocities                                         */
!-----------------------------------------------------------------------

      do i = 1, nnhc
      do j = 1, natom
      do m = 1, ncolor
         vsigma = sqrt(1.d0/beta/qmass_cent(i,m))
         vxbath_cent(j,i,m) = vsigma*gasdev()
         vybath_cent(j,i,m) = vsigma*gasdev()
         vzbath_cent(j,i,m) = vsigma*gasdev()
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid coordinates                                    */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  non-centroid velocities                                     */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*  box coordinates                                             */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, 3
         xbath_box(j,i,m) = 0.d0
         ybath_box(j,i,m) = 0.d0
         zbath_box(j,i,m) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*  box velocities                                              */
!-----------------------------------------------------------------------

      do m = 1, ncolor
      do i = 1, nnhc
      do j = 1, 3
         vsigma = sqrt(1.d0/beta/qmass_box(i,m))
         vxbath_box(j,i,m) = vsigma*gasdev()
         vybath_box(j,i,m) = vsigma*gasdev()
         vzbath_box(j,i,m) = vsigma*gasdev()
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   symmetrize thermostat velocity                             */
!-----------------------------------------------------------------------

      do l = 1, ncolor
      do i = 1, nnhc
         vxbath_box(2,i,l) &
     &      = 0.5d0 * ( vxbath_box(2,i,l) + vybath_box(1,i,l) )
         vybath_box(1,i,l) = vxbath_box(2,i,l)
         vybath_box(3,i,l) &
     &      = 0.5d0 * ( vybath_box(3,i,l) + vzbath_box(2,i,l) )
         vzbath_box(2,i,l) = vybath_box(3,i,l)
         vzbath_box(1,i,l) &
     &      = 0.5d0 * ( vzbath_box(1,i,l) + vxbath_box(3,i,l) )
         vxbath_box(3,i,l) = vzbath_box(1,i,l)
      end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3 (   xbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   ybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (   zbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  vxbath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vybath_cent, natom, nnhc, ncolor  )
      call my_mpi_bcast_real_3 (  vzbath_cent, natom, nnhc, ncolor  )

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 (  xbath_box, 3, nnhc, ncolor )
      call my_mpi_bcast_real_3 (  ybath_box, 3, nnhc, ncolor )
      call my_mpi_bcast_real_3 (  zbath_box, 3, nnhc, ncolor )

      call my_mpi_bcast_real_3 ( vxbath_box, 3, nnhc, ncolor )
      call my_mpi_bcast_real_3 ( vybath_box, 3, nnhc, ncolor )
      call my_mpi_bcast_real_3 ( vzbath_box, 3, nnhc, ncolor )

      return
      end





!***********************************************************************
      subroutine init_bath_mnhc_om_MPI
!***********************************************************************
!=======================================================================
!
!     initialize thermostat coordinates and thermostat velocities
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xbath, ybath, zbath, qmass, vxbath, vybath, vzbath, beta, &
     &   nnhc, nbead, natom, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: iatom, imode, inhc

      real(8) :: gasdev, vsigma

!-----------------------------------------------------------------------
      if ( myrank .eq. 0 ) then
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!     /*   coordinates                                                */
!-----------------------------------------------------------------------

      do imode = 1, nbead
      do inhc = 1, nnhc
      do iatom = 1, natom
         xbath(iatom,inhc,imode) = 0.d0
         ybath(iatom,inhc,imode) = 0.d0
         zbath(iatom,inhc,imode) = 0.d0
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   velocities                                                 */
!-----------------------------------------------------------------------

      do imode = 1, nbead
         vsigma = sqrt(1.d0/beta/qmass(imode))
         do inhc  = 1, nnhc
         do iatom = 1, natom
            vxbath(iatom,inhc,imode) = vsigma*gasdev()
            vybath(iatom,inhc,imode) = vsigma*gasdev()
            vzbath(iatom,inhc,imode) = vsigma*gasdev()
         end do
         end do
      end do

!-----------------------------------------------------------------------
      end if
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_3 (  xbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  ybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 (  zbath, natom, nnhc, nbead )

      call my_mpi_bcast_real_3 ( vxbath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vybath, natom, nnhc, nbead )
      call my_mpi_bcast_real_3 ( vzbath, natom, nnhc, nbead )

      return
      end
