!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     surface hopping in Tully fewest switches
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine hopping_tfs_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, vx, vy, vz, pot, dt, physmass, hbar, nbead, &
     &   natom, myrank

      use multistate_variables, only : &
     &   cstate, dxstate, dystate, dzstate, vstate, &
     &   gxstate, gystate, gzstate, istate_tfs, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      integer, dimension(nbead) :: istate_tfs_old

      real(8) :: vd, rn, ggmin, ggmax, aij, bij, cij, gij, ranf1

      complex(8), dimension(nstate,nstate) :: aa

      real(8),    dimension(nstate,nstate) :: bb
      real(8),    dimension(nstate,nstate) :: gg

      real(8), dimension(nbead) :: rn_bead

!-----------------------------------------------------------------------
!     /*   random number                                              */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         do l = 1, nbead
            rn_bead(l) = ranf1()
         end do
      end if

      call my_mpi_bcast_real_1( rn_bead, nbead )

!-----------------------------------------------------------------------
!     /*   surface hopping                                            */
!-----------------------------------------------------------------------

      do l = 1, nbead

!        /*   preserve the original state   */
         istate_tfs_old(l) = istate_tfs(l)

!        /*   make aa matrix   */

         do j = 1, nstate
         do i = 1, nstate
            aa(i,j) = cstate(i,l)*dconjg(cstate(j,l))
         end do
         end do

!        /*   make bb matrix   */

         do j = 1, nstate
         do i = 1, nstate

            vd = 0.d0

            do k = 1, natom
               vd = vd + vx(k,l)*dxstate(i,j,k,l) &
     &                 + vy(k,l)*dystate(i,j,k,l) &
     &                 + vz(k,l)*dzstate(i,j,k,l)
            end do

            bb(i,j) = 2.d0/hbar*dimag( dconjg(aa(i,j))*vstate(i,j,l) ) &
     &              - 2.d0*dreal( dconjg(aa(i,j))*vd )

         end do
         end do

!        /*   make gg matrix   */

         do j = 1, nstate
         do i = 1, nstate

            if ( dreal(aa(i,i)) .ne. 0.d0 ) then
               gg(i,j) = dt*bb(j,i)/dreal(aa(i,i))
            else
               gg(i,j) = 0.d0
            end if

            if ( gg(i,j) .lt. 0.d0 ) gg(i,j) = 0.d0

         end do
         end do

!        /*   random number   */
         rn = rn_bead(l)

!        /*   examine a hop   */

         if ( rn .lt. gg(istate_tfs(l),1) ) istate_tfs(l) = 1

         do j = 2, nstate

            ggmin = gg(istate_tfs(l),j-1)
            ggmax = gg(istate_tfs(l),j-1) + gg(istate_tfs(l),j)

            if ( ( rn .gt. ggmin ) .and. ( rn .lt. ggmax ) ) then
               istate_tfs(l) = j
            end if

         end do

!-----------------------------------------------------------------------
!        /*   velocity scaling is used only when hopped               */
!-----------------------------------------------------------------------

         if ( istate_tfs(l) .eq. istate_tfs_old(l) ) cycle

!-----------------------------------------------------------------------
!        /*   velocity scaling using nonadiabatic coupling vector     */
!-----------------------------------------------------------------------

!        /*   original state   */
         i = istate_tfs_old(l)

!        /*   new state   */
         j = istate_tfs(l)

!        /*   make aij   */

         aij = 0.d0

         do k = 1, natom

            aij = aij &
     &          + 0.5d0*dxstate(i,j,k,l)*dxstate(i,j,k,l)/physmass(k) &
     &          + 0.5d0*dystate(i,j,k,l)*dystate(i,j,k,l)/physmass(k) &
     &          + 0.5d0*dzstate(i,j,k,l)*dzstate(i,j,k,l)/physmass(k)

         end do

!        /*   make bij   */

         bij = 0.d0

         do k = 1, natom

            bij = bij + vx(k,l)*dxstate(i,j,k,l) &
     &                + vy(k,l)*dystate(i,j,k,l) &
     &                + vz(k,l)*dzstate(i,j,k,l)

         end do

!        /*   make cij from aij and bij  */

         cij = bij*bij + 4.d0*aij*(vstate(i,i,l)-vstate(j,j,l))

!        /*   if cij is larger than 1, accept the hop   */

         if ( cij .gt. 0.d0 ) then

            if ( bij .ge. 0.d0 ) then
               gij = ( bij - sqrt(cij) )/(2.d0*aij)
            else
               gij = ( bij + sqrt(cij) )/(2.d0*aij)
            end if

!           /*   shift velocity to conserve energy  */

            do k = 1, natom
               vx(k,l) = vx(k,l) - gij*dxstate(i,j,k,l)/physmass(k)
               vy(k,l) = vy(k,l) - gij*dystate(i,j,k,l)/physmass(k)
               vz(k,l) = vz(k,l) - gij*dzstate(i,j,k,l)/physmass(k)
            end do

!        /*   if cij is smaller than 1, reject the hop   */

         else if ( cij .lt. 0.d0 ) then

            gij = bij/aij

!           /*   shift back the velocity   */

            do k = 1, natom
               vx(k,l) = vx(k,l) - gij*dxstate(i,j,k,l)/physmass(k)
               vy(k,l) = vy(k,l) - gij*dystate(i,j,k,l)/physmass(k)
               vz(k,l) = vz(k,l) - gij*dzstate(i,j,k,l)/physmass(k)
            end do

!           /*   hop back to original state   */
            istate_tfs(l) = istate_tfs_old(l)

         end if

!        /*   update potential   */
         pot(l) = vstate(istate_tfs(l),istate_tfs(l),l)

!        /*   update forces   */

         do k = 1, natom
            fx(k,l) = - gxstate(istate_tfs(l),istate_tfs(l),k,l)
            fy(k,l) = - gystate(istate_tfs(l),istate_tfs(l),k,l)
            fz(k,l) = - gzstate(istate_tfs(l),istate_tfs(l),k,l)
         end do

      end do

      return
      end
