!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Dec 27, 2018 by M. Shiga
!      Description:     potential as force integration along the string
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getpot_string
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    fictmass, pot, natom

      use string_variables, only : &
     &    pot_string, pot_string_mean, pot_string_max, pot_string_min, &
     &    rc_arc, pot_arc, narc, i_string_max

      use string_variables, only : &
     &    xmax_string, ymax_string, zmax_string, dmax_string

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   paramter s from reactant (0) to product (1)   */
      real(8) :: s

!     /*   mass weighted coordinates   */
      real(8), dimension(3*natom) :: r1, r2, f1, f2

!     /*   integers   */
      integer :: i, j, k

!     /*   real numbers   */
      real(8) :: dx, dy, dz, factor, drc2, pot_s

!-----------------------------------------------------------------------
!     /*   initialize spline - s:r                                    */
!-----------------------------------------------------------------------

      call spline_init_crd_string

!-----------------------------------------------------------------------
!     /*   initialize spline - s:f                                    */
!-----------------------------------------------------------------------

      call spline_init_frc_string

!-----------------------------------------------------------------------
!     /*   initialize spline - s:pot                                  */
!-----------------------------------------------------------------------

      call spline_init_pot_string

!c-----------------------------------------------------------------------
!     /*   calculate free energy                                      */
!-----------------------------------------------------------------------

!     /*   force integral   */
      pot_string(1) = pot(1)

!     /*   force integral   */
      pot_arc(1) = pot(1)

!     /*   mean of force integral   */
      pot_string_mean = pot(1)

!     /*   max of force integral   */
      pot_string_max = pot(1)

!     /*   min of force integral   */
      pot_string_min = pot(1)

!     /*   reaction coordinate   */
      rc_arc(1) = 0.d0

!     /*   parameter s at the reactant is zero  */
      s = 0.d0

!     /*   get old mass weighted coordinates and old force   */
      call spline_crd_frc_string( r1, f1, pot_s, s )

!     /*   scan from the reactant to the product   */
      do j = 2, narc

!        /*   parameter s   */
         s = dble(j-1) / dble(narc-1)

!        /*   get new mass weighted coordinates and new force   */
         call spline_crd_frc_string( r2, f2, pot_s, s )

!        /*   calculate arc length by linear approximation   */
         k = 0

!        /*   measure of reaction coordinate   */
         drc2 = 0.d0

!        //   loop of atoms
         do i = 1, natom

!c           /*   mass factor   */
!            factor = sqrt( fictmass(i,1) )

!           /*   index   */
            k = k + 1

!           /*   displacement    */
            dx = r2(k) - r1(k)

!           /*   index   */
            k = k + 1

!           /*   displacement   */
            dy = r2(k) - r1(k)

!           /*   index   */
            k = k + 1

!           /*   displacement   */
            dz = r2(k) - r1(k)

!           /*   measure of reaction coordinate   */
            drc2 = drc2 + dx*dx + dy*dy + dz*dz

!        //   loop of atoms
         end do

!        /*   reaction coordinate   */
         rc_arc(j) = rc_arc(j-1) + sqrt(drc2)

!        //   at maximum
         if ( pot_arc(j) .gt. pot_string_max ) then

!           //   place
            i_string_max = j

!           //   displacement
            do k = 1, 3*natom
               dmax_string(k) = ( r2(k) - r1(k) ) / sqrt(drc2)
            end do

!           //   index
            k = 0

!           //   loop of atoms
            do i = 1, natom

!              /*   mass factor   */
               factor = sqrt( fictmass(i,1) )

!              /*   index   */
               k = k + 1

!              /*   displacement    */
               xmax_string(i) = r2(k) / factor

!              /*   index   */
               k = k + 1

!              /*   displacement   */
               ymax_string(i) = r2(k) / factor

!              /*   index   */
               k = k + 1

!              /*   displacement   */
               zmax_string(i) = r2(k) / factor

!           //   loop of atoms
            end do

!        //   at maximum
         end if

!        //   potential
         pot_arc(j) = pot_s

!        /*   mean of force integral   */
         pot_string_mean = pot_string_mean + pot_arc(j)

!        /*   max of force integral   */
         pot_string_max = max(pot_string_max,pot_arc(j))

!        /*   min of force integral   */
         pot_string_min = min(pot_string_min,pot_arc(j))

!        /*   save old mass weighted coordinates */
         r1(:) = r2(:)

!        /*   save old forces   */
         f1(:) = f2(:)

!     /*   scan from the reactant to the product   */
      end do

!     /*   mean of force integral   */
      pot_string_mean = pot_string_mean / narc

      return
      end





!***********************************************************************
      subroutine adapt_dt_string
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   dt, iexit

!      use common_variables, only :
!     &   pot, nbead

      use string_variables, only : &
     &   auto_string, dtmax_string, dtmin_string

      use string_variables, only : &
     &   pot_string_max

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   last value of potential sum
      real(8), save :: sumpot_last

!     //   current value of potential sum
      real(8) :: sumpot

!     //   acceptance ratio
      real(8) :: ratio

!     //   flag
      integer, save :: iset = 0

!     //   number of acceptable moves
      integer, save :: naccept

!     //   number of trial moves
      integer, save :: ntrial

!-----------------------------------------------------------------------
!     //   parameters
!-----------------------------------------------------------------------

!     //   maximum number of trial moves
      integer :: maxtrial = 5

!     //   dt scaling factor
      real(8), parameter :: scaling_factor = 0.70d0

!     //   minimum value of dt
      real(8), parameter :: dtmin_factor   = 0.01d0

!     //   acceptable ratio
      real(8), parameter :: ratio_accept   = 0.80d0

!     //   inacceptable ratio
      real(8), parameter :: ratio_reject   = 0.40d0

!-----------------------------------------------------------------------
!     //   if dt is smaller than zero, stop
!-----------------------------------------------------------------------

      if ( dt .le. 0.d0 ) then

         iexit = 1

         return

      end if

!-----------------------------------------------------------------------
!     //   automatic step control must be turned on
!-----------------------------------------------------------------------

      if ( auto_string(1:3) .ne. 'ON ' ) return

!-----------------------------------------------------------------------
!     //   start
!-----------------------------------------------------------------------

!     //   initial setting
      if ( iset .eq.  0 ) then

!        //   set a minimum value of dt
         dtmin_string = dtmax_string * dtmin_factor

!        //   acceptable moves
         naccept = 0

!        //   number of moves
         ntrial = 0

!        //   current value of potential sum
!         sumpot = sum(pot(:)) / dble(nbead)
         sumpot = pot_string_max

!        //   laast value of potential sum
         sumpot_last = sumpot

!        //   if dt is smaller than dtmin, stop
         if ( dt .lt. dtmin_string ) iexit = 1

!        //   end of initial setting
         iset = 1

!        //  finish
         return

!     //  end of initial setting
      end if

!     //   current value of potential sum
!      sumpot = sum(pot(:)) / dble(nbead)
      sumpot = pot_string_max

!     //   update number of moves
      ntrial = ntrial + 1

!     //   if move is acceptable update counter
      if ( sumpot .lt. sumpot_last ) naccept = naccept + 1

!     //   only at maxtrial step
      if ( ntrial .lt. maxtrial ) return

!     //   acceptance ratio
      ratio = dble(naccept) / dble(ntrial)

!     //   if ratio is acceptable
      if      ( ratio .ge. ratio_accept ) then

!        //   dt is scaled up
         dt = dt / scaling_factor
         dt = min( dtmax_string, dt )

!     //   if ratio is inacceptable
      else if ( ratio .le. ratio_reject ) then

!        //   dt is scaled down
         dt = dt * scaling_factor**2
!         dt = max( dt, dtmin_string )

!        //   if dt is smaller than dtmin, stop
         if ( dt .lt. dtmin_string ) iexit = 1

!     //   end of if statement
      end if

!     //   reset counter at maxtrial step
      ntrial = 0

!     //   reset acceptable at maxtrial step
      naccept = 0

!     //   save last value of potential sum
      sumpot_last = sumpot

      return
      end
