!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 7, 2019 by M. Shiga
!      Description:     hessian calculation in parallel
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine gethess_paral_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, fxp, fyp, fzp, fxm, fym, fzm, fdiff, &
     &   hessian, ux, uy, uz, fux, fuy, fuz, potential, hamiltonian, &
     &   natom, nbead, istep, istep_start, istep_end, nstep, &
     &   iounit, iexit, atom_change, myrank

      use hess_variables, only : &
     &   ihess_atom, ihess_xyz, ihess_pm, ihess_natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, ibead, itest

      real(8) :: dr, fxn, fyn, fzn

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!           /*   check existence of file   */
            call testfile ( 'fd.out', 6, itest )

!           /*   reset step if file do not exist   */
            if ( itest .eq. 1 ) then
               call system('rm -f step.ini')
               istep_start = 0
            end if

!        /*   master rank   */
         end if

!        /*   communicate step number   */
         call my_mpi_bcast_int_0 ( istep_start )

!        /*   finite difference parameter   */
         call read_real1_MPI ( fdiff, '<fdiff>', 7, iounit )

!        /*   memory allocation:  forces   */

         if ( .not. allocated( fxp ) ) &
     &      allocate( fxp(natom,1) )
         if ( .not. allocated( fyp ) ) &
     &      allocate( fyp(natom,1) )
         if ( .not. allocated( fzp ) ) &
     &      allocate( fzp(natom,1) )
         if ( .not. allocated( fxm ) ) &
     &      allocate( fxm(natom,1) )
         if ( .not. allocated( fym ) ) &
     &      allocate( fym(natom,1) )
         if ( .not. allocated( fzm ) ) &
     &      allocate( fzm(natom,1) )

!        /*   memory allocation:  hessian   */
         if ( .not. allocated( hessian ) ) &
     &      allocate( hessian(3*natom,3*natom,1) )

!        /*   index   */

         if ( .not. allocated( ihess_atom ) ) &
     &      allocate( ihess_atom(6*natom) )
         if ( .not. allocated( ihess_xyz ) ) &
     &      allocate( ihess_xyz(6*natom) )
         if ( .not. allocated( ihess_pm ) ) &
     &      allocate( ihess_pm(6*natom) )

!        /*   set ended   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   make index                                                 */
!-----------------------------------------------------------------------

!     /*   initialize   */
      ihess_atom(:) = 0
      ihess_xyz(:)  = 0
      ihess_pm(:)   = 0

!     /*   counter   */
      l = 0

      do i = 1, natom

!        //   skip frozen atoms
         if ( atom_change(i)(1:7) .eq. 'FREEZE ' ) cycle

         do j = 1, 3
         do k = 1, 2

!           /*   update counter   */
            l = l + 1

!           /*   atom   */
            ihess_atom(l) = i

!           /*   xyz   */
            ihess_xyz(l)  = j

!           /*   plus or minus   */
            ihess_pm(l)   = k

         end do
         end do

      end do

!     /*   number of unfrozen atoms   */
      ihess_natom = l / 6

!-----------------------------------------------------------------------
!     /*   set all beads to original position                         */
!-----------------------------------------------------------------------

      do m = 1, nbead
         x(:,m) = ux(:,1)
         y(:,m) = uy(:,1)
         z(:,m) = uz(:,1)
      end do

!-----------------------------------------------------------------------
!     /*   number of cycles necessary                                 */
!-----------------------------------------------------------------------

      nstep = ( 6*ihess_natom ) / nbead + 1

!-----------------------------------------------------------------------
!     /*   origin                                                     */
!-----------------------------------------------------------------------

      istep = istep_start

      istep_end = istep

      if ( istep .le. nstep ) then

         istep = 0

         call getforce_MPI

         hamiltonian = potential

         call standard_output_nma_MPI

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   main loop: start   */
      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        /*   prepare shifted position                                */
!-----------------------------------------------------------------------

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   l is an index that runs from 0 to 6*ihess_natom   */
            l = (istep-1)*nbead + ibead - 1

!           /*   l: if l is zero then set to the origin   */
            if ( l .eq. 0 ) then

!              /*   set all atoms to original position   */
               x(:,ibead) = ux(:,1)
               y(:,ibead) = uy(:,1)
               z(:,ibead) = uz(:,1)

!           /*   l: if l is not zero   */
            else if ( l .le. 6*ihess_natom ) then

!              /*   atom   */
               i = ihess_atom(l)

!              /*   x, y or z   */
               j = ihess_xyz(l)

!              /*   plus or minus   */
               k = ihess_pm(l)

!              /*   finite difference shift   */
               dr = dble(2*k-3) * fdiff

!              /*   set an atom to shifted position   */
               if ( j .eq. 1 ) x(i,ibead) = ux(i,1) + dr
               if ( j .eq. 2 ) y(i,ibead) = uy(i,1) + dr
               if ( j .eq. 3 ) z(i,ibead) = uz(i,1) + dr

!           /*   l: if l exceeds 6*ihess_natom   */
            else

!              /*   set all atoms to original position   */
               x(:,ibead) = ux(:,1)
               y(:,ibead) = uy(:,1)
               z(:,ibead) = uz(:,1)

!           /*   l: endif   */
            end if

!        /*   loop of beads: end   */
         end do

!-----------------------------------------------------------------------
!        /*   calculate forces                                        */
!-----------------------------------------------------------------------

         call getforce_MPI

!-----------------------------------------------------------------------
!        /*   standard output                                         */
!-----------------------------------------------------------------------

         call standard_output_nma_MPI

!-----------------------------------------------------------------------
!        /*   print forces in shifted positions                       */
!-----------------------------------------------------------------------

!        /*   master rank   */
         if ( myrank .eq. 0 ) then

!        /*   open file   */
         open( iounit, file = 'fd.out', access = 'append' )

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   l is an index that runs from 0 to 6*ihess_natom   */
            l = (istep-1)*nbead + ibead - 1

!           /*   write potential   */
            if ( l .eq. 0 ) write( iounit, '(e24.16)' ) potential

!           /*   l: if l exceeds 6*ihess_natom, skip   */
            if ( l .gt. 6*ihess_natom ) exit

!           /*   print forces   */
            do i = 1, natom
               fxn = fx(i,ibead) * dble(nbead)
               fyn = fy(i,ibead) * dble(nbead)
               fzn = fz(i,ibead) * dble(nbead)
               write( iounit, '(3e24.16)' ) fxn, fyn, fzn
            end do

!        /*   loop of beads: end   */
         end do

!        /*   close file   */
         close( iounit )

!        /*   master rank   */
         end if

!-----------------------------------------------------------------------
!        /*   move back to original position                          */
!-----------------------------------------------------------------------

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   reset   */
            x(:,ibead) = ux(:,1)
            y(:,ibead) = uy(:,1)
            z(:,ibead) = uz(:,1)

!        /*   loop of beads: end   */
         end do

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         istep_end = istep

         call backup_nma_MPI

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) then
            call my_mpi_finalize_2
            stop
         end if

!     /*   main loop: end   */
      end do

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   print one line  */
      if ( myrank .eq. 0 ) write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   once ended, update one more step and save                  */
!-----------------------------------------------------------------------

      call backup_nma_MPI

!-----------------------------------------------------------------------
!     /*   wait for all processors to end                             */
!-----------------------------------------------------------------------

      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   make hessian matrix                                        */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   initialize hessian   */
      hessian(:,:,1) = 0.d0

!     /*   open file   */
      open( iounit, file = 'fd.out' )

!     /*   read force at original position   */
      read ( iounit, * ) potential

!     /*   read force at original position   */
      do j = 1, natom
         read ( iounit, * ) fux(j,1), fuy(j,1), fuz(j,1)
      end do

!     /*   loop of atoms   */
      do i = 1, natom

!        //   skip frozen atoms
         if ( atom_change(i)(1:7) .eq. 'FREEZE ' ) cycle

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 1
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 2
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 3
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!     /*   loop of atoms   */
      end do

!     /*   close file   */
      close( iounit )

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_real_3( hessian, 3*natom, 3*natom, 1 )

!     /*   communicate   */
      call my_mpi_bcast_real_2( fux, 3*natom, 1 )
      call my_mpi_bcast_real_2( fuy, 3*natom, 1 )
      call my_mpi_bcast_real_2( fuz, 3*natom, 1 )

!     /*   force at original position   */
      fx(:,1) = fux(:,1)
      fy(:,1) = fuy(:,1)
      fz(:,1) = fuz(:,1)

      return
      end





!***********************************************************************
      subroutine standard_output_nma_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only :  &
     &   hamiltonian, potential, char_date, iprint_std, &
     &   iounit_std, istep, iounit, nstep, myrank

      implicit none

      integer :: itest
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         if ( myrank .ne. 0 ) return

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step   final  potential [au]     origin [au]    ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step   final  potential [au]     origin [au]    ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step   final  potential [au]  reference [au]    ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   only master rank                                           */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   open file   */
         open( iounit_std, file = 'standard.out', access = 'append' )

!        /*   output to file   */
         write( iounit_std, '(2i8,2f16.8,4x,a28)' ) &
     &      istep, nstep, potential, hamiltonian, char_date

!        /*   close file   */
         close( iounit_std )

!        /*   output   */
         write( 6, '(2i8,2f16.8,4x,a28)' ) &
     &      istep, nstep, potential, hamiltonian, char_date

!        /*   make sure the output   */
         flush( 6 )

      end if

      return
      end
