!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     hessian calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine gethess
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      call gethess_fdiff

!-----------------------------------------------------------------------
!     /*   write                                                      */
!-----------------------------------------------------------------------

      call restart_hess( 2 )

      return
      end





!***********************************************************************
      subroutine restart_hess( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : hessian, nbead, natom, iounit

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, idummy, ioption

!-----------------------------------------------------------------------
!     /*   read                                                       */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         open ( iounit, file = 'hessian.out' )

         do k = 1, nbead
         do i = 1, 3*natom
         do j = 1, 3*natom
            read( iounit, * ) idummy, idummy, hessian(i,j,k)
         end do
         end do
         end do

         close( iounit )

      end if

!-----------------------------------------------------------------------
!     /*   write hessian                                              */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         open ( iounit, file = 'hessian.out' )

         do k = 1, nbead
         do i = 1, 3*natom
         do j = 1, 3*natom
            write( iounit, '(2i6,e24.16)' ) i, j, hessian(i,j,k)
         end do
         end do
         end do

         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine gethess_fdiff
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, fxp, fyp, fzp, fxm, fym, fzm, &
     &   fdiff, hessian, natom, nbead, iounit, atom_change

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   finite difference parameter   */
         call read_real1 ( fdiff, '<fdiff>', 7, iounit )

!        /*   memory allocation:  forces   */

         if ( .not. allocated( fxp ) ) &
     &      allocate( fxp(natom,nbead) )
         if ( .not. allocated( fyp ) ) &
     &      allocate( fyp(natom,nbead) )
         if ( .not. allocated( fzp ) ) &
     &      allocate( fzp(natom,nbead) )
         if ( .not. allocated( fxm ) ) &
     &      allocate( fxm(natom,nbead) )
         if ( .not. allocated( fym ) ) &
     &      allocate( fym(natom,nbead) )
         if ( .not. allocated( fzm ) ) &
     &      allocate( fzm(natom,nbead) )

!        /*   memory allocation:  hessian   */
         if ( .not. allocated( hessian ) ) &
     &      allocate( hessian(3*natom,3*natom,nbead) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   make hessian matrix                                        */
!-----------------------------------------------------------------------

      write( 6, '(a)' ) 'Hessian calculation for each atom.'
      write( 6, '(a)' ) 

      hessian(:,:,:) = 0.d0

      do i = 1, natom

!        //   skip frozen atoms
         if ( atom_change(i)(1:7) .eq. 'FREEZE ' ) cycle

         x(i,:) = x(i,:) - fdiff

         call getforce

         fxm(:,:) = fx(:,:)
         fym(:,:) = fy(:,:)
         fzm(:,:) = fz(:,:)

         x(i,:) = x(i,:) + 2.d0*fdiff

         call getforce

         fxp(:,:) = fx(:,:)
         fyp(:,:) = fy(:,:)
         fzp(:,:) = fz(:,:)

         x(i,:) = x(i,:) - fdiff

         do m = 1, nbead
         do j = 1, natom
            k = 3*(i-1) + 1
            l = 3*(j-1) + 1
            hessian(k,l,m)  =  - ( fxp(j,m) - fxm(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 2
            hessian(k,l,m)  =  - ( fyp(j,m) - fym(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 3
            hessian(k,l,m)  =  - ( fzp(j,m) - fzm(j,m) ) /(2.d0*fdiff)
         end do
         end do

         y(i,:) = y(i,:) - fdiff

         call getforce

         fxm(:,:) = fx(:,:)
         fym(:,:) = fy(:,:)
         fzm(:,:) = fz(:,:)

         y(i,:) = y(i,:) + 2.d0*fdiff

         call getforce

         fxp(:,:) = fx(:,:)
         fyp(:,:) = fy(:,:)
         fzp(:,:) = fz(:,:)

         y(i,:) = y(i,:) - fdiff

         do m = 1, nbead
         do j = 1, natom
            k = 3*(i-1) + 2
            l = 3*(j-1) + 1
            hessian(k,l,m)  =  - ( fxp(j,m) - fxm(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 2
            hessian(k,l,m)  =  - ( fyp(j,m) - fym(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 3
            hessian(k,l,m)  =  - ( fzp(j,m) - fzm(j,m) ) /(2.d0*fdiff)
         end do
         end do

         z(i,:) = z(i,:) - fdiff

         call getforce

         fxm(:,:) = fx(:,:)
         fym(:,:) = fy(:,:)
         fzm(:,:) = fz(:,:)

         z(i,:) = z(i,:) + 2.d0*fdiff

         call getforce

         fxp(:,:) = fx(:,:)
         fyp(:,:) = fy(:,:)
         fzp(:,:) = fz(:,:)

         z(i,:) = z(i,:) - fdiff

         do m = 1, nbead
         do j = 1, natom
            k = 3*(i-1) + 3
            l = 3*(j-1) + 1
            hessian(k,l,m)  =  - ( fxp(j,m) - fxm(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 2
            hessian(k,l,m)  =  - ( fyp(j,m) - fym(j,m) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 3
            hessian(k,l,m)  =  - ( fzp(j,m) - fzm(j,m) ) /(2.d0*fdiff)
         end do
         end do

         if ( mod(i,12) .eq. 0 ) then
            write( 6, '(i6)', advance='yes' ) i
         else
            write( 6, '(i6)', advance='no' ) i
         end if

      end do

      write( 6, '(a)' )
      if ( mod(natom,12) .ne. 0 ) then
         write( 6, '(a)' )
      end if

      write( 6, '(a)' ) 'Hessian calculation done.'
      write( 6, '(a)' )

      call getforce

      return
      end





!***********************************************************************
      subroutine gethess_paral
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, fxp, fyp, fzp, fxm, fym, fzm, fdiff, &
     &   hessian, ux, uy, uz, fux, fuy, fuz, potential, hamiltonian, &
     &   natom, nbead, istep, istep_start, istep_end, nstep, &
     &   iounit, iexit, atom_change

      use hess_variables, only : &
     &   ihess_atom, ihess_xyz, ihess_pm, ihess_natom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, ibead, itest

      real(8) :: dr, fxn, fyn, fzn

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   check existence of file   */
         call testfile ( 'fd.out', 6, itest, iounit )

!        /*   reset step if file do not exist   */
         if ( itest .eq. 1 ) then
            call system('rm -f step.ini')
            istep_start = 0
         end if

!        /*   finite difference parameter   */
         call read_real1 ( fdiff, '<fdiff>', 7, iounit )

!        /*   memory allocation:  forces   */

         if ( .not. allocated( fxp ) ) &
     &      allocate( fxp(natom,1) )
         if ( .not. allocated( fyp ) ) &
     &      allocate( fyp(natom,1) )
         if ( .not. allocated( fzp ) ) &
     &      allocate( fzp(natom,1) )
         if ( .not. allocated( fxm ) ) &
     &      allocate( fxm(natom,1) )
         if ( .not. allocated( fym ) ) &
     &      allocate( fym(natom,1) )
         if ( .not. allocated( fzm ) ) &
     &      allocate( fzm(natom,1) )

!        /*   memory allocation:  hessian   */
         if ( .not. allocated( hessian ) ) &
     &      allocate( hessian(3*natom,3*natom,1) )

!        /*   index   */

         if ( .not. allocated( ihess_atom ) ) &
     &      allocate( ihess_atom(6*natom) )
         if ( .not. allocated( ihess_xyz ) ) &
     &      allocate( ihess_xyz(6*natom) )
         if ( .not. allocated( ihess_pm ) ) &
     &      allocate( ihess_pm(6*natom) )

!        /*   set ended   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   make index                                                 */
!-----------------------------------------------------------------------

!     /*   initialize   */
      ihess_atom(:) = 0
      ihess_xyz(:)  = 0
      ihess_pm(:)   = 0

!     /*   counter   */
      l = 0

      do i = 1, natom

!        //   skip frozen atoms
         if ( atom_change(i)(1:7) .eq. 'FREEZE ' ) cycle

         do j = 1, 3
         do k = 1, 2

!           /*   update counter   */
            l = l + 1

!           /*   atom   */
            ihess_atom(l) = i

!           /*   xyz   */
            ihess_xyz(l)  = j

!           /*   plus or minus   */
            ihess_pm(l)   = k

         end do
         end do

      end do

!     /*   number of unfrozen atoms   */
      ihess_natom = l / 6

!-----------------------------------------------------------------------
!     /*   set all beads to original position                         */
!-----------------------------------------------------------------------

      do m = 1, nbead
         x(:,m) = ux(:,1)
         y(:,m) = uy(:,1)
         z(:,m) = uz(:,1)
      end do

!-----------------------------------------------------------------------
!     /*   number of cycles necessary                                 */
!-----------------------------------------------------------------------

      nstep = ( 6*ihess_natom ) / nbead + 1

!-----------------------------------------------------------------------
!     /*   origin                                                     */
!-----------------------------------------------------------------------

      istep = istep_start

      istep_end = istep

      if ( istep .le. nstep ) then

         istep = 0

         call getforce

         hamiltonian = potential

         call standard_output_nma

      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   main loop: start   */
      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        /*   prepare shifted position                                */
!-----------------------------------------------------------------------

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   l is an index that runs from 0 to 6*ihess_natom   */
            l = (istep-1)*nbead + ibead - 1

!           /*   l: if l is zero then set to the origin   */
            if ( l .eq. 0 ) then

!              /*   set all atoms to original position   */
               x(:,ibead) = ux(:,1)
               y(:,ibead) = uy(:,1)
               z(:,ibead) = uz(:,1)

!           /*   l: if l is not zero   */
            else if ( l .le. 6*ihess_natom ) then

!              /*   atom   */
               i = ihess_atom(l)

!              /*   x, y or z   */
               j = ihess_xyz(l)

!              /*   plus or minus   */
               k = ihess_pm(l)

!              /*   finite difference shift   */
               dr = dble(2*k-3) * fdiff

!              /*   set an atom to shifted position   */
               if ( j .eq. 1 ) x(i,ibead) = ux(i,1) + dr
               if ( j .eq. 2 ) y(i,ibead) = uy(i,1) + dr
               if ( j .eq. 3 ) z(i,ibead) = uz(i,1) + dr

!           /*   l: if l exceeds 6*ihess_natom   */
            else

!              /*   set all atoms to original position   */
               x(:,ibead) = ux(:,1)
               y(:,ibead) = uy(:,1)
               z(:,ibead) = uz(:,1)

!           /*   l: endif   */
            end if

!        /*   loop of beads: end   */
         end do

!-----------------------------------------------------------------------
!        /*   calculate forces                                        */
!-----------------------------------------------------------------------

         call getforce

!-----------------------------------------------------------------------
!        /*   standard output                                         */
!-----------------------------------------------------------------------

         call standard_output_nma

!-----------------------------------------------------------------------
!        /*   print forces in shifted positions                       */
!-----------------------------------------------------------------------

!        /*   open file   */
         open( iounit, file = 'fd.out', access = 'append' )

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   l is an index that runs from 0 to 6*ihess_natom   */
            l = (istep-1)*nbead + ibead - 1

!           /*   write potential   */
            if ( l .eq. 0 ) write( iounit, '(e24.16)' ) potential

!           /*   l: if l exceeds 6*ihess_natom, skip   */
            if ( l .gt. 6*ihess_natom ) exit

!           /*   print forces   */
            do i = 1, natom
               fxn = fx(i,ibead) * dble(nbead)
               fyn = fy(i,ibead) * dble(nbead)
               fzn = fz(i,ibead) * dble(nbead)
               write( iounit, '(3e24.16)' ) fxn, fyn, fzn
            end do

!        /*   loop of beads: end   */
         end do

!        /*   close file   */
         close( iounit )

!-----------------------------------------------------------------------
!        /*   move back to original position                          */
!-----------------------------------------------------------------------

!        /*   loop of beads: start   */
         do ibead = 1, nbead

!           /*   reset   */
            x(:,ibead) = ux(:,1)
            y(:,ibead) = uy(:,1)
            z(:,ibead) = uz(:,1)

!        /*   loop of beads: end   */
         end do

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         istep_end = istep

         call backup_nma

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit

         if ( iexit .eq. 1 ) stop

!     /*   main loop: end   */
      end do

!-----------------------------------------------------------------------
!     /*   standard output                                            */
!-----------------------------------------------------------------------

!     /*   print one line  */
      write ( 6, '(a)' )

!-----------------------------------------------------------------------
!     /*   once ended, update one more step and save                  */
!-----------------------------------------------------------------------

      call backup_nma

!-----------------------------------------------------------------------
!     /*   make hessian matrix                                        */
!-----------------------------------------------------------------------

!     /*   initialize hessian   */
      hessian(:,:,1) = 0.d0

!     /*   open file   */
      open( iounit, file = 'fd.out' )

!     /*   read force at original position   */
      read ( iounit, * ) potential

!     /*   read force at original position   */

      do j = 1, natom
         read ( iounit, * ) fux(j,1), fuy(j,1), fuz(j,1)
      end do

!     /*   loop of atoms   */
      do i = 1, natom

!        //   skip frozen atoms
         if ( atom_change(i)(1:7) .eq. 'FREEZE ' ) cycle

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 1
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 1
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 2
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 2
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!        /*   read force at minus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxm(j,1), fym(j,1), fzm(j,1)
         end do

!        /*   read force at plus shifted position   */
         do j = 1, natom
            read ( iounit, * ) fxp(j,1), fyp(j,1), fzp(j,1)
         end do

!        /*   hessian as finite difference   */

         do j = 1, natom
            k = 3*(i-1) + 3
            l = 3*(j-1) + 1
            hessian(k,l,1)  =  - ( fxp(j,1) - fxm(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 2
            hessian(k,l,1)  =  - ( fyp(j,1) - fym(j,1) ) /(2.d0*fdiff)
            k = 3*(i-1) + 3
            l = 3*(j-1) + 3
            hessian(k,l,1)  =  - ( fzp(j,1) - fzm(j,1) ) /(2.d0*fdiff)
         end do

!     /*   loop of atoms   */
      end do

!     /*   close file   */
      close( iounit )

!     /*   force at original position   */
      fx(:,1) = fux(:,1)
      fy(:,1) = fuy(:,1)
      fz(:,1) = fuz(:,1)

      return
      end





!***********************************************************************
      subroutine standard_output_nma
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only :  &
     &   hamiltonian, potential, char_date, iprint_std, &
     &   iounit_std, istep, iounit, nstep

      implicit none

      integer :: itest
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         if ( iprint_std .le. 0 ) return

         call testfile ( 'standard.out', 12, itest, iounit )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out' )

            write( iounit_std, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( iounit_std, '(a)' ) &
     &      '    step   final  potential [au]     origin [au]    ' // &
     &      'wall clock time           '
            write( iounit_std, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            close( iounit_std )

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step   final  potential [au]     origin [au]    ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         else

            write( 6, '(a)' ) &
     &      '====================================================' // &
     &      '=========================='
            write( 6, '(a)' ) &
     &      '    step   final  potential [au]  reference [au]    ' // &
     &      'wall clock time           '
            write( 6, '(a)' ) &
     &      '----------------------------------------------------' // &
     &      '--------------------------'

            flush( 6 )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   print output                                               */
!-----------------------------------------------------------------------

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   open file   */
         open( iounit_std, file = 'standard.out', access = 'append' )

!        /*   output to file   */
         write( iounit_std, '(2i8,2f16.8,4x,a28)' ) &
     &      istep, nstep, potential, hamiltonian, char_date

!        /*   close file   */
         close( iounit_std )

!        /*   output   */
         write( 6, '(2i8,2f16.8,4x,a28)' ) &
     &      istep, nstep, potential, hamiltonian, char_date

!        /*   make sure the output   */
         flush( 6 )

      end if

      return
      end





!***********************************************************************
      subroutine backup_nma
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep_end, nstep, iexit, iprint_rest, iounit

      use lbfgs_variables, only : &
     &   iflag

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( istep_end .ge. nstep ) then
         continue
      else if ( iexit .eq. 1 ) then
         continue
      else if ( iflag .ne. 0 ) then
         continue
      else
         if ( iprint_rest .le. 0 ) then
            return
         else
            if ( mod(istep_end,iprint_rest) .eq. 0 ) then
               continue
            else
               return
            end if
         end if
      end if

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
      close( iounit )

!-----------------------------------------------------------------------
!     /*   print final geometry                                       */
!-----------------------------------------------------------------------

      call print_final_xyz

      return
      end

