!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     harmonic forces for constraints
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_ref_cons
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fux_ref, fuy_ref, fuz_ref

      use cons_variables, only : &
     &   pot_ref_cons, fref_cons, scons, ncons, itype_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: k

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   cv   */
      scons(:,:)  = 0.d0

!     /*   harmonic potential   */
      pot_ref_cons(:) = 0.d0

!     /*   harmonic force for constraints   */
      fref_cons(:,:)  = 0.d0

!     /*   harmonic force for atoms   */
      fux_ref(:,:)    = 0.d0
      fuy_ref(:,:)    = 0.d0
      fuz_ref(:,:)    = 0.d0

!-----------------------------------------------------------------------
!     /*   harmonic potential part                                    */
!-----------------------------------------------------------------------

      do k = 1, ncons

!        /*   linear bonding   */
         if ( itype_cons(k) .eq. 1 )  call lin_ref_cons( k )

!        /*   angular bonding   */
         if ( itype_cons(k) .eq. 2 )  call angl_ref_cons( k )

!        /*   dihedral bonding   */
         if ( itype_cons(k) .eq. 3 )  call dih_ref_cons( k )

!        /*   bond difference   */
         if ( itype_cons(k) .eq. 4 )  call diff_ref_cons( k )

!        /*   coordination number   */
         if ( itype_cons(k) .eq. 5 )  call cord_ref_cons( k )

!        /*   difference in coordination number   */
         if ( itype_cons(k) .eq. 6 )  call dcord_ref_cons( k )

!        /*   center of mass   */
         if ( itype_cons(k) .eq. 7 )  call xyz_ref_cons( k )

!        /*   difference in center of mass   */
         if ( itype_cons(k) .eq. 8 )  call dxyz_ref_cons( k )

      end do

      return
      end





!***********************************************************************
      subroutine lin_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fux_ref, fuy_ref, fuz_ref, nbead

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, i_cons, j_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, m, n

      real(8) :: xij, yij, zij, rij, dr, &
     &           drdxi, drdyi, drdzi, drdxj, drdyj, drdzj

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      i = i_cons(m)
      j = j_cons(m)

      do n = 1, nbead

         xij = x(i,n) - x(j,n)
         yij = y(i,n) - y(j,n)
         zij = z(i,n) - z(j,n)

         call pbc_atom ( xij, yij, zij )

         rij = sqrt( xij*xij + yij*yij + zij*zij )

         drdxi = + xij/rij
         drdyi = + yij/rij
         drdzi = + zij/rij

         drdxj = - xij/rij
         drdyj = - yij/rij
         drdzj = - zij/rij

         scons(m,n) = rij

         dr = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*dr*dr

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m)*dr

         fux_ref(i,n) = fux_ref(i,n) - fc_cons(m)*dr*drdxi
         fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m)*dr*drdyi
         fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m)*dr*drdzi

         fux_ref(j,n) = fux_ref(j,n) - fc_cons(m)*dr*drdxj
         fuy_ref(j,n) = fuy_ref(j,n) - fc_cons(m)*dr*drdyj
         fuz_ref(j,n) = fuz_ref(j,n) - fc_cons(m)*dr*drdzj

      end do

      return
      end





!***********************************************************************
      subroutine angl_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, fux_ref, fuy_ref, fuz_ref, nbead

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, &
     &   i_cons, j_cons, k_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, n

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij2, rkj2, rijk, pijk, &
     &           qijk, bijk, aijk, const, dadxi, dadxj, dadxk, &
     &           dadyi, dadyj, dadyk, dadzi, dadzj, dadzk, da

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      i = i_cons(m)
      j = j_cons(m)
      k = k_cons(m)

      do n = 1, nbead

         xij = x(i,n) - x(j,n)
         yij = y(i,n) - y(j,n)
         zij = z(i,n) - z(j,n)

         call pbc_atom ( xij, yij, zij )

         xkj = x(k,n) - x(j,n)
         ykj = y(k,n) - y(j,n)
         zkj = z(k,n) - z(j,n)

         call pbc_atom ( xkj, ykj, zkj )

         rij2 = xij*xij + yij*yij + zij*zij
         rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

         rijk = sqrt( rij2*rkj2 )

         pijk = xij*xkj + yij*ykj + zij*zkj

         qijk  = pijk/rijk

         qijk = max( qijk, -1.d0 )
         qijk = min( qijk,  1.d0 )

         bijk = acos( qijk )

         aijk = bijk*(180.d0/pi)

         aijk = aijk - 360.d0 * nint( aijk / 360.d0 )

         if ( abs(bijk)    .lt. tiny_value ) cycle
         if ( abs(bijk-pi) .lt. tiny_value ) cycle
         if ( abs(bijk+pi) .lt. tiny_value ) cycle

         const = - 1.d0 /sin(bijk) /rijk *(180.d0/pi)

         dadxi = const*( xkj - pijk/rij2*xij )
         dadxk = const*( xij - pijk/rkj2*xkj )
         dadxj = - dadxi - dadxk

         dadyi = const*( ykj - pijk/rij2*yij )
         dadyk = const*( yij - pijk/rkj2*ykj )
         dadyj = - dadyi - dadyk

         dadzi = const*( zkj - pijk/rij2*zij )
         dadzk = const*( zij - pijk/rkj2*zkj )
         dadzj = - dadzi - dadzk

         scons(m,n) = aijk

         da = scons(m,n) - rcons(m,n)

         da = da - 360.d0 * nint( da /360.d0 )

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*da*da

         const = fc_cons(m)*da

         fref_cons(m,n) = fref_cons(m,n) + const

         fux_ref(i,n) = fux_ref(i,n) - const*dadxi
         fux_ref(j,n) = fux_ref(j,n) - const*dadxj
         fux_ref(k,n) = fux_ref(k,n) - const*dadxk

         fuy_ref(i,n) = fuy_ref(i,n) - const*dadyi
         fuy_ref(j,n) = fuy_ref(j,n) - const*dadyj
         fuy_ref(k,n) = fuy_ref(k,n) - const*dadyk

         fuz_ref(i,n) = fuz_ref(i,n) - const*dadzi
         fuz_ref(j,n) = fuz_ref(j,n) - const*dadzj
         fuz_ref(k,n) = fuz_ref(k,n) - const*dadzk

      end do

      return
      end





!***********************************************************************
      subroutine dih_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, fux_ref, fuy_ref, fuz_ref, nbead

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, &
     &   i_cons, j_cons, k_cons, l_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: tiny_value = 1.d-4

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           f1, psi, dpsi, const, sign_phi, sin_phi

      real(8) :: dpsidxi, dpsidxj, dpsidxk, dpsidxl, &
     &           dpsidyi, dpsidyj, dpsidyk, dpsidyl, &
     &           dpsidzi, dpsidzj, dpsidzk, dpsidzl

      real(8) :: ax, ay, az, a1, a2, xkl, ykl, zkl, xki, yki, zki

      real(8) :: daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl

!      real(8)::  daxdxk, daxdyk, daxdzk, daydxk, daydyk, daydzk,
!     &           dazdxk, dazdyk, dazdzk, dadxk, dadyk, dadzk

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      i = i_cons(m)
      j = j_cons(m)
      k = k_cons(m)
      l = l_cons(m)

!-----------------------------------------------------------------------

      do n = 1, nbead

         xij = x(i,n) - x(j,n)
         yij = y(i,n) - y(j,n)
         zij = z(i,n) - z(j,n)

         call pbc_atom ( xij, yij, zij )

         xkj = x(k,n) - x(j,n)
         ykj = y(k,n) - y(j,n)
         zkj = z(k,n) - z(j,n)

         call pbc_atom ( xkj, ykj, zkj )

         xlj = x(l,n) - x(j,n)
         ylj = y(l,n) - y(j,n)
         zlj = z(l,n) - z(j,n)

         call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

         xijk = yij*zkj - zij*ykj
         yijk = zij*xkj - xij*zkj
         zijk = xij*ykj - yij*xkj

         xjkl = ylj*zkj - zlj*ykj
         yjkl = zlj*xkj - xlj*zkj
         zjkl = xlj*ykj - ylj*xkj

         rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
         rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

         rijkl2 = sqrt(rijk2*rjkl2)

         rijk2inv  = 1.d0 / rijk2
         rjkl2inv  = 1.d0 / rjkl2
         rijkl2inv = 1.d0 / rijkl2

         cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

         cos_phi = max( cos_phi, -1.d0 )
         cos_phi = min( cos_phi,  1.d0 )

         phi = acos( cos_phi )

         sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &            + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &            + ( xijk*yjkl - yijk*xjkl ) * zkj

         sign_phi = sign( 1.d0, sign_phi )

         phi = phi * sign_phi

         psi = phi * (180.d0/pi)

         psi = psi - 360.d0 * nint( psi / 360.d0 )

         scons(m,n) = psi

         if ( ( abs(phi)    .gt. tiny_value ) .and. &
     &        ( abs(phi-pi) .gt. tiny_value ) .and. &
     &        ( abs(phi+pi) .gt. tiny_value ) ) then

            f1 = - 1.d0/sin(phi) * (180.d0/pi)

            dpsidxi = f1 * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
            dpsidyi = f1 * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
            dpsidzi = f1 * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

            dpsidxl = f1 * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
            dpsidyl = f1 * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
            dpsidzl = f1 * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

            dpsidxk = f1 * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
            dpsidyk = f1 * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
            dpsidzk = f1 * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

            dpsidxj = - ( dpsidxi + dpsidxk + dpsidxl )
            dpsidyj = - ( dpsidyi + dpsidyk + dpsidyl )
            dpsidzj = - ( dpsidzi + dpsidzk + dpsidzl )

         else

            xki = - xij + xkj
            yki = - yij + ykj
            zki = - zij + zkj

            xkl = - xlj + xkj
            ykl = - ylj + ykj
            zkl = - zlj + zkj

            ax = yijk*zjkl - zijk*yjkl
            ay = zijk*xjkl - xijk*zjkl
            az = xijk*yjkl - yijk*xjkl

            a2 = ax*ax + ay*ay + az*az

            a1 = sqrt( a2 )

            sin_phi = a1 / rijkl2

            sin_phi = max( sin_phi, -1.d0 )
            sin_phi = min( sin_phi,  1.d0 )

            phi = sign_phi * asin( sin_phi )

            if ( cos_phi .lt. 0.d0 ) phi = pi - phi

            psi = phi * 180.d0 / pi

            psi = psi - 360.d0 * nint( psi / 360.d0 )

            daxdxi = - zjkl * zkj - ykj * yjkl
            daxdyi = + yjkl * xkj
            daxdzi = + zjkl * xkj

            daydxi = + xjkl * ykj
            daydyi = - xjkl * xkj - zkj * zjkl
            daydzi = + zjkl * ykj

            dazdxi = + xjkl * zkj
            dazdyi = + yjkl * zkj
            dazdzi = - yjkl * ykj - xkj * xjkl

            daxdxj = - yijk * ykl + zjkl * zki + yjkl * yki - zijk * zkl
            daxdyj = + yijk * xkl - yjkl * xki
            daxdzj = + zijk * xkl - zjkl * xki

            daydxj = + xijk * ykl - xjkl * yki
            daydyj = - zijk * zkl + xjkl * xki + zjkl * zki - xijk * xkl
            daydzj = + zijk * ykl - zjkl * yki

            dazdxj = + xijk * zkl - xjkl * zki
            dazdyj = + yijk * zkl - yjkl * zki
            dazdzj = - xijk * xkl + yjkl * yki + xjkl * xki - yijk * ykl

!            daxdxk = - yjkl * yij + zijk * zlj + yijk * ylj - zjkl * zij
!            daxdyk = + yjkl * xij - yijk * xlj
!            daxdzk = + zjkl * xij - zijk * xlj
!
!            daydxk = + xjkl * yij - xijk * ylj
!            daydyk = - zjkl * zij + xijk * xlj + zijk * zlj - xjkl * xij
!            daydzk = + zjkl * yij - zijk * ylj
!
!            dazdxk = + xjkl * zij - xijk * zlj
!            dazdyk = + yjkl * zij - yijk * zlj
!            dazdzk = - xjkl * xij + yijk * ylj + xijk * xlj - yjkl * yij

            daxdxl = + zijk * zkj + ykj * yijk
            daxdyl = - yijk * xkj
            daxdzl = - zijk * xkj

            daydxl = - xijk * ykj
            daydyl = + xijk * xkj + zkj * zijk
            daydzl = - zijk * ykj

            dazdxl = - xijk * zkj
            dazdyl = - yijk * zkj
            dazdzl = + yijk * ykj + xkj * xijk

            dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
            dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
            dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

            dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
            dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
            dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!            dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!            dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!            dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

            dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
            dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
            dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

            f1 = sign_phi/cos_phi * (180.d0/pi)

            dpsidxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

            dpsidyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

            dpsidzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

            dpsidxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

            dpsidyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

            dpsidzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!            dpsidxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!            dpsidyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!            dpsidzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

            dpsidxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

            dpsidyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

            dpsidzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

            dpsidxk = - dpsidxi - dpsidxj - dpsidxl
            dpsidyk = - dpsidyi - dpsidyj - dpsidyl
            dpsidzk = - dpsidzi - dpsidzj - dpsidzl

         end if

         dpsi = scons(m,n) - rcons(m,n)

         dpsi = dpsi - 360.d0 * nint( dpsi / 360.d0 )

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*dpsi*dpsi

         const = fc_cons(m)*dpsi

         fref_cons(m,n) = fref_cons(m,n) + const

         fux_ref(i,n) = fux_ref(i,n) - const*dpsidxi
         fux_ref(j,n) = fux_ref(j,n) - const*dpsidxj
         fux_ref(k,n) = fux_ref(k,n) - const*dpsidxk
         fux_ref(l,n) = fux_ref(l,n) - const*dpsidxl

         fuy_ref(i,n) = fuy_ref(i,n) - const*dpsidyi
         fuy_ref(j,n) = fuy_ref(j,n) - const*dpsidyj
         fuy_ref(k,n) = fuy_ref(k,n) - const*dpsidyk
         fuy_ref(l,n) = fuy_ref(l,n) - const*dpsidyl

         fuz_ref(i,n) = fuz_ref(i,n) - const*dpsidzi
         fuz_ref(j,n) = fuz_ref(j,n) - const*dpsidzj
         fuz_ref(k,n) = fuz_ref(k,n) - const*dpsidzk
         fuz_ref(l,n) = fuz_ref(l,n) - const*dpsidzl

      end do

      return
      end





!***********************************************************************
      subroutine diff_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, fux_ref, fuy_ref, fuz_ref, nbead

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, &
     &   i_cons, j_cons, k_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, n

      real(8) :: xij, yij, zij, rij, xkj, ykj, zkj, rkj, dr, &
     &           drdxi, drdyi, drdzi, drdxj, drdyj, drdzj, &
     &           drdxk, drdyk, drdzk

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      i = i_cons(m)
      j = j_cons(m)
      k = k_cons(m)

      do n = 1, nbead

         xij = x(i,n) - x(j,n)
         yij = y(i,n) - y(j,n)
         zij = z(i,n) - z(j,n)

         call pbc_atom ( xij, yij, zij )

         rij = sqrt( xij*xij + yij*yij + zij*zij )

         xkj = x(k,n) - x(j,n)
         ykj = y(k,n) - y(j,n)
         zkj = z(k,n) - z(j,n)

         call pbc_atom ( xkj, ykj, zkj )

         rkj = sqrt( xkj*xkj + ykj*ykj + zkj*zkj )

         drdxi = + xij/rij
         drdyi = + yij/rij
         drdzi = + zij/rij

         drdxj = - xij/rij + xkj/rkj
         drdyj = - yij/rij + ykj/rkj
         drdzj = - zij/rij + zkj/rkj

         drdxk = - xkj/rkj
         drdyk = - ykj/rkj
         drdzk = - zkj/rkj

         scons(m,n) = rij - rkj

         dr = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*dr*dr

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m)*dr

         fux_ref(i,n) = fux_ref(i,n) - fc_cons(m)*dr*drdxi
         fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m)*dr*drdyi
         fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m)*dr*drdzi

         fux_ref(j,n) = fux_ref(j,n) - fc_cons(m)*dr*drdxj
         fuy_ref(j,n) = fuy_ref(j,n) - fc_cons(m)*dr*drdyj
         fuz_ref(j,n) = fuz_ref(j,n) - fc_cons(m)*dr*drdzj

         fux_ref(k,n) = fux_ref(k,n) - fc_cons(m)*dr*drdxk
         fuy_ref(k,n) = fuy_ref(k,n) - fc_cons(m)*dr*drdyk
         fuz_ref(k,n) = fuz_ref(k,n) - fc_cons(m)*dr*drdzk

      end do

      return
      end





!***********************************************************************
      subroutine cord_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fux_ref, fuy_ref, fuz_ref, natom, nbead, ikind

      use cons_variables, only : &
     &   req_cons, scons, rcons, pot_ref_cons, fref_cons, fc_cons, &
     &   nu_cons, mu_cons, i_cons, j_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: nu, mu, i, j, m, n

      real(8) :: req, cn, dcn, xij, yij, zij, rij, fa, fb, dfa, dfb, &
     &           drdxi, drdyi, drdzi, drdxj, drdyj, drdzj, &
     &           dcndr, dcndxi, dcndyi, dcndzi, dcndxj, dcndyj, dcndzj

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      nu  = nu_cons(m,1)

      mu  = mu_cons(m,1)

      req = req_cons(m,1)

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do n = 1, nbead

!-----------------------------------------------------------------------
!        /*   calculate coordination number                           */
!-----------------------------------------------------------------------

         cn = 0.d0

         do i = 1, natom-1
         do j = i+1, natom

            if ( ( ( ikind(i) .eq. i_cons(m) ) .and. &
     &             ( ikind(j) .eq. j_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_cons(m) ) .and. &
     &             ( ikind(j) .eq. i_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               cn = cn + ( 1.d0 - (rij/req)**nu ) &
     &                 / ( 1.d0 - (rij/req)**mu )

            end if

         end do
         end do

         scons(m,n) = cn

!-----------------------------------------------------------------------
!        /*   calculate potential and forces of harmonic term         */
!-----------------------------------------------------------------------

         dcn = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*dcn*dcn

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m)*dcn

         do i = 1, natom-1
         do j = i+1, natom

            if ( ( ( ikind(i) .eq. i_cons(m) ) .and. &
     &             ( ikind(j) .eq. j_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_cons(m) ) .and. &
     &             ( ikind(j) .eq. i_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               drdxi = + xij/rij
               drdyi = + yij/rij
               drdzi = + zij/rij

               drdxj = - xij/rij
               drdyj = - yij/rij
               drdzj = - zij/rij

               fa  = 1.d0 - (rij/req)**nu
               fb  = 1.d0 - (rij/req)**mu

               dfa = - nu * (rij/req)**(nu-1) /req
               dfb = - mu * (rij/req)**(mu-1) /req

               dcndr  = ( dfa*fb - fa*dfb ) / ( fb*fb )

               dcndxi = dcndr * drdxi
               dcndyi = dcndr * drdyi
               dcndzi = dcndr * drdzi

               dcndxj = dcndr * drdxj
               dcndyj = dcndr * drdyj
               dcndzj = dcndr * drdzj

               fux_ref(i,n) = fux_ref(i,n) - fc_cons(m)*dcn*dcndxi
               fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m)*dcn*dcndyi
               fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m)*dcn*dcndzi

               fux_ref(j,n) = fux_ref(j,n) - fc_cons(m)*dcn*dcndxj
               fuy_ref(j,n) = fuy_ref(j,n) - fc_cons(m)*dcn*dcndyj
               fuz_ref(j,n) = fuz_ref(j,n) - fc_cons(m)*dcn*dcndzj

            end if

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine dcord_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fux_ref, fuy_ref, fuz_ref, natom, nbead, ikind

      use cons_variables, only : &
     &   req_cons, scons, rcons, pot_ref_cons, fref_cons, fc_cons, &
     &   nu_cons, mu_cons, i_cons, j_cons, k_cons, l_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: nu1, nu2, mu1, mu2, i, j, m, n

      real(8) :: req1, req2, cn1, cn2, &
     &           dcn, xij, yij, zij, rij, fa, fb, dfa, dfb, &
     &           drdxi, drdyi, drdzi, drdxj, drdyj, drdzj, &
     &           dcndr, dcndxi, dcndyi, dcndzi, dcndxj, dcndyj, dcndzj

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      nu1  = nu_cons(m,1)
      nu2  = nu_cons(m,2)
      mu1  = mu_cons(m,1)
      mu2  = mu_cons(m,2)

      req1 = req_cons(m,1)
      req2 = req_cons(m,2)

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do n = 1, nbead

!-----------------------------------------------------------------------
!        /*   calculate coordination number                           */
!-----------------------------------------------------------------------

         cn1 = 0.d0
         cn2 = 0.d0

         do i = 1, natom-1
         do j = i+1, natom

            if ( ( ( ikind(i) .eq. i_cons(m) ) .and. &
     &             ( ikind(j) .eq. j_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_cons(m) ) .and. &
     &             ( ikind(j) .eq. i_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               cn1 = cn1 + ( 1.d0 - (rij/req1)**nu1 ) &
     &                   / ( 1.d0 - (rij/req1)**mu1 )

            end if

            if ( ( ( ikind(i) .eq. k_cons(m) ) .and. &
     &             ( ikind(j) .eq. l_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. l_cons(m) ) .and. &
     &             ( ikind(j) .eq. k_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               cn2 = cn2 + ( 1.d0 - (rij/req2)**nu2 ) &
     &                   / ( 1.d0 - (rij/req2)**mu2 )

            end if

         end do
         end do

         scons(m,n) = cn1 - cn2

!-----------------------------------------------------------------------
!        /*   calculate potential and forces of harmonic term         */
!-----------------------------------------------------------------------

         dcn = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) + 0.5d0*fc_cons(m)*dcn*dcn

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m)*dcn

         do i = 1, natom-1
         do j = i+1, natom

            if ( ( ( ikind(i) .eq. i_cons(m) ) .and. &
     &             ( ikind(j) .eq. j_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_cons(m) ) .and. &
     &             ( ikind(j) .eq. i_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               drdxi = + xij/rij
               drdyi = + yij/rij
               drdzi = + zij/rij

               drdxj = - xij/rij
               drdyj = - yij/rij
               drdzj = - zij/rij

               fa  = 1.d0 - (rij/req1)**nu1
               fb  = 1.d0 - (rij/req1)**mu1

               dfa = - nu1 * (rij/req1)**(nu1-1) /req1
               dfb = - mu1 * (rij/req1)**(mu1-1) /req1

               dcndr  = ( dfa*fb - fa*dfb ) / ( fb*fb )

               dcndxi = dcndr * drdxi
               dcndyi = dcndr * drdyi
               dcndzi = dcndr * drdzi

               dcndxj = dcndr * drdxj
               dcndyj = dcndr * drdyj
               dcndzj = dcndr * drdzj

               fux_ref(i,n) = fux_ref(i,n) - fc_cons(m)*dcn*dcndxi
               fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m)*dcn*dcndyi
               fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m)*dcn*dcndzi

               fux_ref(j,n) = fux_ref(j,n) - fc_cons(m)*dcn*dcndxj
               fuy_ref(j,n) = fuy_ref(j,n) - fc_cons(m)*dcn*dcndyj
               fuz_ref(j,n) = fuz_ref(j,n) - fc_cons(m)*dcn*dcndzj

            end if

            if ( ( ( ikind(i) .eq. k_cons(m) ) .and. &
     &             ( ikind(j) .eq. l_cons(m) ) ) .or. &
     &           ( ( ikind(i) .eq. l_cons(m) ) .and. &
     &             ( ikind(j) .eq. k_cons(m) ) ) ) then

               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

               call pbc_atom ( xij, yij, zij )

               rij = sqrt( xij*xij + yij*yij + zij*zij )

               drdxi = + xij/rij
               drdyi = + yij/rij
               drdzi = + zij/rij

               drdxj = - xij/rij
               drdyj = - yij/rij
               drdzj = - zij/rij

               fa  = 1.d0 - (rij/req2)**nu2
               fb  = 1.d0 - (rij/req2)**mu2

               dfa = - nu1 * (rij/req2)**(nu2-1) /req2
               dfb = - mu1 * (rij/req2)**(mu2-1) /req2

               dcndr  = ( dfa*fb - fa*dfb ) / ( fb*fb )

               dcndxi = dcndr * drdxi
               dcndyi = dcndr * drdyi
               dcndzi = dcndr * drdzi

               dcndxj = dcndr * drdxj
               dcndyj = dcndr * drdyj
               dcndzj = dcndr * drdzj

               fux_ref(i,n) = fux_ref(i,n) + fc_cons(m)*dcn*dcndxi
               fuy_ref(i,n) = fuy_ref(i,n) + fc_cons(m)*dcn*dcndyi
               fuz_ref(i,n) = fuz_ref(i,n) + fc_cons(m)*dcn*dcndzi

               fux_ref(j,n) = fux_ref(j,n) + fc_cons(m)*dcn*dcndxj
               fuy_ref(j,n) = fuy_ref(j,n) + fc_cons(m)*dcn*dcndyj
               fuz_ref(j,n) = fuz_ref(j,n) + fc_cons(m)*dcn*dcndzj

            end if

         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine xyz_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fux_ref, fuy_ref, fuz_ref, natom, nbead, ikind, mbox

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, i_cons, j_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, m, nk, m1, m2, m3, kx, ky, kz

      real(8) :: xk, yk, zk, xi, yi, zi, xn, yn, zn, dxyz

!-----------------------------------------------------------------------
!     /*   prefactor                                                  */
!-----------------------------------------------------------------------

      kx = 0
      ky = 0
      kz = 0

      if ( i_cons(m) .eq. 1 ) kx = 1
      if ( i_cons(m) .eq. 2 ) ky = 1
      if ( i_cons(m) .eq. 3 ) kz = 1

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do n = 1, nbead

!-----------------------------------------------------------------------
!        /*   calculate center of mass                               */
!-----------------------------------------------------------------------

         nk = 0

         xk = 0.d0
         yk = 0.d0
         zk = 0.d0

         scons(m,n) = 0.d0

         do i = 1, natom

            if ( ikind(i) .eq. j_cons(m) ) then

               xi = x(i,n)
               yi = y(i,n)
               zi = z(i,n)

               m1 = mbox(1,i,n)
               m2 = mbox(2,i,n)
               m3 = mbox(3,i,n)

               call pbc_unfold( xi, yi, zi, m1, m2, m3 )

               nk = nk + 1

               xk = xk + xi
               yk = yk + yi
               zk = zk + zi

            end if

         end do

         xk = xk / dble(nk)
         yk = yk / dble(nk)
         zk = zk / dble(nk)

         scons(m,n) = dble(kx) * xk + dble(ky) * yk + dble(kz) * zk

!-----------------------------------------------------------------------
!        /*   calculate potential and forces of harmonic term         */
!-----------------------------------------------------------------------

         dxyz = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) &
     &                   + 0.5d0 * fc_cons(m) * dxyz * dxyz

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m) * dxyz

         xn = dble(kx) / dble(nk)
         yn = dble(ky) / dble(nk)
         zn = dble(kz) / dble(nk)

         do i = 1, natom

            if ( ikind(i) .eq. j_cons(m) ) then

               fux_ref(i,n) = fux_ref(i,n) - fc_cons(m) * xn * dxyz
               fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m) * yn * dxyz
               fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m) * zn * dxyz

            end if

         end do

      end do

      return
      end





!***********************************************************************
      subroutine dxyz_ref_cons( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fux_ref, fuy_ref, fuz_ref, natom, nbead, ikind, mbox

      use cons_variables, only : &
     &   scons, rcons, pot_ref_cons, fref_cons, fc_cons, i_cons, &
     &   j_cons, k_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, m, nk, nl, m1, m2, m3, kx, ky, kz

      real(8) :: xk, yk, zk, xl, yl, zl, xi, yi, zi, &
     &           xm, ym, zm, xn, yn, zn, dxyz

!-----------------------------------------------------------------------
!     /*   prefactor                                                  */
!-----------------------------------------------------------------------

      kx = 0
      ky = 0
      kz = 0

      if ( i_cons(m) .eq. 1 ) kx = 1
      if ( i_cons(m) .eq. 2 ) ky = 1
      if ( i_cons(m) .eq. 3 ) kz = 1

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do n = 1, nbead

!-----------------------------------------------------------------------
!        /*   calculate difference in center of masses                */
!-----------------------------------------------------------------------

         nk = 0
         nl = 0

         xk = 0.d0
         yk = 0.d0
         zk = 0.d0

         xl = 0.d0
         yl = 0.d0
         zl = 0.d0

         scons(m,n) = 0.d0

         do i = 1, natom

            if ( ikind(i) .eq. j_cons(m) ) then

               xi = x(i,n)
               yi = y(i,n)
               zi = z(i,n)

               m1 = mbox(1,i,n)
               m2 = mbox(2,i,n)
               m3 = mbox(3,i,n)

               call pbc_unfold( xi, yi, zi, m1, m2, m3 )

               nk = nk + 1

               xk = xk + xi
               yk = yk + yi
               zk = zk + zi

            end if

            if ( ikind(i) .eq. k_cons(m) ) then

               xi = x(i,n)
               yi = y(i,n)
               zi = z(i,n)

               m1 = mbox(1,i,n)
               m2 = mbox(2,i,n)
               m3 = mbox(3,i,n)

               call pbc_unfold( xi, yi, zi, m1, m2, m3 )

               nl = nl + 1

               xl = xl + xi
               yl = yl + yi
               zl = zl + zi

            end if

         end do

         if ( nk .ne. 0 ) then
            xk = xk / dble(nk)
            yk = yk / dble(nk)
            zk = zk / dble(nk)
         end if

         if ( nl .ne. 0 ) then
            xl = xl / dble(nl)
            yl = yl / dble(nl)
            zl = zl / dble(nl)
         end if

         scons(m,n) = dble(kx) * ( xk - xl ) &
     &              + dble(ky) * ( yk - yl ) &
     &              + dble(kz) * ( zk - zl )

!-----------------------------------------------------------------------
!        /*   calculate potential and forces of harmonic term         */
!-----------------------------------------------------------------------

         dxyz = scons(m,n) - rcons(m,n)

         pot_ref_cons(n) = pot_ref_cons(n) &
     &                   + 0.5d0 * fc_cons(m) * dxyz * dxyz

         fref_cons(m,n) = fref_cons(m,n) + fc_cons(m) * dxyz

         xm = 0.d0
         ym = 0.d0
         zm = 0.d0

         if ( nk .ne. 0 ) then
            xm = dble(kx) / dble(nk)
            ym = dble(ky) / dble(nk)
            zm = dble(kz) / dble(nk)
         end if

         xn = 0.d0
         yn = 0.d0
         zn = 0.d0

         if ( nl .ne. 0 ) then
            xn = dble(kx) / dble(nl)
            yn = dble(ky) / dble(nl)
            zn = dble(kz) / dble(nl)
         end if

         do i = 1, natom

            if ( ikind(i) .eq. j_cons(m) ) then

               fux_ref(i,n) = fux_ref(i,n) - fc_cons(m) * xm * dxyz
               fuy_ref(i,n) = fuy_ref(i,n) - fc_cons(m) * ym * dxyz
               fuz_ref(i,n) = fuz_ref(i,n) - fc_cons(m) * zm * dxyz

            end if

            if ( ikind(i) .eq. k_cons(m) ) then

               fux_ref(i,n) = fux_ref(i,n) + fc_cons(m) * xn * dxyz
               fuy_ref(i,n) = fuy_ref(i,n) + fc_cons(m) * yn * dxyz
               fuz_ref(i,n) = fuz_ref(i,n) + fc_cons(m) * zn * dxyz

            end if

         end do

      end do

      return
      end
