!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     hills force in metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_hills_tass_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead, myrank, nprocs

      use tass_variables, only : &
     &   pot_tass, ftass, rtass, gc_meta, gh_meta, gw_meta, dtemp_meta, &
     &   params_meta, fc_tass, ipbc_tass, ng_meta, ntass, ntass_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      integer :: ibead = 1, jbead = 1, kbead = 1

      real(8) :: f0, f1, f2, f3, f4, f5, f6, f7, dr

!cc      real(8) :: scale = 1.d-2
      real(8) :: scale = 0.25d0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   hills potential   */
      pot_tass(:) = 0.d0

!     /*   hills force   */
      ftass(:,:)  = 0.d0

!     /*   constant   */
      f5 = boltz * dtemp_meta

!-----------------------------------------------------------------------
!     /*   history-dependent gaussian potential part                  */
!-----------------------------------------------------------------------

!     /*   loop of walkers   */
      do j = 1, nbead

!        /*   only my bead   */
         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!        /*   sum   */
         f7 = 0.d0

!        /*   constraints   */
         if ( ntass_cons .eq. 0 ) ibead = 1
         if ( ntass_cons .eq. 0 ) jbead = ng_meta
         if ( ntass_cons .eq. 0 ) kbead = 1
         if ( ntass_cons .eq. 1 ) ibead = j
         if ( ntass_cons .eq. 1 ) jbead = ng_meta
         if ( ntass_cons .eq. 1 ) kbead = nbead

!        /*   loop of gaussians   */
         do i = ibead, jbead, kbead

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, ntass

!              /*   displacement   */
               f1 = rtass(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential   */
            f7 = f7 + f0

!           /*   loop of cv dimension   */
            do k = 1, ntass

!              /*   displacement   */
               f1 = rtass(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   prefactor   */
               f4 = f1 / ( f2*f2 )

!              /*   force   */
               ftass(k,j) = ftass(k,j) + f0 * f4

!           /*   loop of cv dimension   */
            end do

!        /*   loop of gaussians   */
         end do

!        /*   conventional metadynamics   */
         if ( dtemp_meta .le. 0.d0 ) then

!           /*   potential   */
            pot_tass(j) = f7

!        /*   well tempered metadynamics   */
         else

!           /*   constant   */
            f6 = f5 * log( 1.d0 + f7/f5 )

!           /*   potential   */
            pot_tass(j) = f6

!           /*   force   */
            do k = 1, ntass
               ftass(k,j) = ftass(k,j) * exp( - f6/f5 )
            end do

!        /*   metadynamics   */
         end if

!-----------------------------------------------------------------------
!        /*   half harmonic potential when cv is out of bounds        */
!-----------------------------------------------------------------------

!        /*   loop of cvs   */
         do k = 1, ntass

!           /*   distance from boundary   */
            dr = min( rtass(k,j) - params_meta(1,k), 0.d0 )

!           /*   potential   */
            pot_tass(j) = pot_tass(j) &
     &         + scale * 0.5d0 * fc_tass(k) * dr * dr

!           /*   force   */
            ftass(k,j) = ftass(k,j) &
     &         - scale * fc_tass(k) * dr

!           /*   distance from boundary   */
            dr = max( rtass(k,j) - params_meta(2,k), 0.d0 )

!           /*   potential   */
            pot_tass(j) = pot_tass(j) &
     &         + scale * 0.5d0 * fc_tass(k) * dr * dr

!           /*   force   */
            ftass(k,j) = ftass(k,j) &
     &         - scale * fc_tass(k) * dr

!        /*   loop of cvs   */
         end do

!     /*   loop of walkers   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   hills potential   */
      call my_mpi_allreduce_real_1 ( pot_tass, nbead )

!     /*   hills force   */
      call my_mpi_allreduce_real_2 ( ftass, ntass, nbead )

      return
      end





!***********************************************************************
      subroutine getforce_lifted_tass_MPI( j )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead

      use tass_variables, only : &
     &   gh_meta, rtass, gc_meta, gw_meta, pot_lifted, dtemp_meta, &
     &   ipbc_tass, ng_meta, ntass, ntass_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      integer :: ibead = 1, jbead = 1, kbead = 1

      integer :: ig_meta = 1, jg_meta = 1, kg_meta = 1

      real(8) :: f0, f1, f2, f3, f5, f7, f8, f9

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!-----------------------------------------------------------------------
!     /*   history-dependent gaussian potential part                  */
!-----------------------------------------------------------------------

!     /*   conventional metadynamics   */
      if ( dtemp_meta .le. 0.d0 ) then

!        /*   last gaussian   */
         i = ng_meta

!        /*   for initial steps   */
         if ( i .lt. 1 ) return

!        /*   constraints   */
         if ( ntass_cons .eq. 0 ) ibead = 1
         if ( ntass_cons .eq. 0 ) jbead = nbead
         if ( ntass_cons .eq. 0 ) kbead = 1
         if ( ntass_cons .eq. 1 ) ibead = j
         if ( ntass_cons .eq. 1 ) jbead = j
         if ( ntass_cons .eq. 1 ) kbead = 1

!        /*   loop of walkers   */
         do l = ibead, jbead, kbead

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, ntass

!              /*   displacement   */
               f1 = rtass(k,l) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential correction   */
            pot_lifted = pot_lifted + f0

!        /*   loop of walkers   */
         end do

!     /*   well tempered metadynamics   */
      else

!        /*   constraints   */
         if ( ntass_cons .eq. 0 ) ig_meta = 1
         if ( ntass_cons .eq. 0 ) jg_meta = ng_meta-1
         if ( ntass_cons .eq. 0 ) kg_meta = 1
         if ( ntass_cons .eq. 1 ) ig_meta = j
         if ( ntass_cons .eq. 1 ) jg_meta = ng_meta-nbead
         if ( ntass_cons .eq. 1 ) kg_meta = nbead

!        /*   constraints   */
         if ( ntass_cons .eq. 0 ) ibead = 1
         if ( ntass_cons .eq. 0 ) jbead = nbead
         if ( ntass_cons .eq. 0 ) kbead = 1
         if ( ntass_cons .eq. 1 ) ibead = j
         if ( ntass_cons .eq. 1 ) jbead = j
         if ( ntass_cons .eq. 1 ) kbead = 1

!        /*   loop of beads   */
         do l = ibead, jbead, kbead

!           /*   potential   */
            f7 = 0.d0

!           /*   loop of gaussians   */
            do i = ig_meta, jg_meta, kg_meta

!              /*   height   */
               f0 = gh_meta(i)

!              /*   loop of cv dimension   */
               do k = 1, ntass

!                 /*   displacement   */
                  f1 = rtass(k,l) - gc_meta(k,i)

!                 /*   periodic boundary condition   */
                  call pbc_tass( ipbc_tass(k), f1 )

!                 /*   width   */
                  f2 = gw_meta(k,i)

!                 /*   exponent   */
                  f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!                 /*   potential   */
                  f0 = f0 * exp(-f3)

!              /*   loop of cv dimension   */
               end do

!              /*   potential   */
               f7 = f7 + f0

!           /*   loop of gaussians   */
            end do

!           /*   constant   */
            f8 = f5 * log( 1.d0 + f7/f5 )

!           /*   loop of gaussians   */
            i = ng_meta

!           /*   for initial steps   */
            if ( i .lt. 1 ) return

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, ntass

!              /*   displacement   */
               f1 = rtass(k,l) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_tass( ipbc_tass(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential   */
            f7 = f7 + f0

!           /*   constant   */
            f9 = f5 * log( 1.d0 + f7/f5 )

!           /*   potential correction   */
            pot_lifted = pot_lifted + f9 - f8

!        /*   loop of beads   */
         end do

!     /*   metadynamics   */
      end if

      return
      end
