!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     hills force in metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_hills_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead, myrank, nprocs

      use meta_variables, only : &
     &   pot_meta, fmeta, rmeta, gc_meta, gh_meta, gw_meta, dtemp_meta, &
     &   params_meta, fc_meta, ipbc_meta, ng_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: f0, f1, f2, f3, f4, f5, f6, f7, dr

!cc      real(8) :: scale = 1.d-2
      real(8) :: scale = 0.25d0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   hills potential   */
      pot_meta(:) = 0.d0

!     /*   hills force   */
      fmeta(:,:)  = 0.d0

!     /*   constant   */
      f5 = boltz * dtemp_meta

!-----------------------------------------------------------------------
!     /*   history-dependent gaussian potential part                  */
!-----------------------------------------------------------------------

!     /*   loop of walkers   */
      do j = 1, nbead

!        /*   only my bead   */
         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!        /*   sum   */
         f7 = 0.d0

!        /*   loop of gaussians   */
         do i = 1, ng_meta

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, nmeta

!              /*   displacement   */
               f1 = rmeta(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_meta( ipbc_meta(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential   */
            f7 = f7 + f0

!           /*   loop of cv dimension   */
            do k = 1, nmeta

!              /*   displacement   */
               f1 = rmeta(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_meta( ipbc_meta(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   prefactor   */
               f4 = f1 / ( f2*f2 )

!              /*   force   */
               fmeta(k,j) = fmeta(k,j) + f0 * f4

!           /*   loop of cv dimension   */
            end do

!        /*   loop of gaussians   */
         end do

!        /*   conventional metadynamics   */
         if ( dtemp_meta .le. 0.d0 ) then

!           /*   potential   */
            pot_meta(j) = f7

!        /*   well tempered metadynamics   */
         else

!           /*   constant   */
            f6 = f5 * log( 1.d0 + f7/f5 )

!           /*   potential   */
            pot_meta(j) = f6

!           /*   force   */
            do k = 1, nmeta
               fmeta(k,j) = fmeta(k,j) * exp( - f6/f5 )
            end do

!        /*   metadynamics   */
         end if

!-----------------------------------------------------------------------
!        /*   half harmonic potential when cv is out of bounds        */
!-----------------------------------------------------------------------

!        /*   loop of cvs   */
         do k = 1, nmeta

!           /*   distance from boundary   */
            dr = min( rmeta(k,j) - params_meta(1,k), 0.d0 )

!           /*   potential   */
            pot_meta(j) = pot_meta(j) &
     &         + scale * 0.5d0 * fc_meta(k) * dr * dr

!           /*   force   */
            fmeta(k,j) = fmeta(k,j) &
     &         - scale * fc_meta(k) * dr

!           /*   distance from boundary   */
            dr = max( rmeta(k,j) - params_meta(2,k), 0.d0 )

!           /*   potential   */
            pot_meta(j) = pot_meta(j) &
     &         + scale * 0.5d0 * fc_meta(k) * dr * dr

!           /*   force   */
            fmeta(k,j) = fmeta(k,j) &
     &         - scale * fc_meta(k) * dr

!        /*   loop of cvs   */
         end do

!     /*   loop of walkers   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   hills potential   */
      call my_mpi_allreduce_real_1 ( pot_meta, nbead )

!     /*   hills force   */
      call my_mpi_allreduce_real_2 ( fmeta, nmeta, nbead )

      return
      end





!***********************************************************************
      subroutine getforce_lifted_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   boltz, nbead

      use meta_variables, only : &
     &   gh_meta, rmeta, gc_meta, gw_meta, pot_lifted, dtemp_meta, &
     &   ipbc_meta, ng_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: f0, f1, f2, f3, f5, f7, f8, f9

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      f5 = boltz * dtemp_meta

!-----------------------------------------------------------------------
!     /*   history-dependent gaussian potential part                  */
!-----------------------------------------------------------------------

!     /*   conventional metadynamics   */
      if ( dtemp_meta .le. 0.d0 ) then

!        /*   last gaussian   */
         i = ng_meta

!        /*   loop of walkers   */
         do j = 1, nbead

!           /*   for initial steps   */
            if ( i .lt. 1 ) cycle

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, nmeta

!              /*   displacement   */
               f1 = rmeta(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_meta( ipbc_meta(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential correction   */
            pot_lifted = pot_lifted + f0

!        /*   loop of walkers   */
         end do

!     /*   well tempered metadynamics   */
      else

!        /*   loop of walkers   */
         do j = 1, nbead

!           /*   potential   */
            f7 = 0.d0

!           /*   loop of gaussians   */
            do i = 1, ng_meta-1

!              /*   height   */
               f0 = gh_meta(i)

!              /*   loop of cv dimension   */
               do k = 1, nmeta

!                 /*   displacement   */
                  f1 = rmeta(k,j) - gc_meta(k,i)

!                 /*   periodic boundary condition   */
                  call pbc_meta( ipbc_meta(k), f1 )

!                 /*   width   */
                  f2 = gw_meta(k,i)

!                 /*   exponent   */
                  f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!                 /*   potential   */
                  f0 = f0 * exp(-f3)

!              /*   loop of cv dimension   */
               end do

!              /*   potential   */
               f7 = f7 + f0

!           /*   loop of gaussians   */
            end do

!           /*   constant   */
            f8 = f5 * log( 1.d0 + f7/f5 )

!           /*   loop of gaussians   */
            i = ng_meta

!           /*   height   */
            f0 = gh_meta(i)

!           /*   loop of cv dimension   */
            do k = 1, nmeta

!              /*   displacement   */
               f1 = rmeta(k,j) - gc_meta(k,i)

!              /*   periodic boundary condition   */
               call pbc_meta( ipbc_meta(k), f1 )

!              /*   width   */
               f2 = gw_meta(k,i)

!              /*   exponent   */
               f3 = 0.5d0 * ( f1*f1 ) / ( f2*f2 )

!              /*   potential   */
               f0 = f0 * exp(-f3)

!           /*   loop of cv dimension   */
            end do

!           /*   potential   */
            f7 = f7 + f0

!           /*   constant   */
            f9 = f5 * log( 1.d0 + f7/f5 )

!           /*   potential correction   */
            pot_lifted = pot_lifted + f9 - f8

!        /*   loop of walkers   */
         end do

!     /*   metadynamics   */
      end if

      return
      end
