!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from alchemical mixture
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_alchem_MPI
!***********************************************************************
!=======================================================================
!
!     alchemical mixture
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, potential, dipx, dipy, dipz, vir, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      call setup_alchem_MPI

!-----------------------------------------------------------------------
!     /*   initialize, potential, forces, dipole, virial              */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   alchem: direct calculation                                 */
!-----------------------------------------------------------------------

      call getforce_alchem_a_MPI

      call getforce_alchem_b_MPI

!-----------------------------------------------------------------------
!     /*   alchem: sum of all contributions                           */
!-----------------------------------------------------------------------

      call getforce_alchem_sum_MPI

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j) / dble(natom)
            fy(i,j) = fy(i,j) / dble(natom)
            fz(i,j) = fz(i,j) / dble(natom)

         end do

      end do

      potential = potential / dble(nbead)

      return
      end





!***********************************************************************
      subroutine getforce_alchem_sum_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, nbead, natom

      use alchem_variables, only : &
     &   pot_a, fx_a, fy_a, fz_a, vir_a, pot_alchem, ratio_alchem, &
     &   pot_b, fx_b, fy_b, fz_b, vir_b, fx_alchem, fy_alchem, &
     &   fz_alchem, vir_a, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize variables   */
      implicit none

!     /*   integers   */
      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   mixed potential                                            */
!-----------------------------------------------------------------------

      do j = 1, nbead

         pot(j) = ratio_alchem(1,j) * pot_a(j) &
     &          + ratio_alchem(2,j) * pot_b(j)

      end do

!-----------------------------------------------------------------------
!     /*   mixed potential: i-th mixture for j-th bead                */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead

         pot_alchem(i,j) = ratio_alchem(1,i) * pot_a(j) &
     &                   + ratio_alchem(2,i) * pot_b(j)

      end do
      end do

!-----------------------------------------------------------------------
!     /*   mixed force                                                */
!-----------------------------------------------------------------------

      do j = 1, nbead

         do k = 1, natom

            fx(k,j) = 0.d0
            fy(k,j) = 0.d0
            fz(k,j) = 0.d0

         end do

         do k = 1, natom

            fx(k,j) = fx(k,j) + ratio_alchem(1,j) * fx_a(k,j)
            fy(k,j) = fy(k,j) + ratio_alchem(1,j) * fy_a(k,j)
            fz(k,j) = fz(k,j) + ratio_alchem(1,j) * fz_a(k,j)

         end do

         do k = 1, natom

            fx(k,j) = fx(k,j) + ratio_alchem(2,j) * fx_b(k,j)
            fy(k,j) = fy(k,j) + ratio_alchem(2,j) * fy_b(k,j)
            fz(k,j) = fz(k,j) + ratio_alchem(2,j) * fz_b(k,j)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   mixed potential: i-th mixture for j-th bead                */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, nbead

         do k = 1, natom

            fx_alchem(k,i,j) = 0.d0
            fy_alchem(k,i,j) = 0.d0
            fz_alchem(k,i,j) = 0.d0

         end do

         do k = 1, natom

            fx_alchem(k,i,j) = fx_alchem(k,i,j) &
     &                       + ratio_alchem(1,i) * fx_a(k,j)
            fy_alchem(k,i,j) = fy_alchem(k,i,j) &
     &                       + ratio_alchem(1,i) * fy_a(k,j)
            fz_alchem(k,i,j) = fz_alchem(k,i,j) &
     &                       + ratio_alchem(1,i) * fz_a(k,j)

         end do

         do k = 1, natom

            fx_alchem(k,i,j) = fx_alchem(k,i,j) &
     &                       + ratio_alchem(2,i) * fx_b(k,j)
            fy_alchem(k,i,j) = fy_alchem(k,i,j) &
     &                       + ratio_alchem(2,i) * fy_b(k,j)
            fz_alchem(k,i,j) = fz_alchem(k,i,j) &
     &                       + ratio_alchem(2,i) * fz_b(k,j)

         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   mixed virial                                               */
!-----------------------------------------------------------------------

      do j = 1, 3
      do i = 1, 3

         vir(i,j) = 0.d0

         do k = 1, nbead
            vir(i,j) = vir(i,j) + ratio_alchem(1,k) * vir_a(i,j) &
     &                          + ratio_alchem(2,k) * vir_b(i,j)

         end do

         vir(i,j) = vir(i,j) / dble(nbead)

      end do
      end do

      return
      end





!***********************************************************************
      subroutine getforce_alchem_a_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, box, mbox, nbead, iounit, pimd_command, natom, &
     &   myrank, nprocs

      use alchem_variables, only : &
     &   pot_a, fx_a, fy_a, fz_a, vir_a, alchem_scr_dir_a

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!     /*   real numbers   */
      real(8) :: xi, yi, zi

!     /*   characters   */
      character(len=80) :: char_file
      character(len=80) :: char_dir
      character(len=3)  :: char_num

!     /*   real numbers   */
      real(8) :: x_t(natom), y_t(natom), z_t(natom)
      real(8) :: fx_t(natom), fy_t(natom), fz_t(natom)
      real(8) :: e_t(1)
      real(8) :: vir_t(3,3)

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do i = 1, nbead

         pot_a(i) = 0.d0

         do j = 1, natom
            fx_a(j,i) = 0.d0
            fy_a(j,i) = 0.d0
            fz_a(j,i) = 0.d0
         end do

      end do

      do j = 1, 3
      do k = 1, 3
         vir_a(k,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         call int3_to_char( j, char_num )

!-----------------------------------------------------------------------
!        /*   x_t, y_t, z_t  =  xyz of the primary subsystem.         */
!-----------------------------------------------------------------------

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            x_t(i) = xi
            y_t(i) = yi
            z_t(i) = zi

         end do

!-----------------------------------------------------------------------
!        /*   make geometry.ini                                       */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/geometry.ini'

         call iogeometry( 1, char_file, len(char_file), iounit, &
     &                       x_t, y_t, z_t, natom, 1 )

!-----------------------------------------------------------------------
!        /*   make box.ini                                            */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/box.ini'

         call iobox( 1, char_file, len(char_file), iounit, box )

!-----------------------------------------------------------------------
!        /*   execute at lower directories                            */
!-----------------------------------------------------------------------

         char_dir = trim(alchem_scr_dir_a) // '/' // char_num

         call system ('cd ' // char_dir // '; ' // trim(pimd_command) &
     &                 // ' > ./monitor.out; cd ../../')

!-----------------------------------------------------------------------
!        /*   read forces.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/forces.out'

         call ioforce ( 2, char_file, len(char_file), iounit, &
     &                  e_t, fx_t, fy_t, fz_t, natom, 1 )

!-----------------------------------------------------------------------
!        /*   read virial.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_a) // '/' // char_num // &
     &               '/virial.out'

         call iovirial( 2, char_file, len(char_file), iounit, vir_t )

!-----------------------------------------------------------------------
!        /*    energy                                                 */
!-----------------------------------------------------------------------

         pot_a(j) = e_t(1)

!-----------------------------------------------------------------------
!        /*    force                                                  */
!-----------------------------------------------------------------------

         do i = 1, natom

            fx_a(i,j) = fx_t(i)
            fy_a(i,j) = fy_t(i)
            fz_a(i,j) = fz_t(i)

         end do

!-----------------------------------------------------------------------
!        /*   virial                                                  */
!-----------------------------------------------------------------------

         do i = 1, 3
         do k = 1, 3
            vir_a(k,i) = vir_a(k,i) + vir_t(k,i)
         end do
         end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_real_1 ( pot_a, nbead )
      call my_mpi_allreduce_real_2 ( fx_a, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_a, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_a, natom, nbead )
      call my_mpi_allreduce_real_2 ( vir_a, 3, 3 )

      return
      end





!***********************************************************************
      subroutine getforce_alchem_b_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, box, mbox, nbead, iounit, pimd_command, natom, &
     &   myrank, nprocs

      use alchem_variables, only : &
     &   pot_b, fx_b, fy_b, fz_b, vir_b, alchem_scr_dir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!     /*   real numbers   */
      real(8) :: xi, yi, zi

!     /*   characters   */
      character(len=80) :: char_file
      character(len=80) :: char_dir
      character(len=3)  :: char_num

!     /*   real numbers   */
      real(8) :: x_t(natom), y_t(natom), z_t(natom)
      real(8) :: fx_t(natom), fy_t(natom), fz_t(natom)
      real(8) :: e_t(1)
      real(8) :: vir_t(3,3)

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do i = 1, nbead

         pot_b(i) = 0.d0

         do j = 1, natom
            fx_b(j,i) = 0.d0
            fy_b(j,i) = 0.d0
            fz_b(j,i) = 0.d0
         end do

      end do

      do j = 1, 3
      do k = 1, 3
         vir_b(k,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         call int3_to_char( j, char_num )

!-----------------------------------------------------------------------
!        /*   x_t, y_t, z_t  =  xyz of the primary subsystem.         */
!-----------------------------------------------------------------------

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            x_t(i) = xi
            y_t(i) = yi
            z_t(i) = zi

         end do

!-----------------------------------------------------------------------
!        /*   make geometry.ini                                       */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/geometry.ini'

         call iogeometry( 1, char_file, len(char_file), iounit, &
     &                       x_t, y_t, z_t, natom, 1 )

!-----------------------------------------------------------------------
!        /*   make box.ini                                            */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/box.ini'

         call iobox( 1, char_file, len(char_file), iounit, box )

!-----------------------------------------------------------------------
!        /*   execute at lower directories                            */
!-----------------------------------------------------------------------

         char_dir = trim(alchem_scr_dir_b) // '/' // char_num

         call system ('cd ' // char_dir // '; ' // trim(pimd_command) &
     &                 // ' > ./monitor.out; cd ../../')

!-----------------------------------------------------------------------
!        /*   read forces.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/forces.out'

         call ioforce ( 2, char_file, len(char_file), iounit, &
     &                  e_t, fx_t, fy_t, fz_t, natom, 1 )

!-----------------------------------------------------------------------
!        /*   read virial.out                                         */
!-----------------------------------------------------------------------

         char_file = trim(alchem_scr_dir_b) // '/' // char_num // &
     &               '/virial.out'

         call iovirial( 2, char_file, len(char_file), iounit, vir_t )

!-----------------------------------------------------------------------
!        /*    energy                                                 */
!-----------------------------------------------------------------------

         pot_b(j) = e_t(1)

!-----------------------------------------------------------------------
!        /*    force                                                  */
!-----------------------------------------------------------------------

         do i = 1, natom

            fx_b(i,j) = fx_t(i)
            fy_b(i,j) = fy_t(i)
            fz_b(i,j) = fz_t(i)

         end do

!-----------------------------------------------------------------------
!        /*   virial                                                  */
!-----------------------------------------------------------------------

         do i = 1, 3
         do k = 1, 3
            vir_b(k,i) = vir_b(k,i) + vir_t(k,i)
         end do
         end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_real_1 ( pot_b, nbead )
      call my_mpi_allreduce_real_2 ( fx_b, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_b, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_b, natom, nbead )
      call my_mpi_allreduce_real_2 ( vir_b, 3, 3 )

      return
      end
