!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     calculate energy and force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, vir_bead, potential, x, y, z, nbead, &
     &   ipotential, myrank

      use XMPI_variables, only : &
     &   jstart_atom, jend_atom, jstart_bead, jend_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      pot(:) = 0.d0

      potential = 0.d0

      do j = jstart_bead, jend_bead
         fx(:,j) = 0.d0
         fy(:,j) = 0.d0
         fz(:,j) = 0.d0
      end do

      vir(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   communicate atoms for jstart_bead < j < jend_bead          */
!-----------------------------------------------------------------------

      call my_mpi_bcast_xyz_XMPI( x, y, z, 1 )

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

!     ===  n2p2  ===

      if ( ipotential(1:5) .eq. 'N2P2 ' ) then

         call force_n2p2_XMPI

!     ===  classical force field  ===

      else if  ( ipotential(1:3) .eq. 'MM ' ) then

         call force_mm_XMPI

!     ===  embedded atom method  ===

      else if  ( ipotential(1:4) .eq. 'EAM ' ) then

         call force_eam_XMPI

!     ===  helium  ===

      else if  ( ipotential(1:7) .eq. 'HELIUM ' ) then

         call force_helium_aziz1992_XMPI

!     ===  helium  ===

      else

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &      'Error - XMPI not supported for ' // trim(ipotential) // '.'
            write( 6, '(a)' )
         end if

         call error_handling_MPI( 1, 'subroutine getforce_MPI', 23 )

      end if

!-----------------------------------------------------------------------
!     /*   collect potential, virial of beads (but not forces!)       */
!-----------------------------------------------------------------------

!     //   note: forces are not shared because they are large
      call my_mpi_bcast_pot_XMPI( pot, vir_bead )

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      do j = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         fx(i,j) = fx(i,j) / dble(nbead)
         fy(i,j) = fy(i,j) / dble(nbead)
         fz(i,j) = fz(i,j) / dble(nbead)
      end do
      end do

      do j = 1, nbead
         potential = potential + pot(j) / dble(nbead)
      end do

      do j = 1, nbead
         vir(:,:) = vir(:,:) + vir_bead(:,:,j) / dble(nbead)
      end do

!-----------------------------------------------------------------------
!     /*   apply virial correction                                    */
!-----------------------------------------------------------------------

      call correct_virial_XMPI

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force_XMPI

      return
      end
