!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     calculate energy and force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, dipx, dipy, dipz, vir, potential, &
     &   natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      call getforce_main_MPI

!-----------------------------------------------------------------------
!     /*   external potential                                         */
!-----------------------------------------------------------------------

      call force_external_MPI

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

!-----------------------------------------------------------------------
!     /*   apply virial correction                                    */
!-----------------------------------------------------------------------

      call correct_virial

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force_MPI

      return
      end





!***********************************************************************
      subroutine getforce_rem_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, dipx, dipy, dipz, vir, potential, vir_bead, &
     &   natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

      vir_bead(:,:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      call getforce_main_MPI

!-----------------------------------------------------------------------
!     /*   external potential                                         */
!-----------------------------------------------------------------------

      call force_external_MPI

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

      return
      end





!***********************************************************************
      subroutine getforce_main_MPI
!***********************************************************************
!=======================================================================
!
!     warning - vir_bead is implemented for vasp and aenet only.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ipotential, qmmm_embedding, ensemble, method, myrank

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      if ( ( method(1:5) .eq. 'REHMC' ) .and. &
     &     ( ensemble(1:3) .eq. 'NPT' ) ) then

         if ( ( ipotential(1:5) .ne. 'VASP ' ) .and. &
     &        ( ipotential(1:5) .ne. 'AENET' ) ) then

            if ( myrank .eq. 0 ) then

               write( 6, '(a)' ) &
     &            'Sorry - bead virial is not implemented yet for ' // &
     &            trim(ipotential) // '.'

            end if

!           /*   error termination   */
            call error_handling_MPI &
     &         ( 1, 'subroutine getforce_main_MPI', 28 )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

!     ===  water  ===

      if      ( ipotential(1:6) .eq. 'WATER ' ) then

         call force_water_MPI

!     ===  metal-water interface  ===

     else if ( ipotential(1:12) .eq. 'METALWATER ' ) then

         call force_metalwater_MPI

!     ===  hydrogen  ===

      else if  ( ipotential(1:9) .eq. 'HYDROGEN ' ) then

         call force_hydrogen_silveragoldman_MPI

!     ===  helium  ===

      else if  ( ipotential(1:7) .eq. 'HELIUM ' ) then

!cc         call force_helium_aziz1991_MPI
         call force_helium_aziz1992_MPI

!     ===  polarizable mm  ===

      else if ( ipotential(1:4) .eq. 'POL ' ) then

         call force_pol_MPI

!     ===  mm  ===

      else if ( ipotential(1:3) .eq. 'MM ' ) then

         call force_mm_MPI

!     ===  QM/MM electronic embedding  ===

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'EE ' ) ) then

         call force_ee_MPI

!     ===  QM/MM mechanical embedding  ===

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

         call force_me_MPI

!     ===  ONIOM  ===

      else if ( ipotential(1:5) .eq. 'ONIOM ' ) then

         call force_oniom_MPI

!     ===  eam  ===

      else if ( ipotential(1:4) .eq. 'EAM ' ) then

         call force_eam_MPI

!     ===  adp  ===

      else if ( ipotential(1:4) .eq. 'ADP ' ) then

         call force_adp_MPI

!     ===  mopac  ===

      else if ( ipotential(1:6) .eq. 'MOPAC ' ) then

         call force_mopac_MPI

!     ===  smash potential  ===

      else if ( ipotential(1:6) .eq. 'SMASH ' ) then

         call force_smash_MPI

!     ===  ntchem potential  ===

      else if ( ipotential(1:7) .eq. 'NTCHEM ' ) then

         call force_ntchem_MPI

!     ===  g16  ===

      else if ( ipotential(1:4) .eq. 'G16 ' ) then

         call force_g16_MPI

!     ===  g09  ===

      else if ( ipotential(1:4) .eq. 'G09 ' ) then

         call force_g09_MPI

!     ===  g03  ===

      else if ( ipotential(1:4) .eq. 'G03 ' ) then

         call force_g03_MPI

!     ===  g98  ===

      else if ( ipotential(1:4) .eq. 'G98 ' ) then

         call force_g98_MPI

!     ===  turbomole  ===

      else if ( ipotential(1:10) .eq. 'TURBOMOLE ' ) then

         call force_turbo_MPI

!     ===  gamess  ===

      else if ( ipotential(1:7) .eq. 'GAMESS ' ) then

         call force_gamess_MPI

!     ===  PHASE/0 ===

      else if ( ipotential(1:7) .eq. 'PHASE0 ' ) then

         call force_phase0_MPI

!     ===  vasp6 (embedded)  ===

      else if ( ipotential(1:6) .eq. 'VASP6 ' ) then 

         call force_vasp6_MPI

!     ===  vasp (system call) ===

      else if ( ipotential(1:6) .eq. 'VASP5 ' ) then

         call force_vasp5_MPI

!     ===  vasp (embedded) ===

      else if ( ipotential(1:5) .eq. 'VASP ' ) then

         call force_vasp_MPI

!     ===  quantum espresso ===

      else if ( ipotential(1:3) .eq. 'QE ' ) then

         call force_qe_MPI

!     ===  quantum espresso (git) ===

      else if ( ipotential(1:5) .eq. 'QEGIT ' ) then

         call force_qegit_MPI

!     ===  molpro  ===

      else if ( ipotential(1:7) .eq. 'MOLPRO ' ) then

         call force_molpro_MPI

!     ===  orca  ===

      else if ( ipotential(1:5) .eq. 'ORCA ' ) then

         call force_orca_MPI

!     ===  dftb  ===

      else if ( ipotential(1:5) .eq. 'DFTB ' ) then

         call force_dftb_MPI

!     ===  xtb  ===

      else if ( ipotential(1:4) .eq. 'XTB ' ) then

         call force_xtb_MPI

!     === dftb as a library ===

      else if (ipotential(1:8) .eq. 'DFTBLIB ') then

         call force_libdftb_MPI

!     === cp2k as a library ===

      else if (ipotential(1:8) .eq. 'CP2KLIB ') then

         call force_libcp2k_MPI

!     ===  abinit-mp 5.0 (system call)  ===

      else if ( ipotential(1:11) .eq. 'ABINIT-MP5 ' ) then

         call force_abinit_mp5_MPI

!     ===  abinit-mp 5.0 (embedded)  ===

      else if ( ipotential(1:10) .eq. 'ABINIT-MP ' ) then

         call force_abinit_mp_MPI

!     ===  user defined  ===

      else if ( ipotential(1:5) .eq. 'USER ' ) then

         call force_user_MPI

!     ===  quartic potential (test)  ===

      else if ( ipotential(1:8) .eq. 'QUARTIC ' ) then

         call force_quartic_MPI

!     ===  pair potential  ===

      else if ( ipotential(1:5) .eq. 'PAIR ' ) then

         call force_pair_MPI

!     ===  pair potential  ===

      else if ( ipotential(1:8) .eq. 'TERSOFF ' ) then

         call force_tersoff_MPI

!     ===  tully's model (test)  ===

      else if ( ipotential(1:6) .eq. 'TULLY ' ) then

         call force_tully_MPI

!     ===  tip4p (test)  ===

      else if ( ipotential(1:6) .eq. 'TIP4P ' ) then

         call force_mm_tip4p_MPI

!     ===  cartesian (test)  ===

      else if ( ipotential(1:5) .eq. 'CART ' ) then

         call force_cart_MPI

!     ===  aenet  ===

      else if ( ipotential(1:6) .eq. 'AENET ' ) then

         call force_aenet_MPI

!     ===  n2p2  ===

      else if ( ipotential(1:5) .eq. 'N2P2 ' ) then

      call force_n2p2_MPI

!     ===  mtp  ===

      else if ( ipotential(1:4) .eq. 'MTP ' ) then

         call force_mtp_MPI

!     ===  rubnnet4md  ===

      else if ( ipotential(1:7) .eq. 'LIBNNP ' ) then

         call force_libnnp_MPI

!     ===  muller-brown  ===

      else if ( ipotential(1:7) .eq. 'MULLER ' ) then

         call force_muller_MPI

!     ===  ojamae-shavitt-singer  ===

      else if ( ipotential(1:4) .eq. 'OSS ' ) then

         call force_oss_MPI

!     ===  harmonic oscillators bath (test)  ===

      else if ( ipotential(1:4) .eq. 'HOB ' ) then

         call force_hob_MPI

!     ===  Matlantis PFP  ===

      else if ( ipotential(1:4) .eq. 'PFP ' ) then

         call force_pfp_MPI

!     ===  error  ===

      else

         call error_handling_MPI &
     &      ( 1, 'subroutine getforce_main_MPI', 28 )

      end if

      return
      end

