!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Apr 24, 2022 by M. Shiga
!      Description:     calculate energy and force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getforce
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, dipx, dipy, dipz, vir, potential, &
     &   natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      call getforce_main

!-----------------------------------------------------------------------
!     /*   external potential                                         */
!-----------------------------------------------------------------------

      call force_external

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

!-----------------------------------------------------------------------
!     /*   apply virial correction                                    */
!-----------------------------------------------------------------------

      call correct_virial

!-----------------------------------------------------------------------
!     /*   correct forces on centroid                                 */
!-----------------------------------------------------------------------

      call correct_force

      return
      end





!***********************************************************************
      subroutine getforce_rem
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, dipx, dipy, dipz, vir, potential, vir_bead, &
     &   natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: dp

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      do k = 1, nbead

         pot(k) = 0.d0

         do i = 1, natom

            fx(i,k) = 0.d0
            fy(i,k) = 0.d0
            fz(i,k) = 0.d0

         end do

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

      end do

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = 0.d0
      end do
      end do

      vir_bead(:,:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      call getforce_main

!-----------------------------------------------------------------------
!     /*   external potential                                         */
!-----------------------------------------------------------------------

      call force_external

!-----------------------------------------------------------------------
!     /*   potential and force are divided by nbead                   */
!-----------------------------------------------------------------------

      dp = dble(nbead)

      potential = 0.d0

      do j = 1, nbead

         potential = potential + pot(j)

         do i = 1, natom

            fx(i,j) = fx(i,j)/dp
            fy(i,j) = fy(i,j)/dp
            fz(i,j) = fz(i,j)/dp

         end do

      end do

      potential = potential/dp

      do j = 1, 3
      do i = 1, 3
         vir(i,j) = vir(i,j)/dp
      end do
      end do

      return
      end





!***********************************************************************
      subroutine getforce_main
!***********************************************************************
!=======================================================================
!
!     warning - vir_bead is implemented for vasp and aenet only.
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ipotential, qmmm_embedding, ensemble, method

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

      if ( ( method(1:5) .eq. 'REHMC' ) .and. &
     &     ( ensemble(1:3) .eq. 'NPT' ) ) then

         if ( ( ipotential(1:5) .ne. 'VASP ' ) .and. &
     &        ( ipotential(1:5) .ne. 'AENET' ) ) then

            write( 6, '(a)' ) &
     &         'Sorry - bead virial is not implemented yet for ' // &
     &         trim(ipotential) // '.'

!           /*   error termination   */
            call error_handling ( 1, 'subroutine getforce_main', 24 )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   calculate potential and force                              */
!-----------------------------------------------------------------------

!     ===  water  ===

      if      ( ipotential(1:6) .eq. 'WATER ' ) then

         call force_water

!     ===  metal-water interface  ===

      else if  ( ipotential(1:11) .eq. 'METALWATER ' ) then

         call force_metalwater

!     ===  hydrogen  ===

      else if  ( ipotential(1:9) .eq. 'HYDROGEN ' ) then

         call force_hydrogen_silveragoldman

!     ===  helium  ===

      else if  ( ipotential(1:7) .eq. 'HELIUM ' ) then

!cc         call force_helium_aziz1991
         call force_helium_aziz1992

!     ===  mm  ===

      else if ( ipotential(1:3) .eq. 'MM ' ) then

         call force_mm

!     ===  polarizable mm  ===

      else if ( ipotential(1:4) .eq. 'POL ' ) then

         call force_pol

!     ===  QM/MM electronic embedding  ===

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'EE ' ) ) then

         call force_ee

!     ===  QM/MM mechanical embedding  ===

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

         call force_me

!     ===  ONIOM  ===

      else if ( ipotential(1:5) .eq. 'ONIOM ' ) then

         call force_oniom

!     ===  eam  ===

      else if ( ipotential(1:4) .eq. 'EAM ' ) then

         call force_eam

!     ===  adp  ===

      else if ( ipotential(1:4) .eq. 'ADP ' ) then

         call force_adp

!     ===  mopac  ===

      else if ( ipotential(1:6) .eq. 'MOPAC ' ) then

         call force_mopac

!     ===  smash potential  ===

      else if ( ipotential(1:6) .eq. 'SMASH ' ) then

         call force_smash

!     ===  ntchem potential  ===

      else if ( ipotential(1:7) .eq. 'NTCHEM ' ) then

         call force_ntchem

!     ===  g16  ===

      else if ( ipotential(1:4) .eq. 'G16 ' ) then

         call force_g16

!     ===  g09  ===

      else if ( ipotential(1:4) .eq. 'G09 ' ) then

         call force_g09

!     ===  g03  ===

      else if ( ipotential(1:4) .eq. 'G03 ' ) then

         call force_g03

!     ===  g98  ===

      else if ( ipotential(1:4) .eq. 'G98 ' ) then

         call force_g98

!     ===  turbomole  ===

      else if ( ipotential(1:10) .eq. 'TURBOMOLE ' ) then

         call force_turbo

!     ===  gamess  ===

      else if ( ipotential(1:7) .eq. 'GAMESS ' ) then

         call force_gamess

!     ===  PHASE/0 ===

      else if ( ipotential(1:7) .eq. 'PHASE0 ' ) then

         call force_phase0

!     ===  vasp (system call) ===

      else if ( ipotential(1:6) .eq. 'VASP5 ' ) then

         call force_vasp5

!     ===  vasp (embedded) ===

      else if ( ipotential(1:6) .eq. 'VASP  ' ) then

         call force_vasp

!     ===  vasp (embedded) ===

      else if ( ipotential(1:6) .eq. 'VASP6 ' ) then

         call force_vasp6

!     ===  quantum espresso ===

      else if ( ipotential(1:2) .eq. 'QE' ) then

!cc         call force_qe
         call force_qe7

!     ===  molpro  ===

      else if ( ipotential(1:7) .eq. 'MOLPRO ' ) then

         call force_molpro

!     ===  orca  ===

      else if ( ipotential(1:5) .eq. 'ORCA ' ) then

         call force_orca

!     ===  dftb  ===

      else if ( ipotential(1:5) .eq. 'DFTB ' ) then

         call force_dftb

!     ===  xtb  ===

      else if ( ipotential(1:4) .eq. 'XTB ' ) then

         call force_xtb

!     === dftb as a library ===

      else if (ipotential(1:8) .eq. 'DFTBLIB ') then

         call force_libdftb

!     === cp2k as a library ===

      else if (ipotential(1:8) .eq. 'CP2KLIB ') then

         call force_libcp2k

!     ===  abinit-mp 5.0 (system call)  ===

      else if ( ipotential(1:11) .eq. 'ABINIT-MP5 ' ) then

         call force_abinit_mp5

!     ===  abinit-mp 5.0 (embedded)  ===
!
      else if ( ipotential(1:10) .eq. 'ABINIT-MP ' ) then

         call force_abinit_mp

!     ===  user defined  ===

      else if ( ipotential(1:5) .eq. 'USER ' ) then

         call force_user

!     ===  quartic potential (test)  ===

      else if ( ipotential(1:8) .eq. 'QUARTIC ' ) then

         call force_quartic

!     ===  pair potential  ===

      else if ( ipotential(1:5) .eq. 'PAIR ' ) then

         call force_pair

!     ===  pair potential  ===

      else if ( ipotential(1:8) .eq. 'TERSOFF ' ) then

         call force_tersoff

!     ===  tully's model (test)  ===

      else if ( ipotential(1:6) .eq. 'TULLY ' ) then

         call force_tully

!     ===  tip4p (test)  ===

      else if ( ipotential(1:6) .eq. 'TIP4P ' ) then

         call force_mm_tip4p

!     ===  cartesian (test)  ===

      else if ( ipotential(1:5) .eq. 'CART ' ) then

         call force_cart

!     ===  aenet  ===

      else if ( ipotential(1:6) .eq. 'AENET ' ) then

         call force_aenet

!     ===  n2p2  ===

      else if ( ipotential(1:5) .eq. 'N2P2 ' ) then

      call force_n2p2

!     ===  mtp  ===

      else if ( ipotential(1:4) .eq. 'MTP ' ) then

         call force_mtp

!     ===  rubnnet4md  ===

      else if ( ipotential(1:7) .eq. 'LIBNNP ' ) then

         call force_libnnp

!     ===  muller-brown  ===

      else if ( ipotential(1:7) .eq. 'MULLER ' ) then

         call force_muller

!     ===  ojamae-shavitt-singer  ===

      else if ( ipotential(1:4) .eq. 'OSS ' ) then

         call force_oss

!     ===  harmonic oscillators bath (test)  ===

      else if ( ipotential(1:4) .eq. 'HOB ' ) then

         call force_hob

!     ===  Matlantis PFP  ===

      else if ( ipotential(1:4) .eq. 'PFP ' ) then

         call force_pfp

!     ===  error  ===

      else

         call error_handling( 1, 'subroutine getforce', 19 )

      end if

      return
      end
