!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy in path integral hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getenergy_second_hmc_nvt
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom

      real(8) :: factqk

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:)

      potential_hmc    =  potential_second

      hamiltonian_hmc  =  ekin + qkin + potential_second

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_second_hmc_npt_c1
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vvol, volume, volmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   hamiltonian_sys, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

!-----------------------------------------------------------------------
!     /*   update box velocity                                        */
!-----------------------------------------------------------------------

      boxdot(1,1) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vvol / ( 3.d0*volume**(2.d0/3.d0) )

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*volmass*vvol*vvol + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:)

      potential_hmc    =  potential_second

      hamiltonian_hmc  =  hamiltonian

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_second_hmc_npt_c2
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vlog, volume, boxmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   hamiltonian_sys, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i

      real(8) :: factqk

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(1,1) = vlog * volume**(1.d0/3.d0)
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vlog * volume**(1.d0/3.d0)
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vlog * volume**(1.d0/3.d0)

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*boxmass(1,1)*vlog*vlog + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:)

      potential_hmc    =  potential_second

      hamiltonian_hmc  =  hamiltonian

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_second_hmc_npt_pp
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vbox, volume, boxmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   box, beta, hamiltonian_sys, cmtk, potential_cmtk, natom, nbead, &
     &   ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k

      real(8) :: factqk

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = pressure*volume

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:)

      potential_hmc    =  potential_second

      potential_cmtk   =  (1.d0 - cmtk) / beta * log(volume)

      hamiltonian_hmc  =  hamiltonian + potential_cmtk

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_second_hmc_ntt_pp
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vbox, volume, boxmass, vir, &
     &   ebaro, fictmass, vux, vuy, vuz, pres_iso, gbox, boxinv_ref, &
     &   strain, volume_ref, tension, box, beta, hamiltonian_sys, &
     &   pres, cmtk, potential_cmtk, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, l

      real(8) :: factqk

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   metric matrix                                              */
!-----------------------------------------------------------------------

      gbox(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         gbox(i,j) = gbox(i,j) + box(k,i)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   strain matrix                                              */
!-----------------------------------------------------------------------

      strain(:,:) = 0.d0

      do l = 1, 3
      do k = 1, 3
      do j = 1, 3
      do i = 1, 3

         strain(i,j) = strain(i,j) &
     &               + boxinv_ref(k,i)*gbox(k,l)*boxinv_ref(l,j)

      end do
      end do
      end do
      end do

      do i = 1, 3
         strain(i,i) = strain(i,i) - 1.d0
      end do

      strain(:,:) = 0.5d0*strain(:,:)

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.d0

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro - volume_ref*tension(i,j)*strain(j,i)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:)

      potential_hmc    =  potential_second

      potential_cmtk   =  (1.d0 - cmtk) / beta * log(volume)

      hamiltonian_hmc  =  hamiltonian + potential_cmtk

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine correct_dual_hamiltonian
!***********************************************************************
!=======================================================================
!
!     correction of HMC hamiltonian for dual potential
!
!     params_dual(1)     = weight of high-level potential
!     1 - params_dual(1) = weight of low-level potential
!
!=======================================================================

!     /*   shared variables   */
      use hmc_variables, only : &
     &    hamiltonian_hmc, potential_cor

!     /*   shared variables   */
      use dual_variables, only : &
     &    potential_high, potential_low, params_dual

!     //   local variables
      implicit none

!     //   dual potential correction
      potential_cor = potential_high - potential_low

!     //   correction of HMC hamiltonian
      hamiltonian_hmc = hamiltonian_hmc &
     &   - ( 1.d0 - params_dual(1) ) * potential_cor

      return
      end
