!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 30, 2020 by M. Shiga
!      Description:     energy in replica exchange hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getenergy_rehmc_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ekin, vx, vy, vz, physmass, potential, pot, hamiltonian, &
     &   natom, nbead, iounit, istep, myrank

      use rehmc_variables, only: &
     &   jstep

      use rehmc_variables, only : &
     &   hamiltonian_bead, ekin_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      integer :: ioption = 0

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do i = 1, nbead

         ekin_bead(i) = 0.d0

         do j = 1, natom

            ekin_bead(i) = ekin_bead(i) &
     &         + physmass(j) * vx(j,i) * vx(j,i) &
     &         + physmass(j) * vy(j,i) * vy(j,i) &
     &         + physmass(j) * vz(j,i) * vz(j,i)

         end do

         ekin_bead(i) = 0.5d0 * ekin_bead(i)

         ekin = ekin + ekin_bead(i)

      end do

      ekin = ekin / dble(nbead)

!-----------------------------------------------------------------------
!     /*   potential =  fictitious potential                          */
!-----------------------------------------------------------------------

      potential = 0.d0

      do i = 1, nbead

         potential = potential + pot(i)

      end do

      potential = potential / dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian = 0.d0

      do i = 1, nbead

         hamiltonian_bead(i) = ekin_bead(i) + pot(i)

         hamiltonian = hamiltonian + hamiltonian_bead(i)

      end do

      hamiltonian = hamiltonian / dble(nbead)

!-----------------------------------------------------------------------
!     /*   check                                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file='check.out', access='append' )

            write( iounit, '(i8,i6,3f16.8)' ) &
     &         istep, jstep, hamiltonian, potential, ekin

            close( iounit )

         end if

      end if

      return
      end






!***********************************************************************
      subroutine getenergy_rehmc_npt_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ekin, vx, vy, vz, physmass, potential, pot, hamiltonian, &
     &   hamiltonian_sys, ebaro, boxdot_bead, vbox_bead, box_bead, &
     &   pressure, volume_bead, boxmass, natom, nbead, istep, iounit, &
     &   myrank

      use rehmc_variables, only: &
     &   jstep

      use rehmc_variables, only : &
     &   hamiltonian_bead, ekin_bead, ebaro_bead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      integer :: ioption = 0

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      do l = 1, nbead

         boxdot_bead(:,:,l) = 0.d0

         do k = 1, 3
         do j = 1, 3
         do i = 1, 3

            boxdot_bead(i,j,l) = boxdot_bead(i,j,l) &
     &         + vbox_bead(i,k,l) * box_bead(k,j,l)

         end do
         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      ekin = 0.d0

      do i = 1, nbead

         ekin_bead(i) = 0.d0

         do j = 1, natom

            ekin_bead(i) = ekin_bead(i) &
     &         + physmass(j) * vx(j,i) * vx(j,i) &
     &         + physmass(j) * vy(j,i) * vy(j,i) &
     &         + physmass(j) * vz(j,i) * vz(j,i)

         end do

         ekin_bead(i) = 0.5d0 * ekin_bead(i)

         ekin = ekin + ekin_bead(i)

      end do

      ekin = ekin / dble(nbead)

!-----------------------------------------------------------------------
!     /*   potential =  fictitious potential                          */
!-----------------------------------------------------------------------

      potential = 0.d0

      do i = 1, nbead

         potential = potential + pot(i)

      end do

      potential = potential / dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = 0.d0

      do i = 1, nbead

         hamiltonian_bead(i) = ekin_bead(i) + pot(i)

         hamiltonian_sys = hamiltonian_sys + hamiltonian_bead(i)

      end do

      hamiltonian_sys = hamiltonian_sys / dble(nbead)

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.d0

      do l = 1, nbead

         ebaro_bead(l) = pressure * volume_bead(l)

         do j = 1, 3
         do i = 1, 3
            ebaro_bead(l) = ebaro_bead(l) &
     &         + 0.5d0 * boxmass(i,j)*vbox_bead(i,j,l)*vbox_bead(i,j,l)
         end do
         end do

         hamiltonian_bead(l) = hamiltonian_bead(l) + ebaro_bead(l)

         ebaro = ebaro + ebaro_bead(l)

      end do

      ebaro = ebaro / dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   check                                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         if ( myrank .eq. 0 ) then

            open ( iounit, file='check.out', access='append' )

            write( iounit, '(i8,i6,5f16.8)' ) &
     &         istep, jstep, hamiltonian, hamiltonian_sys, &
     &         potential, ekin, ebaro

            close( iounit )

         end if

      end if

      return
      end
