!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy in path integral hybrid Monte Carlo
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getenergy_fourth_hmc_nvt_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, beta, hbar, physmass, natom, nbead, &
     &   ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, pot_fourth, &
     &   potential_fourth, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j

      real(8) :: factqk, gx, gy, gz, g2, fact1, fact2

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian = ekin + qkin + potential

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   fourth order correction                                    */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      pot_fourth(:) =  0.d0

!     /*   constant   */

      fact1 = ( hbar*hbar*beta*beta ) / ( 24.d0*dble(nbead*nbead) )

!     /*   loop over beads   */

      do i = 1, nbead

         do j = 1, natom

!           /*   constant factor   */

            fact2 = fact1 / physmass(j)

!           /*   gradient (without 1/P factor)   */

            gx = - fx_second(j,i) * dble(nbead)
            gy = - fy_second(j,i) * dble(nbead)
            gz = - fz_second(j,i) * dble(nbead)

            g2 = gx*gx + gy*gy + gz*gz

!           /*   potential correction   */

            pot_fourth(i) = pot_fourth(i) + fact2*g2

         end do

      end do

!-----------------------------------------------------------------------
!     /*   potential is divided by nbead                              */
!-----------------------------------------------------------------------

      potential_fourth = 0.d0

      do j = 1, nbead
         potential_fourth = potential_fourth + pot_fourth(j)
      end do

      potential_fourth = potential_fourth/dble(nbead)

!-----------------------------------------------------------------------
!     /*   add original potential                                     */
!-----------------------------------------------------------------------

      pot_hmc(:)      =  pot_second(:) + pot_fourth(:)

      potential_hmc   =  potential_second + potential_fourth

      hamiltonian_hmc =  ekin + qkin + potential_hmc

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_fourth_hmc_npt_c1_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vvol, volume, volmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   hamiltonian_sys, hbar, beta, physmass, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_fourth, &
     &   pot_fourth, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j

      real(8) :: factqk, gx, gy, gz, g2, fact1, fact2

!-----------------------------------------------------------------------
!     /*   update box velocity                                        */
!-----------------------------------------------------------------------

      boxdot(1,1) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vvol / ( 3.d0*volume**(2.d0/3.d0) )
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vvol / ( 3.d0*volume**(2.d0/3.d0) )

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   fourth order correction                                    */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      pot_fourth(:) =  0.d0

!     /*   constant   */

      fact1 = ( hbar*hbar*beta*beta ) / ( 24.d0*dble(nbead*nbead) )

!     /*   loop over beads   */

      do i = 1, nbead

         do j = 1, natom

!           /*   constant factor   */

            fact2 = fact1 / physmass(j)

!           /*   gradient (without 1/P factor)   */

            gx = - fx_second(j,i) * dble(nbead)
            gy = - fy_second(j,i) * dble(nbead)
            gz = - fz_second(j,i) * dble(nbead)

            g2 = gx*gx + gy*gy + gz*gz

!           /*   potential correction   */

            pot_fourth(i) = pot_fourth(i) + fact2*g2

         end do

      end do

!-----------------------------------------------------------------------
!     /*   potential is divided by nbead                              */
!-----------------------------------------------------------------------

      potential_fourth = 0.d0

      do j = 1, nbead
         potential_fourth = potential_fourth + pot_fourth(j)
      end do

      potential_fourth = potential_fourth/dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential_second

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*volmass*vvol*vvol + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:) + pot_fourth(:)

      potential_hmc    =  potential_second + potential_fourth

      hamiltonian_hmc  =  hamiltonian + potential_fourth

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_fourth_hmc_npt_c2_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vlog, volume, boxmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   hamiltonian_sys, hbar, beta, physmass, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_fourth, &
     &   pot_fourth, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j

      real(8) :: factqk, gx, gy, gz, g2, fact1, fact2

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(1,1) = vlog * volume**(1.d0/3.d0)
      boxdot(1,2) = 0.d0
      boxdot(1,3) = 0.d0
      boxdot(2,1) = 0.d0
      boxdot(2,2) = vlog * volume**(1.d0/3.d0)
      boxdot(2,3) = 0.d0
      boxdot(3,1) = 0.d0
      boxdot(3,2) = 0.d0
      boxdot(3,3) = vlog * volume**(1.d0/3.d0)

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   fourth order correction                                    */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      pot_fourth(:) =  0.d0

!     /*   constant   */

      fact1 = ( hbar*hbar*beta*beta ) / ( 24.d0*dble(nbead*nbead) )

!     /*   loop over beads   */

      do i = 1, nbead

         do j = 1, natom

!           /*   constant factor   */

            fact2 = fact1 / physmass(j)

!           /*   gradient (without 1/P factor)   */

            gx = - fx_second(j,i) * dble(nbead)
            gy = - fy_second(j,i) * dble(nbead)
            gz = - fz_second(j,i) * dble(nbead)

            g2 = gx*gx + gy*gy + gz*gz

!           /*   potential correction   */

            pot_fourth(i) = pot_fourth(i) + fact2*g2

         end do

      end do

!-----------------------------------------------------------------------
!     /*   potential is divided by nbead                              */
!-----------------------------------------------------------------------

      potential_fourth = 0.d0

      do j = 1, nbead
         potential_fourth = potential_fourth + pot_fourth(j)
      end do

      potential_fourth = potential_fourth/dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential_second

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.5d0*boxmass(1,1)*vlog*vlog + pressure*volume

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:) + pot_fourth(:)

      potential_hmc    =  potential_second + potential_fourth

      hamiltonian_hmc  =  hamiltonian + potential_fourth

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_fourth_hmc_npt_pp_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vbox, volume, boxmass, vir, &
     &   pressure, ebaro, pres, fictmass, vux, vuy, vuz, pres_iso, &
     &   box, beta, hbar, physmass, hamiltonian_sys, cmtk, &
     &   potential_cmtk, natom, nbead, ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, potential_fourth, &
     &   pot_fourth, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k

      real(8) :: factqk, gx, gy, gz, g2, fact1, fact2

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   fourth order correction                                    */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      pot_fourth(:) =  0.d0

!     /*   constant   */

      fact1 = ( hbar*hbar*beta*beta ) / ( 24.d0*dble(nbead*nbead) )

!     /*   loop over beads   */

      do i = 1, nbead

         do j = 1, natom

!           /*   constant factor   */

            fact2 = fact1 / physmass(j)

!           /*   gradient (without 1/P factor)   */

            gx = - fx_second(j,i) * dble(nbead)
            gy = - fy_second(j,i) * dble(nbead)
            gz = - fz_second(j,i) * dble(nbead)

            g2 = gx*gx + gy*gy + gz*gz

!           /*   potential correction   */

            pot_fourth(i) = pot_fourth(i) + fact2*g2

         end do

      end do

!-----------------------------------------------------------------------
!     /*   potential is divided by nbead                              */
!-----------------------------------------------------------------------

      potential_fourth = 0.d0

      do j = 1, nbead
         potential_fourth = potential_fourth + pot_fourth(j)
      end do

      potential_fourth = potential_fourth/dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential_second

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = pressure*volume

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:) + pot_fourth(:)

      potential_hmc    =  potential_second + potential_fourth

      potential_cmtk   =  (1.d0 - cmtk) / beta * log(volume)

      hamiltonian_hmc  =  hamiltonian + potential_fourth &
     &                    +  potential_cmtk

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end





!***********************************************************************
      subroutine getenergy_fourth_hmc_ntt_pp_MPI
!***********************************************************************
!=======================================================================
!
!     calculate ``Hamiltonian''
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   hamiltonian, ekin, potential, qkin, ux, uy, uz, fx, fy, fz, &
     &   dnmmass, omega_p2, pot, boxdot, vbox, volume, boxmass, vir, &
     &   ebaro, fictmass, vux, vuy, vuz, pres_iso, gbox, boxinv_ref, &
     &   strain, volume_ref, tension, box, beta, hamiltonian_sys, &
     &   pres, physmass, cmtk, potential_cmtk, beta, hbar, natom, nbead, &
     &   ipotential

      use hmc_variables, only : &
     &   potential_second, pot_second, fx_second, fy_second, fz_second, &
     &   pot_hmc, potential_hmc, hamiltonian_hmc, pot_fourth, &
     &   potential_fourth, potential_cor

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: imode, iatom, i, j, k, l

      real(8) :: factqk, gx, gy, gz, g2, fact1, fact2

!-----------------------------------------------------------------------
!     /*   box velocity                                               */
!-----------------------------------------------------------------------

      boxdot(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         boxdot(i,j) = boxdot(i,j) + vbox(i,k)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   ekin =  fictitious kinetic energy                          */
!-----------------------------------------------------------------------

      call kinetic_energy

!-----------------------------------------------------------------------
!     /*   qkin  =  harmonic potential                                */
!-----------------------------------------------------------------------

      qkin = 0.d0

      do imode = 2, nbead
      do iatom = 1, natom
         factqk = 0.5d0*dnmmass(iatom,imode)*omega_p2
         qkin = qkin &
     &        + factqk*ux(iatom,imode)*ux(iatom,imode) &
     &        + factqk*uy(iatom,imode)*uy(iatom,imode) &
     &        + factqk*uz(iatom,imode)*uz(iatom,imode)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   second order potential                                     */
!-----------------------------------------------------------------------

      potential_second = potential

      pot_second(:)    = pot(:)

      fx_second(:,:)   = fx(:,:)
      fy_second(:,:)   = fy(:,:)
      fz_second(:,:)   = fz(:,:)

!-----------------------------------------------------------------------
!     /*   fourth order correction                                    */
!-----------------------------------------------------------------------

!     /*   zero clear   */

      pot_fourth(:) =  0.d0

!     /*   constant   */

      fact1 = ( hbar*hbar*beta*beta ) / ( 24.d0*dble(nbead*nbead) )

!     /*   loop over beads   */

      do i = 1, nbead

         do j = 1, natom

!           /*   constant factor   */

            fact2 = fact1 / physmass(j)

!           /*   gradient (without 1/P factor)   */

            gx = - fx_second(j,i) * dble(nbead)
            gy = - fy_second(j,i) * dble(nbead)
            gz = - fz_second(j,i) * dble(nbead)

            g2 = gx*gx + gy*gy + gz*gz

!           /*   potential correction   */

            pot_fourth(i) = pot_fourth(i) + fact2*g2

         end do

      end do

!-----------------------------------------------------------------------
!     /*   potential is divided by nbead                              */
!-----------------------------------------------------------------------

      potential_fourth = 0.d0

      do j = 1, nbead
         potential_fourth = potential_fourth + pot_fourth(j)
      end do

      potential_fourth = potential_fourth/dble(nbead)

!-----------------------------------------------------------------------
!     /*   hamiltonian                                                */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + qkin + potential_second

!-----------------------------------------------------------------------
!     /*   metric matrix                                              */
!-----------------------------------------------------------------------

      gbox(:,:) = 0.d0

      do k = 1, 3
      do j = 1, 3
      do i = 1, 3
         gbox(i,j) = gbox(i,j) + box(k,i)*box(k,j)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   strain matrix                                              */
!-----------------------------------------------------------------------

      strain(:,:) = 0.d0

      do l = 1, 3
      do k = 1, 3
      do j = 1, 3
      do i = 1, 3

         strain(i,j) = strain(i,j) &
     &               + boxinv_ref(k,i)*gbox(k,l)*boxinv_ref(l,j)

      end do
      end do
      end do
      end do

      do i = 1, 3
         strain(i,i) = strain(i,i) - 1.d0
      end do

      strain(:,:) = 0.5d0*strain(:,:)

!-----------------------------------------------------------------------
!     /*   ebaro =  barostats                                         */
!-----------------------------------------------------------------------

      ebaro = 0.d0

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro - volume_ref*tension(i,j)*strain(j,i)
      end do
      end do

      do j = 1, 3
      do i = 1, 3
         ebaro = ebaro + 0.5d0*boxmass(i,j)*vbox(i,j)*vbox(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys + ebaro

!-----------------------------------------------------------------------
!     /*   pressure                                                   */
!-----------------------------------------------------------------------

      pres(:,:) =  vir(:,:)

      do i = 1, natom

         pres(1,1) = pres(1,1) + fictmass(i,1)*vux(i,1)*vux(i,1)
         pres(1,2) = pres(1,2) + fictmass(i,1)*vux(i,1)*vuy(i,1)
         pres(1,3) = pres(1,3) + fictmass(i,1)*vux(i,1)*vuz(i,1)
         pres(2,1) = pres(2,1) + fictmass(i,1)*vuy(i,1)*vux(i,1)
         pres(2,2) = pres(2,2) + fictmass(i,1)*vuy(i,1)*vuy(i,1)
         pres(2,3) = pres(2,3) + fictmass(i,1)*vuy(i,1)*vuz(i,1)
         pres(3,1) = pres(3,1) + fictmass(i,1)*vuz(i,1)*vux(i,1)
         pres(3,2) = pres(3,2) + fictmass(i,1)*vuz(i,1)*vuy(i,1)
         pres(3,3) = pres(3,3) + fictmass(i,1)*vuz(i,1)*vuz(i,1)

      end do

      pres(:,:) = pres(:,:) / volume

!-----------------------------------------------------------------------
!     /*   isotropic pressure                                         */
!-----------------------------------------------------------------------

      pres_iso = ( pres(1,1) + pres(2,2) + pres(3,3) ) / 3.d0

!-----------------------------------------------------------------------
!     /*   substitution                                               */
!-----------------------------------------------------------------------

      pot_hmc(:)       =  pot_second(:) + pot_fourth(:)

      potential_hmc    =  potential_second + potential_fourth

      potential_cmtk   =  (1.d0 - cmtk) / beta * log(volume)

      hamiltonian_hmc  =  hamiltonian + potential_fourth &
     &                    + potential_cmtk

!-----------------------------------------------------------------------
!     /*   correction to hamiltonian_hmc for dual potential           */
!-----------------------------------------------------------------------

      potential_cor    = 0.d0

      if ( ipotential(1:5) .eq. 'DUAL ' ) call correct_dual_hamiltonian

      return
      end

