!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     calculate Onsager-Machlup action
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getaction_ref_om
!***********************************************************************

      use common_variables, only : equation_om

      implicit none

!-----------------------------------------------------------------------
!        /*   option of langevin equation                             */
!-----------------------------------------------------------------------

      if      ( equation_om(1:12) .eq. 'OVERDAMPED  ' ) then

         call getaction_ref_om_overdamped

      else if ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) then

         call getaction_ref_om_underdamped

      end if

      return
      end





!***********************************************************************
      subroutine getaction_ref_om_overdamped
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   x, y, z, physmass, ux, uy, uz, omega_p2, dnmmass, &
     &   iounit, natom, nbead

      use om_variables, only: &
     &   fx_ref_om, fy_ref_om, fz_ref_om, action_ref_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: pmi, c0, c1, dbead

      real(8) :: rx, ry, rz

      integer :: ioption = 0

!-----------------------------------------------------------------------
!     /*   initialize action                                          */
!-----------------------------------------------------------------------

      action_ref_om = 0.d0

      fx_ref_om(:,:) = 0.d0
      fy_ref_om(:,:) = 0.d0
      fz_ref_om(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dbead = dble(nbead)

!-----------------------------------------------------------------------
!     /*   action: kinetic part in cartesian space                    */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do j = 1, nbead-1

            k = j + 1

            do i = 1, natom

               pmi = physmass(i)

               c1 = 0.5d0 * pmi * omega_p2

               rx = x(i,k) - x(i,j)
               ry = y(i,k) - y(i,j)
               rz = z(i,k) - z(i,j)

               action_ref_om = action_ref_om &
     &            + c1 * ( rx*rx + ry*ry + rz*rz )

               fx_ref_om(i,j) = fx_ref_om(i,j) + 2.d0 * c1 * rx
               fy_ref_om(i,j) = fy_ref_om(i,j) + 2.d0 * c1 * ry
               fz_ref_om(i,j) = fz_ref_om(i,j) + 2.d0 * c1 * rz

               fx_ref_om(i,k) = fx_ref_om(i,k) - 2.d0 * c1 * rx
               fy_ref_om(i,k) = fy_ref_om(i,k) - 2.d0 * c1 * ry
               fz_ref_om(i,k) = fz_ref_om(i,k) - 2.d0 * c1 * rz

            end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   action: kinetic part in normal mode space                  */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   energy part   */

         action_ref_om = 0.d0

         do j = 2, nbead-1

            do i = 1, natom

               c0 = 0.5d0 * dnmmass(i,j) * omega_p2

               rx = ux(i,j)
               ry = uy(i,j)
               rz = uz(i,j)

               action_ref_om = action_ref_om &
     &            + c0 * ( rx*rx + ry*ry + rz*rz )

            end do

         end do

         do i = 1, natom

            pmi = physmass(i)

            c1 = 0.5d0 * pmi * omega_p2

            rx = x(i,1) - x(i,nbead)
            ry = y(i,1) - y(i,nbead)
            rz = z(i,1) - z(i,nbead)

            action_ref_om = action_ref_om &
     &         + c1 * ( rx*rx + ry*ry + rz*rz ) / dble(nbead-1)

         end do

!        /*   force part (in cartesian)   */

         do j = 1, nbead-1

            k = j + 1

            do i = 1, natom

               pmi = physmass(i)

               c1 = 0.5d0 * pmi * omega_p2

               rx = x(i,k) - x(i,j)
               ry = y(i,k) - y(i,j)
               rz = z(i,k) - z(i,j)

               fx_ref_om(i,j) = fx_ref_om(i,j) + 2.d0 * c1 * rx
               fy_ref_om(i,j) = fy_ref_om(i,j) + 2.d0 * c1 * ry
               fz_ref_om(i,j) = fz_ref_om(i,j) + 2.d0 * c1 * rz

               fx_ref_om(i,k) = fx_ref_om(i,k) - 2.d0 * c1 * rx
               fy_ref_om(i,k) = fy_ref_om(i,k) - 2.d0 * c1 * ry
               fz_ref_om(i,k) = fz_ref_om(i,k) - 2.d0 * c1 * rz

            end do

         end do

      end if

      call nm_trans_force_ref_om( 1 )

      return
      end





!***********************************************************************
      subroutine getaction_ref_om_underdamped
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   x, y, z, physmass, dnmmass, omega_p2, &
     &   ux, uy, uz, natom, nbead

      use om_variables, only: &
     &   fx_ref_om, fy_ref_om, fz_ref_om, action_ref_om, &
     &   dt_om, gamma_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: pmi, c0, c1, c3, c4

      real(8) :: rx, ry, rz, sx, sy, sz, dbead

      integer :: ioption = 0

!-----------------------------------------------------------------------
!     /*   initialize action                                          */
!-----------------------------------------------------------------------

      action_ref_om = 0.d0

      fx_ref_om(:,:) = 0.d0
      fy_ref_om(:,:) = 0.d0
      fz_ref_om(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dbead = dble(nbead)

!-----------------------------------------------------------------------
!     /*   action: kinetic part in cartesian space                    */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do j = 1, nbead-1

            k = j + 1

            do i = 1, natom

               pmi = physmass(i)

               c1 = 0.5d0 * pmi * omega_p2

               rx = x(i,k) - x(i,j)
               ry = y(i,k) - y(i,j)
               rz = z(i,k) - z(i,j)

               action_ref_om = action_ref_om &
     &            + c1 * ( rx*rx + ry*ry + rz*rz )

               fx_ref_om(i,j) = fx_ref_om(i,j) + 2.d0 * c1 * rx
               fy_ref_om(i,j) = fy_ref_om(i,j) + 2.d0 * c1 * ry
               fz_ref_om(i,j) = fz_ref_om(i,j) + 2.d0 * c1 * rz

               fx_ref_om(i,k) = fx_ref_om(i,k) - 2.d0 * c1 * rx
               fy_ref_om(i,k) = fy_ref_om(i,k) - 2.d0 * c1 * ry
               fz_ref_om(i,k) = fz_ref_om(i,k) - 2.d0 * c1 * rz

            end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   action: kinetic part in normal mode space                  */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   energy part   */

         action_ref_om = 0.d0

         do j = 2, nbead-1

            do i = 1, natom

               c0 = 0.5d0 * dnmmass(i,j) * omega_p2

               rx = ux(i,j)
               ry = uy(i,j)
               rz = uz(i,j)

               action_ref_om = action_ref_om &
     &            + c0 * ( rx*rx + ry*ry + rz*rz )

            end do

         end do

         do i = 1, natom

            pmi = physmass(i)

            c1 = 0.5d0 * pmi * omega_p2

            rx = x(i,1) - x(i,nbead)
            ry = y(i,1) - y(i,nbead)
            rz = z(i,1) - z(i,nbead)

            action_ref_om = action_ref_om &
     &                + c1 * ( rx*rx + ry*ry + rz*rz ) / dble(nbead-1)

         end do

!        /*   force part (in cartesian)   */

         do j = 1, nbead-1

            k = j + 1

            do i = 1, natom

               pmi = physmass(i)

               c1 = 0.5d0 * pmi * omega_p2

               rx = x(i,k) - x(i,j)
               ry = y(i,k) - y(i,j)
               rz = z(i,k) - z(i,j)

               fx_ref_om(i,j) = fx_ref_om(i,j) + 2.d0 * c1 * rx
               fy_ref_om(i,j) = fy_ref_om(i,j) + 2.d0 * c1 * ry
               fz_ref_om(i,j) = fz_ref_om(i,j) + 2.d0 * c1 * rz

               fx_ref_om(i,k) = fx_ref_om(i,k) - 2.d0 * c1 * rx
               fy_ref_om(i,k) = fy_ref_om(i,k) - 2.d0 * c1 * ry
               fz_ref_om(i,k) = fz_ref_om(i,k) - 2.d0 * c1 * rz

            end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   action: acceleration part                                  */
!-----------------------------------------------------------------------

      do j = 2, nbead-1

         l = j + 1
         m = j - 1

         do i = 1, natom

            pmi = physmass(i)

            c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

            rx = x(i,l) + x(i,m) - 2.d0*x(i,j)
            ry = y(i,l) + y(i,m) - 2.d0*y(i,j)
            rz = z(i,l) + z(i,m) - 2.d0*z(i,j)

            action_ref_om = action_ref_om &
     &         + c3 * ( rx*rx + ry*ry + rz*rz )

         end do

      end do

!-----------------------------------------------------------------------
!     /*   force of action: acceleration part                         */
!-----------------------------------------------------------------------

      do i = 1, natom

         pmi = physmass(i)

         c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

         sx = x(i,3) - 2.d0*x(i,2) + x(i,1)
         sy = y(i,3) - 2.d0*y(i,2) + y(i,1)
         sz = z(i,3) - 2.d0*z(i,2) + z(i,1)

         fx_ref_om(i,1) = fx_ref_om(i,1) - 2.d0*c3*sx
         fy_ref_om(i,1) = fy_ref_om(i,1) - 2.d0*c3*sy
         fz_ref_om(i,1) = fz_ref_om(i,1) - 2.d0*c3*sz

      end do

      do i = 1, natom

         pmi = physmass(i)

         c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

         sx = x(i,4) - 4.d0*x(i,3) + 5.d0*x(i,2) - 2.d0*x(i,1)
         sy = y(i,4) - 4.d0*y(i,3) + 5.d0*y(i,2) - 2.d0*y(i,1)
         sz = z(i,4) - 4.d0*z(i,3) + 5.d0*z(i,2) - 2.d0*z(i,1)

         fx_ref_om(i,2) = fx_ref_om(i,2) - 2.d0*c3*sx
         fy_ref_om(i,2) = fy_ref_om(i,2) - 2.d0*c3*sy
         fz_ref_om(i,2) = fz_ref_om(i,2) - 2.d0*c3*sz

      end do

      do j = 3, nbead-2

         k = j + 2
         l = j + 1
         m = j - 1
         n = j - 2

         do i = 1, natom

            pmi = physmass(i)

            c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

            sx = x(i,k) - 4.d0*x(i,l) + 6.d0*x(i,j) &
     &         - 4.d0*x(i,m) + x(i,n)
            sy = y(i,k) - 4.d0*y(i,l) + 6.d0*y(i,j) &
     &         - 4.d0*y(i,m) + y(i,n)
            sz = z(i,k) - 4.d0*z(i,l) + 6.d0*z(i,j) &
     &         - 4.d0*z(i,m) + z(i,n)

            fx_ref_om(i,j) = fx_ref_om(i,j) - 2.d0*c3*sx
            fy_ref_om(i,j) = fy_ref_om(i,j) - 2.d0*c3*sy
            fz_ref_om(i,j) = fz_ref_om(i,j) - 2.d0*c3*sz

         end do

      end do

      j = nbead-1
      l = nbead
      m = nbead-2
      n = nbead-3

      do i = 1, natom

         pmi = physmass(i)

         c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

         sx = x(i,n) - 4.d0*x(i,m) + 5.d0*x(i,j) - 2.d0*x(i,l)
         sy = y(i,n) - 4.d0*y(i,m) + 5.d0*y(i,j) - 2.d0*y(i,l)
         sz = z(i,n) - 4.d0*z(i,m) + 5.d0*z(i,j) - 2.d0*z(i,l)

         fx_ref_om(i,j) = fx_ref_om(i,j) - 2.d0*c3*sx
         fy_ref_om(i,j) = fy_ref_om(i,j) - 2.d0*c3*sy
         fz_ref_om(i,j) = fz_ref_om(i,j) - 2.d0*c3*sz

      end do

      j = nbead
      m = nbead-1
      n = nbead-2

      do i = 1, natom

         pmi = physmass(i)

         c3 = 0.25d0 * pmi / dt_om**3 / gamma_om

         sx = x(i,n) - 2.d0*x(i,m) + x(i,j)
         sy = y(i,n) - 2.d0*y(i,m) + y(i,j)
         sz = z(i,n) - 2.d0*z(i,m) + z(i,j)

         fx_ref_om(i,j) = fx_ref_om(i,j) - 2.d0*c3*sx
         fy_ref_om(i,j) = fy_ref_om(i,j) - 2.d0*c3*sy
         fz_ref_om(i,j) = fz_ref_om(i,j) - 2.d0*c3*sz

      end do

!-----------------------------------------------------------------------
!     /*   action: kinetic energy difference part                     */
!-----------------------------------------------------------------------

      do i = 1, natom

         pmi = physmass(i)

         c4 = 0.25d0 * pmi / dt_om / dt_om

         rx = x(i,2) - x(i,1)
         ry = y(i,2) - y(i,1)
         rz = z(i,2) - z(i,1)

         action_ref_om = action_ref_om + c4 * ( rx*rx + ry*ry + rz*rz )

         rx = x(i,nbead) - x(i,nbead-1)
         ry = y(i,nbead) - y(i,nbead-1)
         rz = z(i,nbead) - z(i,nbead-1)

         action_ref_om = action_ref_om + c4 * ( rx*rx + ry*ry + rz*rz )

      end do

!-----------------------------------------------------------------------
!     /*   force of action: kinetic energy difference part            */
!-----------------------------------------------------------------------

      do i = 1, natom

         pmi = physmass(i)

         c4 = 0.25d0 * pmi / dt_om / dt_om

         rx = x(i,2) - x(i,1)
         ry = y(i,2) - y(i,1)
         rz = z(i,2) - z(i,1)

         fx_ref_om(i,1) = fx_ref_om(i,1) + 2.d0*c4*rx
         fy_ref_om(i,1) = fy_ref_om(i,1) + 2.d0*c4*ry
         fz_ref_om(i,1) = fz_ref_om(i,1) + 2.d0*c4*rz

         fx_ref_om(i,2) = fx_ref_om(i,2) - 2.d0*c4*rx
         fy_ref_om(i,2) = fy_ref_om(i,2) - 2.d0*c4*ry
         fz_ref_om(i,2) = fz_ref_om(i,2) - 2.d0*c4*rz

         rx = x(i,nbead) - x(i,nbead-1)
         ry = y(i,nbead) - y(i,nbead-1)
         rz = z(i,nbead) - z(i,nbead-1)

         fx_ref_om(i,nbead-1) = fx_ref_om(i,nbead-1) + 2.d0*c4*rx
         fy_ref_om(i,nbead-1) = fy_ref_om(i,nbead-1) + 2.d0*c4*ry
         fz_ref_om(i,nbead-1) = fz_ref_om(i,nbead-1) + 2.d0*c4*rz

         fx_ref_om(i,nbead)   = fx_ref_om(i,nbead)   - 2.d0*c4*rx
         fy_ref_om(i,nbead)   = fy_ref_om(i,nbead)   - 2.d0*c4*ry
         fz_ref_om(i,nbead)   = fz_ref_om(i,nbead)   - 2.d0*c4*rz

      end do

      call nm_trans_force_ref_om( 1 )

      return
      end
